import React, { useState, useEffect, useCallback, useMemo, useRef } from 'react';
import { Link } from 'react-router-dom';
import { FormDefinition, FormGroup, FormSubmission } from '../types';
import { 
  getFormDefinitions, 
  deleteFormDefinition,
  getFormGroups,
  saveFormGroup,
  deleteFormGroup,
  assignFormToGroup,
  getAllSubmissions
} from '../services/formService';
import Modal from '../components/Modal';
import Chart, { ChartTypeRegistry } from 'chart.js/auto';
import LoadingSpinner from '../components/LoadingSpinner';
import { CHART_PALETTE, CHART_COLORS } from '../constants';

const formatDateForChart = (dateString: string): string => {
  return new Date(dateString).toLocaleDateString('ar-EG', { year: 'numeric', month: 'short', day: 'numeric' });
};

type SelectableChartType = 'bar' | 'line' | 'pie' | 'doughnut';

const DashboardPage: React.FC = () => {
  const [forms, setForms] = useState<FormDefinition[]>([]);
  const [formGroups, setFormGroups] = useState<FormGroup[]>([]);
  const [allSubmissionsData, setAllSubmissionsData] = useState<FormSubmission[]>([]);
  
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [formToDelete, setFormToDelete] = useState<FormDefinition | null>(null);
  
  const [showGroupModal, setShowGroupModal] = useState(false);
  const [groupToEdit, setGroupToEdit] = useState<FormGroup | null>(null);
  const [newGroupName, setNewGroupName] = useState('');

  const [showAssignGroupModal, setShowAssignGroupModal] = useState(false);
  const [formToAssignGroup, setFormToAssignGroup] = useState<FormDefinition | null>(null);
  const [selectedGroupIdForAssignment, setSelectedGroupIdForAssignment] = useState<string | null>(null);

  const [showDeleteGroupModal, setShowDeleteGroupModal] = useState(false);
  const [groupToDelete, setGroupToDelete] = useState<FormGroup | null>(null);

  const [dashboardSearchTerm, setDashboardSearchTerm] = useState('');
  const [debouncedDashboardSearchTerm, setDebouncedDashboardSearchTerm] = useState('');
  const [copiedFormId, setCopiedFormId] = useState<string | null>(null);

  const [showAnalytics, setShowAnalytics] = useState(false);
  const [analyticsStartDate, setAnalyticsStartDate] = useState('');
  const [analyticsEndDate, setAnalyticsEndDate] = useState('');
  const [isAnalyticsLoading, setIsAnalyticsLoading] = useState(false);
  const [selectedChartType, setSelectedChartType] = useState<SelectableChartType>('bar');
  const [isDashboardLoading, setIsDashboardLoading] = useState(true);
  const [dashboardError, setDashboardError] = useState<string | null>(null);


  const formsCreatedChartRef = useRef<HTMLCanvasElement>(null);
  const formsCreatedChartInstance = useRef<Chart | null>(null);
  const submissionsReceivedChartRef = useRef<HTMLCanvasElement>(null);
  const submissionsReceivedChartInstance = useRef<Chart | null>(null);
  const formsPerGroupChartRef = useRef<HTMLCanvasElement>(null);
  const formsPerGroupChartInstance = useRef<Chart | null>(null);
  const submissionsPerFormChartRef = useRef<HTMLCanvasElement>(null);
  const submissionsPerFormChartInstance = useRef<Chart | null>(null);


  const fetchData = useCallback(async () => {
    setIsDashboardLoading(true);
    setDashboardError(null);
    try {
      const formsData = await getFormDefinitions();
      setForms(formsData.sort((a,b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()));
      
      const formGroupsData = await getFormGroups();
      setFormGroups(formGroupsData.sort((a,b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()));
      
      if (showAnalytics) {
          setIsAnalyticsLoading(true);
          try {
            const submissionsData = await getAllSubmissions();
            setAllSubmissionsData(submissionsData);
          } catch (e) {
             console.error("Error fetching all submissions for analytics", e);
             // Optionally set an error state for analytics
          } finally {
            setIsAnalyticsLoading(false);
          }
      }
    } catch (err) {
      console.error("Failed to fetch dashboard data:", err);
      setDashboardError("فشل في تحميل بيانات لوحة التحكم.");
    } finally {
      setIsDashboardLoading(false);
    }
  }, [showAnalytics]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  useEffect(() => {
    const handler = setTimeout(() => {
      setDebouncedDashboardSearchTerm(dashboardSearchTerm);
    }, 300);
    return () => clearTimeout(handler);
  }, [dashboardSearchTerm]);

  const formsInGroups = useMemo(() => {
    const groupMap = new Map<string, FormDefinition[]>();
    formGroups.forEach(group => groupMap.set(group.id, []));
    const ungrouped: FormDefinition[] = [];
    forms.forEach(form => {
      let CroupFound = false; 
      for (const group of formGroups) {
        if (group.formIds.includes(form.id)) {
          groupMap.get(group.id)?.push(form);
          CroupFound = true;
          break; 
        }
      }
      if (!CroupFound) {
        ungrouped.push(form);
      }
    });
    const lowerSearchTerm = debouncedDashboardSearchTerm.toLowerCase();
    if (lowerSearchTerm) {
        const filteredGroupMap = new Map<string, FormDefinition[]>();
        groupMap.forEach((formList, groupId) => {
            const filteredList = formList.filter(form => 
                form.name.toLowerCase().includes(lowerSearchTerm) ||
                (form.description && form.description.toLowerCase().includes(lowerSearchTerm))
            );
            if (filteredList.length > 0) {
                 filteredGroupMap.set(groupId, filteredList);
            }
        });
        const filteredUngrouped = ungrouped.filter(form => 
            form.name.toLowerCase().includes(lowerSearchTerm) ||
            (form.description && form.description.toLowerCase().includes(lowerSearchTerm))
        );
        return { groupMap: filteredGroupMap, ungrouped: filteredUngrouped };
    }
    return { groupMap, ungrouped };
  }, [forms, formGroups, debouncedDashboardSearchTerm]);


  const handleDeleteClick = (form: FormDefinition) => {
    setFormToDelete(form);
    setShowDeleteModal(true);
  };

  const confirmDelete = async () => {
    if (formToDelete) {
      try {
        await deleteFormDefinition(formToDelete.id);
        setFormToDelete(null);
        setShowDeleteModal(false);
        fetchData(); 
      } catch (e) {
        console.error("Failed to delete form:", e);
        // Show error to user
      }
    }
  };

  const handleOpenGroupModal = (group: FormGroup | null = null) => {
    setGroupToEdit(group);
    setNewGroupName(group ? group.name : '');
    setShowGroupModal(true);
  };

  const handleSaveGroup = async () => {
    if (!newGroupName.trim()) return; 
    const groupData: FormGroup = {
      id: groupToEdit ? groupToEdit.id : `group-${Date.now()}`,
      name: newGroupName,
      formIds: groupToEdit ? groupToEdit.formIds : [],
      createdAt: groupToEdit ? groupToEdit.createdAt : new Date().toISOString(),
    };
    try {
      await saveFormGroup(groupData);
      setShowGroupModal(false);
      setNewGroupName('');
      setGroupToEdit(null);
      fetchData();
    } catch (e) {
      console.error("Failed to save group:", e);
       // Show error to user
    }
  };
  
  const handleOpenAssignGroupModal = (form: FormDefinition) => {
    setFormToAssignGroup(form);
    const currentGroup = formGroups.find(g => g.formIds.includes(form.id));
    setSelectedGroupIdForAssignment(currentGroup ? currentGroup.id : null);
    setShowAssignGroupModal(true);
  };

  const handleConfirmAssignGroup = async () => {
    if (formToAssignGroup) {
      try {
        await assignFormToGroup(formToAssignGroup.id, selectedGroupIdForAssignment);
        setShowAssignGroupModal(false);
        setFormToAssignGroup(null);
        setSelectedGroupIdForAssignment(null);
        fetchData();
      } catch (e) {
        console.error("Failed to assign form to group:", e);
        // Show error to user
      }
    }
  };

  const handleDeleteGroupClick = (group: FormGroup) => {
    setGroupToDelete(group);
    setShowDeleteGroupModal(true);
  };

  const confirmDeleteGroup = async () => {
    if (groupToDelete) {
      try {
        await deleteFormGroup(groupToDelete.id);
        setGroupToDelete(null);
        setShowDeleteGroupModal(false);
        fetchData();
      } catch (e) {
        console.error("Failed to delete group:", e);
        // Show error to user
      }
    }
  };

  const handleShareForm = (formId: string) => {
    const shareUrl = `${window.location.origin}${window.location.pathname}#/form/${formId}/fill`;
    navigator.clipboard.writeText(shareUrl)
      .then(() => {
        setCopiedFormId(formId);
        setTimeout(() => setCopiedFormId(null), 2000);
      })
      .catch(err => {
        console.error('Failed to copy form link: ', err);
        alert('فشل نسخ الرابط. الرجاء نسخه يدويًا.');
      });
  };

  const toggleAnalyticsSection = useCallback(async () => {
    const newShowAnalyticsState = !showAnalytics;
    if (newShowAnalyticsState && allSubmissionsData.length === 0) { // Only fetch if turning on and data isn't already there
      setIsAnalyticsLoading(true);
      try {
        const submissionsData = await getAllSubmissions();
        setAllSubmissionsData(submissionsData);
      } catch (e) {
        console.error("Error fetching submissions for analytics:", e);
        // Optionally set an error state to inform the user
      } finally {
        setIsAnalyticsLoading(false);
      }
    }
    setShowAnalytics(newShowAnalyticsState);
  }, [showAnalytics, allSubmissionsData.length]); // Include allSubmissionsData.length to avoid re-fetch if already loaded.

  const filteredFormsForAnalytics = useMemo(() => {
    if (!analyticsStartDate && !analyticsEndDate) return forms;
    const start = analyticsStartDate ? new Date(analyticsStartDate).getTime() : 0;
    const end = analyticsEndDate ? new Date(analyticsEndDate).setHours(23,59,59,999) : Infinity;
    return forms.filter(form => {
        const formTime = new Date(form.createdAt).getTime();
        return formTime >= start && formTime <= end;
    });
  }, [forms, analyticsStartDate, analyticsEndDate]);

  const filteredSubmissionsForAnalytics = useMemo(() => {
    if (!analyticsStartDate && !analyticsEndDate) return allSubmissionsData;
    const start = analyticsStartDate ? new Date(analyticsStartDate).getTime() : 0;
    const end = analyticsEndDate ? new Date(analyticsEndDate).setHours(23,59,59,999) : Infinity;
    return allSubmissionsData.filter(sub => {
        const subTime = new Date(sub.submittedAt).getTime();
        return subTime >= start && subTime <= end;
    });
  }, [allSubmissionsData, analyticsStartDate, analyticsEndDate]);

  const destroyChart = (chartInstance: React.MutableRefObject<Chart | null>) => {
    if (chartInstance.current) {
        chartInstance.current.destroy();
        chartInstance.current = null;
    }
  };
  
  const getChartConfig = (
    chartTitle: string, 
    labels: string[], 
    dataValues: number[], 
    baseColor: string, 
    type: SelectableChartType
  ): {type: keyof ChartTypeRegistry, data: any, options: any } => {
    
    const chartOptions: any = {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: { display: true, position: 'top', labels: { color: '#e2e8f0' } },
        title: { display: true, text: chartTitle, color: '#e2e8f0' }
      }
    };
    
    let datasetConfig: any = {
      label: chartTitle,
      data: dataValues,
    };

    if (type === 'line' || type === 'bar') {
      chartOptions.scales = {
        x: { ticks: { color: '#94a3b8' } },
        y: { ticks: { color: '#94a3b8' }, beginAtZero: true, suggestedMax: Math.max(...dataValues, 0) + 2 }
      };
      if (type === 'line') {
        datasetConfig.borderColor = baseColor;
        datasetConfig.backgroundColor = baseColor.replace('rgb', 'rgba').replace(')', ', 0.1)');
        datasetConfig.tension = 0.1;
        datasetConfig.fill = false;
      } else { // bar
        datasetConfig.backgroundColor = baseColor.replace('rgb', 'rgba').replace(')', ', 0.7)');
      }
    } else { // pie or doughnut
      datasetConfig.backgroundColor = dataValues.map((_, i) => CHART_PALETTE[i % CHART_PALETTE.length]);
      delete chartOptions.scales;
    }

    return {
      type: type,
      data: { labels, datasets: [datasetConfig] },
      options: chartOptions
    };
  };


  useEffect(() => { 
    destroyChart(formsCreatedChartInstance);
    if (showAnalytics && formsCreatedChartRef.current && filteredFormsForAnalytics.length > 0) {
        const countsByDate: Record<string, number> = {};
        filteredFormsForAnalytics.forEach(form => {
            const date = new Date(form.createdAt).toISOString().split('T')[0];
            countsByDate[date] = (countsByDate[date] || 0) + 1;
        });
        const sortedDates = Object.keys(countsByDate).sort((a,b) => new Date(a).getTime() - new Date(b).getTime());
        const labels = sortedDates.map(formatDateForChart);
        const data = sortedDates.map(date => countsByDate[date]);
        
        if (data.length === 0) return;
        const config = getChartConfig('النماذج المنشأة عبر الوقت', labels, data, CHART_COLORS.formsCreated, selectedChartType);
        formsCreatedChartInstance.current = new Chart(formsCreatedChartRef.current, config);
    }
    return () => destroyChart(formsCreatedChartInstance);
  }, [showAnalytics, filteredFormsForAnalytics, selectedChartType]);

  useEffect(() => { 
    destroyChart(submissionsReceivedChartInstance);
    if (showAnalytics && submissionsReceivedChartRef.current && filteredSubmissionsForAnalytics.length > 0) {
        const countsByDate: Record<string, number> = {};
        filteredSubmissionsForAnalytics.forEach(sub => {
            const date = new Date(sub.submittedAt).toISOString().split('T')[0];
            countsByDate[date] = (countsByDate[date] || 0) + 1;
        });
        const sortedDates = Object.keys(countsByDate).sort((a,b) => new Date(a).getTime() - new Date(b).getTime());
        const labels = sortedDates.map(formatDateForChart);
        const data = sortedDates.map(date => countsByDate[date]);
        
        if (data.length === 0) return;
        const config = getChartConfig('التقديمات المستلمة عبر الوقت', labels, data, CHART_COLORS.submissionsReceived, selectedChartType);
        submissionsReceivedChartInstance.current = new Chart(submissionsReceivedChartRef.current, config);
    }
    return () => destroyChart(submissionsReceivedChartInstance);
  }, [showAnalytics, filteredSubmissionsForAnalytics, selectedChartType]);

  useEffect(() => { 
    destroyChart(formsPerGroupChartInstance);
    if (showAnalytics && formsPerGroupChartRef.current && formGroups.length > 0) {
        const labels = formGroups.map(g => g.name);
        const data = formGroups.map(g => {
            // Count forms within the date range if filters are active
             const formsInThisGroup = filteredFormsForAnalytics.filter(form => g.formIds.includes(form.id));
             return formsInThisGroup.length;
        });
        
        if (data.every(d => d === 0) && filteredFormsForAnalytics.length > 0) { // If all groups have 0 forms for the period, but forms exist
             // Potentially render placeholder if no forms in any group for selected period
        }
        if (labels.length === 0 || data.length === 0) return;

        const config = getChartConfig('توزيع النماذج حسب المجموعة (ضمن النطاق الزمني)', labels, data, CHART_COLORS.formsPerGroup, selectedChartType);
        formsPerGroupChartInstance.current = new Chart(formsPerGroupChartRef.current, config);
    }
     return () => destroyChart(formsPerGroupChartInstance);
  }, [showAnalytics, formGroups, filteredFormsForAnalytics, selectedChartType]);

  useEffect(() => { 
    destroyChart(submissionsPerFormChartInstance);
    if (showAnalytics && submissionsPerFormChartRef.current && filteredSubmissionsForAnalytics.length > 0 && forms.length > 0) {
        const countsByFormId: Record<string, number> = {};
        filteredSubmissionsForAnalytics.forEach(sub => {
            countsByFormId[sub.formId] = (countsByFormId[sub.formId] || 0) + 1;
        });
        const formIdToNameMap = forms.reduce((acc, form) => { acc[form.id] = form.name; return acc; }, {} as Record<string, string>);
        
        const labels = Object.keys(countsByFormId).map(formId => formIdToNameMap[formId] || `ID: ${formId}`).filter(label => countsByFormId[Object.keys(formIdToNameMap).find(key => formIdToNameMap[key] === label || `ID: ${key}` === label)!] > 0);
        const data = labels.map(label => countsByFormId[Object.keys(formIdToNameMap).find(key => formIdToNameMap[key] === label || `ID: ${key}` === label)!]);


        if (labels.length === 0 || data.length === 0) return;

        const config = getChartConfig('التقديمات لكل نموذج (ضمن النطاق الزمني)', labels, data, CHART_COLORS.submissionsPerForm, selectedChartType);
        submissionsPerFormChartInstance.current = new Chart(submissionsPerFormChartRef.current, config);
    }
    return () => destroyChart(submissionsPerFormChartInstance);
  }, [showAnalytics, filteredSubmissionsForAnalytics, forms, selectedChartType]);
  
  const renderFormCard = (form: FormDefinition, currentGroupId: string | null) => (
    <div key={form.id} className="bg-slate-800 p-6 rounded-lg shadow-xl hover:shadow-2xl transition-shadow duration-300 flex flex-col justify-between">
      <div>
        <h2 className="text-2xl font-semibold text-sky-400 mb-2 truncate" title={form.name}>{form.name}</h2>
        <button
            onClick={() => handleOpenAssignGroupModal(form)}
            className="text-xs text-purple-400 hover:text-purple-300 mb-3 inline-flex items-center py-1 px-2 rounded hover:bg-slate-700 transition-colors"
        >
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-4 h-4 mr-1">
                <path strokeLinecap="round" strokeLinejoin="round" d="M2.25 12.75V12A2.25 2.25 0 014.5 9.75h15A2.25 2.25 0 0121.75 12v.75m-8.69-6.44l-2.12-2.12a1.5 1.5 0 00-1.061-.44H4.5A2.25 2.25 0 002.25 6v12a2.25 2.25 0 002.25 2.25h15A2.25 2.25 0 0021.75 18V9a2.25 2.25 0 00-2.25-2.25h-5.379a1.5 1.5 0 01-1.06-.44z" />
            </svg>
            {currentGroupId ? 'تغيير المجموعة' : 'تعيين إلى مجموعة'}
        </button>
        <p className="text-sm text-slate-400 mb-1">الحقول: {form.fields.length}</p>
        <p className="text-sm text-slate-500 mb-4">تاريخ الإنشاء: {new Date(form.createdAt).toLocaleDateString('ar-EG', { year: 'numeric', month: 'long', day: 'numeric' })}</p>
        {form.description && <p className="text-sm text-slate-300 mb-4 italic truncate" title={form.description}>"{form.description}"</p>}
      </div>
      <div className="mt-4 space-y-2">
        <div className="flex space-x-2">
          <Link
            to={`/form/${form.id}/fill`}
            className="flex-1 text-center bg-green-500 hover:bg-green-600 text-white font-medium py-2 px-4 rounded-md transition-colors"
          >
            تعبئة
          </Link>
          <button
            onClick={() => handleShareForm(form.id)}
            title="مشاركة رابط النموذج"
            className="flex-1 text-center bg-teal-500 hover:bg-teal-600 text-white font-medium py-2 px-4 rounded-md transition-colors relative"
          >
            {copiedFormId === form.id ? 'تم النسخ!' : 'مشاركة'}
          </button>
        </div>
        <Link
          to={`/form/${form.id}/submissions`}
          className="block w-full text-center bg-indigo-500 hover:bg-indigo-600 text-white font-medium py-2 px-4 rounded-md transition-colors"
        >
          عرض الردود
        </Link>
        <div className="flex space-x-2">
           <Link
            to={`/edit-form/${form.id}`}
            className="flex-1 text-center bg-yellow-500 hover:bg-yellow-600 text-white font-medium py-2 px-4 rounded-md transition-colors"
          >
            تعديل
          </Link>
          <button
            onClick={() => handleDeleteClick(form)}
            className="flex-1 bg-red-600 hover:bg-red-700 text-white font-medium py-2 px-4 rounded-md transition-colors"
          >
            حذف
          </button>
        </div>
      </div>
    </div>
  );

  const noFormsMatchSearch = debouncedDashboardSearchTerm.trim() && forms.length > 0 && 
                             Array.from(formsInGroups.groupMap.values()).every(list => list.length === 0) &&
                             formsInGroups.ungrouped.length === 0;

  if (isDashboardLoading) {
    return <div className="flex justify-center items-center h-screen"><LoadingSpinner text="جاري تحميل البيانات..." /></div>;
  }
  if (dashboardError) {
    return <div className="text-center py-10 text-red-400">{dashboardError}</div>;
  }


  return (
    <div className="space-y-8">
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <h1 className="text-4xl font-bold text-sky-400">لوحة تحكم النماذج</h1>
        <div className="flex gap-2">
          <button
            onClick={() => handleOpenGroupModal(null)}
            className="bg-purple-500 hover:bg-purple-600 text-white font-semibold py-2 px-6 rounded-lg shadow-md hover:shadow-lg transition duration-150 ease-in-out inline-flex items-center self-start md:self-auto"
          >
             <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5 ml-2"><path strokeLinecap="round" strokeLinejoin="round" d="M2.25 12.75V12A2.25 2.25 0 014.5 9.75h15A2.25 2.25 0 0121.75 12v.75m-8.69-6.44l-2.12-2.12a1.5 1.5 0 00-1.061-.44H4.5A2.25 2.25 0 002.25 6v12a2.25 2.25 0 002.25 2.25h15A2.25 2.25 0 0021.75 18V9a2.25 2.25 0 00-2.25-2.25h-5.379a1.5 1.5 0 01-1.06-.44z" /></svg>
            إنشاء مجموعة
          </button>
          <Link
            to="/new-form"
            className="bg-sky-500 hover:bg-sky-600 text-white font-semibold py-2 px-6 rounded-lg shadow-md hover:shadow-lg transition duration-150 ease-in-out inline-flex items-center self-start md:self-auto"
          >
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5 ml-2">
              <path strokeLinecap="round" strokeLinejoin="round" d="M12 4.5v15m7.5-7.5h-15" />
            </svg>
            إنشاء نموذج جديد
          </Link>
        </div>
      </div>

      <div className="mt-8">
        <button
            onClick={toggleAnalyticsSection}
            className="w-full text-left px-6 py-3 bg-slate-700 hover:bg-slate-600 text-sky-300 font-semibold rounded-lg shadow-md transition duration-150 ease-in-out flex justify-between items-center"
        >
            <span>إحصائيات وتقارير النماذج</span>
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className={`w-6 h-6 transition-transform duration-300 ${showAnalytics ? 'rotate-180' : ''}`}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M19.5 8.25l-7.5 7.5-7.5-7.5" />
            </svg>
        </button>
      </div>

      {showAnalytics && (
        <div className="mt-4 p-6 bg-slate-800/70 rounded-lg shadow-xl border border-slate-700 space-y-8">
            <div className="flex flex-col sm:flex-row justify-between items-center mb-4 gap-4">
                 <h2 className="text-2xl font-semibold text-sky-400">تحليلات النماذج</h2>
                 <div>
                    <label htmlFor="chartTypeSelect" className="block text-sm font-medium text-gray-300 mb-1">نوع الرسم البياني:</label>
                    <select 
                        id="chartTypeSelect" 
                        value={selectedChartType} 
                        onChange={(e) => setSelectedChartType(e.target.value as SelectableChartType)}
                        className="w-full sm:w-auto px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 text-gray-200"
                    >
                        <option value="bar">Bar Chart (أعمدة)</option>
                        <option value="line">Line Chart (خطي)</option>
                        <option value="pie">Pie Chart (دائري)</option>
                        <option value="doughnut">Doughnut Chart (دونات)</option>
                    </select>
                </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6 items-end">
                <div>
                    <label htmlFor="analyticsStartDate" className="block text-sm font-medium text-gray-300 mb-1">من تاريخ:</label>
                    <input type="date" id="analyticsStartDate" value={analyticsStartDate} onChange={e => setAnalyticsStartDate(e.target.value)} className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 text-gray-200" />
                </div>
                <div>
                    <label htmlFor="analyticsEndDate" className="block text-sm font-medium text-gray-300 mb-1">إلى تاريخ:</label>
                    <input type="date" id="analyticsEndDate" value={analyticsEndDate} onChange={e => setAnalyticsEndDate(e.target.value)} className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 text-gray-200" />
                </div>
            </div>

            {isAnalyticsLoading ? <div className="text-center py-8"><LoadingSpinner text="جاري تحميل بيانات الإحصائيات..." /></div> : 
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                <div className="bg-slate-700/50 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-semibold text-sky-300 mb-3">النماذج المنشأة عبر الوقت</h3>
                    {filteredFormsForAnalytics.length > 0 ? <canvas ref={formsCreatedChartRef} className="max-h-80"></canvas> : <p className="text-slate-400 text-center py-4">لا توجد نماذج منشأة في النطاق الزمني المحدد.</p>}
                </div>
                <div className="bg-slate-700/50 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-semibold text-sky-300 mb-3">التقديمات المستلمة عبر الوقت</h3>
                    {filteredSubmissionsForAnalytics.length > 0 ? <canvas ref={submissionsReceivedChartRef} className="max-h-80"></canvas> : <p className="text-slate-400 text-center py-4">لا توجد تقديمات في النطاق الزمني المحدد.</p>}
                </div>
                <div className="bg-slate-700/50 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-semibold text-sky-300 mb-3">توزيع النماذج حسب المجموعة</h3>
                    {formGroups.filter(g => filteredFormsForAnalytics.some(f => g.formIds.includes(f.id))).length > 0 ? <canvas ref={formsPerGroupChartRef} className="max-h-80"></canvas> : <p className="text-slate-400 text-center py-4">لا توجد مجموعات بها نماذج ضمن النطاق الزمني المحدد.</p>}
                </div>
                 <div className="bg-slate-700/50 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-semibold text-sky-300 mb-3">التقديمات لكل نموذج</h3>
                    {(filteredSubmissionsForAnalytics.length > 0 && forms.length > 0) ? <canvas ref={submissionsPerFormChartRef} className="max-h-80"></canvas> : <p className="text-slate-400 text-center py-4">لا توجد بيانات تقديمات كافية لعرض هذا الرسم البياني.</p>}
                </div>
            </div>
            }
        </div>
      )}


      <div className="mb-6">
        <input 
          type="text"
          placeholder="ابحث في النماذج بالاسم أو الوصف..."
          value={dashboardSearchTerm}
          onChange={(e) => setDashboardSearchTerm(e.target.value)}
          className="w-full px-4 py-3 bg-slate-700 border border-slate-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200 placeholder-slate-400"
        />
      </div>

      {forms.length === 0 && !debouncedDashboardSearchTerm.trim() ? (
        <div className="text-center py-10 bg-slate-800 rounded-lg shadow-xl">
          <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-16 h-16 mx-auto text-slate-500 mb-4">
            <path strokeLinecap="round" strokeLinejoin="round" d="M19.5 14.25v-2.625a3.375 3.375 0 00-3.375-3.375h-1.5A1.125 1.125 0 0113.5 7.125v-1.5a3.375 3.375 0 00-3.375-3.375H8.25m0 12.75h7.5m-7.5 3H12M10.5 2.25H5.625c-.621 0-1.125.504-1.125 1.125v17.25c0 .621.504 1.125 1.125 1.125h12.75c.621 0 1.125-.504 1.125-1.125V11.25a9 9 0 00-9-9z" />
          </svg>
          <p className="text-xl text-slate-400">لم يتم إنشاء نماذج بعد.</p>
          <p className="text-slate-500 mt-2">ابدأ بإنشاء أول نموذج ديناميكي لك!</p>
        </div>
      ) : noFormsMatchSearch ? (
         <div className="text-center py-10 bg-slate-800 rounded-lg shadow-xl">
           <p className="text-xl text-slate-400">لا توجد نماذج تطابق معايير البحث الخاصة بك.</p>
           <p className="text-slate-500 mt-2">حاول تعديل شروط البحث الخاصة بك.</p>
         </div>
      ) : (
        <>
          {formGroups.filter(g => formsInGroups.groupMap.has(g.id) && (formsInGroups.groupMap.get(g.id)?.length ?? 0) > 0 || !debouncedDashboardSearchTerm.trim()).map(group => (
            <div key={group.id} className="mb-10 p-4 border border-slate-700 rounded-lg bg-slate-800/50">
              <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4 pb-2 border-b border-slate-600 gap-2">
                <h2 className="text-2xl font-semibold text-purple-400">{group.name}</h2>
                <div className="flex flex-wrap gap-2 items-center">
                  <Link
                    to={`/group/${group.id}/submissions`}
                    className="text-indigo-400 hover:text-indigo-300 text-sm py-1 px-2 rounded hover:bg-slate-700 transition-colors inline-flex items-center"
                  >
                     <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-4 h-4 mr-1"><path strokeLinecap="round" strokeLinejoin="round" d="M20.25 6.375c0 2.278-3.694 4.125-8.25 4.125S3.75 8.653 3.75 6.375m16.5 0c0-2.278-3.694-4.125-8.25-4.125S3.75 4.097 3.75 6.375m16.5 0v11.25c0 2.278-3.694 4.125-8.25 4.125s-8.25-1.847-8.25-4.125V6.375m16.5 0v3.75m-16.5-3.75v3.75m16.5 0v3.75C20.25 16.153 16.556 18 12 18s-8.25-1.847-8.25-4.125v-3.75m16.5 0c0 2.278-3.694 4.125-8.25 4.125s-8.25-1.847-8.25-4.125" /></svg>
                    عرض جميع ردود المجموعة
                  </Link>
                  <button onClick={() => handleOpenGroupModal(group)} className="text-yellow-400 hover:text-yellow-300 text-sm">تعديل اسم المجموعة</button>
                  <button onClick={() => handleDeleteGroupClick(group)} className="text-red-400 hover:text-red-300 text-sm">حذف المجموعة</button>
                </div>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {formsInGroups.groupMap.get(group.id)?.map(form => renderFormCard(form, group.id))}
              </div>
            </div>
          ))}

          {formsInGroups.ungrouped.length > 0 && (
            <div className="mb-10 p-4 border border-slate-700 rounded-lg bg-slate-800/30">
              <h2 className="text-2xl font-semibold text-slate-400 mb-4 pb-2 border-b border-slate-600">نماذج غير مجمعة</h2>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {formsInGroups.ungrouped.map(form => renderFormCard(form, null))}
              </div>
            </div>
          )}
        </>
      )}
      
      <div className="mt-8 p-4 bg-slate-800 rounded-lg shadow-md">
        <h3 className="text-lg font-semibold text-sky-400 mb-2">إشعار تخزين البيانات</h3>
        <p className="text-sm text-slate-300">
          يتم تخزين جميع تعريفات النماذج وبيانات التقديم محليًا في localStorage بالمتصفح الخاص بك. 
          لا يتم إرسال هذه البيانات إلى أي خادم أو قاعدة بيانات تلقائيًا. 
          للتخزين الدائم أو التكامل مع أنظمة مثل MySQL/phpMyAdmin، ستحتاج عادةً إلى تطبيق خلفي للتعامل مع تقديمات البيانات. يمكنك تصدير التقديمات كـ JSON من صفحة "عرض الردود" للنقل اليدوي.
        </p>
      </div>

      <Modal
        isOpen={showDeleteModal}
        onClose={() => setShowDeleteModal(false)}
        title="تأكيد الحذف"
        primaryActionText="حذف"
        onPrimaryAction={confirmDelete}
        secondaryActionText="إلغاء"
        onSecondaryAction={() => setShowDeleteModal(false)}
      >
        <p>هل أنت متأكد أنك تريد حذف النموذج "{formToDelete?.name}"؟ هذا الإجراء لا يمكن التراجع عنه وسيحذف أيضًا جميع ردوده.</p>
      </Modal>

      <Modal
        isOpen={showGroupModal}
        onClose={() => { setShowGroupModal(false); setGroupToEdit(null); setNewGroupName('');}}
        title={groupToEdit ? "تعديل اسم المجموعة" : "إنشاء مجموعة جديدة"}
        primaryActionText={groupToEdit ? "حفظ التغييرات" : "إنشاء"}
        onPrimaryAction={handleSaveGroup}
        secondaryActionText="إلغاء"
        onSecondaryAction={() => { setShowGroupModal(false); setGroupToEdit(null); setNewGroupName('');}}
      >
        <div>
          <label htmlFor="groupName" className="block text-sm font-medium text-gray-300 mb-1">اسم المجموعة <span className="text-red-400">*</span></label>
          <input 
            type="text" 
            id="groupName" 
            value={newGroupName} 
            onChange={(e) => setNewGroupName(e.target.value)} 
            className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200" 
            required 
          />
        </div>
      </Modal>
      
      <Modal
        isOpen={showAssignGroupModal}
        onClose={() => {setShowAssignGroupModal(false); setFormToAssignGroup(null); setSelectedGroupIdForAssignment(null);}}
        title={`تعيين النموذج "${formToAssignGroup?.name}" إلى مجموعة`}
        primaryActionText="حفظ"
        onPrimaryAction={handleConfirmAssignGroup}
        secondaryActionText="إلغاء"
        onSecondaryAction={() => {setShowAssignGroupModal(false); setFormToAssignGroup(null); setSelectedGroupIdForAssignment(null);}}
      >
        <div>
          <label htmlFor="assignGroupSelect" className="block text-sm font-medium text-gray-300 mb-1">اختر مجموعة</label>
          <select 
            id="assignGroupSelect" 
            value={selectedGroupIdForAssignment || ''}
            onChange={(e) => setSelectedGroupIdForAssignment(e.target.value || null)}
            className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200"
          >
            <option value="">-- إزالة من المجموعة (غير مجمع) --</option>
            {formGroups.map(group => (
              <option key={group.id} value={group.id}>{group.name}</option>
            ))}
          </select>
           <p className="text-xs text-slate-400 mt-2">
            إذا لم تظهر المجموعة التي تريدها، قم بإنشائها أولاً من لوحة التحكم.
          </p>
        </div>
      </Modal>

      <Modal
        isOpen={showDeleteGroupModal}
        onClose={() => {setShowDeleteGroupModal(false); setGroupToDelete(null);}}
        title={`حذف المجموعة "${groupToDelete?.name}"`}
        primaryActionText="حذف المجموعة"
        onPrimaryAction={confirmDeleteGroup}
        secondaryActionText="إلغاء"
        onSecondaryAction={() => {setShowDeleteGroupModal(false); setGroupToDelete(null);}}
      >
        <p>هل أنت متأكد أنك تريد حذف المجموعة "{groupToDelete?.name}"؟</p>
        <p className="text-sm text-slate-400 mt-2">
          النماذج الموجودة في هذه المجموعة لن تُحذف، ولكنها ستصبح "غير مجمعة".
        </p>
      </Modal>

    </div>
  );
};

export default DashboardPage;
