
import React, { useState, useEffect, useCallback } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { FormDefinition, FormSubmissionData, FormSubmission, FieldType, FileValue, FormFieldDefinition } from '../types';
import { getFormDefinitionById, saveFormSubmission } from '../services/formService';
import FormFieldInput from '../components/FormFieldInput';
import LoadingSpinner from '../components/LoadingSpinner';

const getPlainStringForWhatsApp = (fieldDef: FormFieldDefinition | undefined, value: any): string => {
  if (value === null || value === undefined || (Array.isArray(value) && value.length === 0)) return "لم يتم الإدخال";
  if (!fieldDef) {
    if (value instanceof Date) return value.toLocaleString('ar-EG', { dateStyle: 'full', timeStyle: 'short' });
    return String(value);
  }
  switch (fieldDef.type) {
    case FieldType.PASSWORD: return '********';
    case FieldType.CHECKBOX: return value ? 'نعم' : 'لا';
    case FieldType.CHECKBOX_GROUP: return Array.isArray(value) ? value.join('، ') : "بيانات غير صالحة";
    case FieldType.FILE:
      const fileMeta = value as FileValue;
      return fileMeta?.name ? `${fileMeta.name} (${(fileMeta.size / 1024).toFixed(2)} ك.ب)` : "لا يوجد ملف";
    case FieldType.DATE: return value ? new Date(value).toLocaleDateString('ar-EG', { year: 'numeric', month: 'long', day: 'numeric' }) : "لم يتم الإدخال";
    case FieldType.TIME: return String(value) || "لم يتم الإدخال";
    case FieldType.DATETIME_LOCAL: return value ? new Date(value).toLocaleString('ar-EG', {dateStyle: 'long', timeStyle: 'short'}) : "لم يتم الإدخال";
    case FieldType.RANGE: return String(value);
    default: return String(value);
  }
};

const formatSubmissionForWhatsApp = (
  formName: string,
  submissionTime: string, // ISO string
  submittedData: FormSubmissionData,
  allFormFields: FormFieldDefinition[],
  location?: { latitude?: number; longitude?: number }
): string => {
  let message = `*استمارة جديدة: ${formName}*\n\n`;
  message += `*تاريخ الإرسال:*\n${new Date(submissionTime).toLocaleString('ar-EG', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit',
    hour12: true,
  })}\n\n`;
  message += `------------------------------------\n`;
  message += `*تفاصيل الرد:*\n\n`;

  for (const fieldId in submittedData) {
    if (Object.prototype.hasOwnProperty.call(submittedData, fieldId)) {
      const fieldDef = allFormFields.find(f => f.id === fieldId);
      if (fieldDef) { // Only include fields that are part of the form definition
        const value = submittedData[fieldId];
        const displayValue = getPlainStringForWhatsApp(fieldDef, value);
        message += `*${fieldDef.label}:*\n${displayValue}\n\n`;
      }
    }
  }
  
  message += `------------------------------------\n`;

  if (location && location.latitude && location.longitude) {
    message += `*الموقع الجغرافي:*\n`;
    message += `خط العرض: ${location.latitude.toFixed(5)}\n`;
    message += `خط الطول: ${location.longitude.toFixed(5)}\n`;
    message += `رابط الموقع: https://www.google.com/maps?q=${location.latitude},${location.longitude}\n\n`;
  }

  message += `---
تم إرسال هذا الرد عبر منشئ النماذج الديناميكي.`;

  return message;
};


const FillFormPage: React.FC = () => {
  const { formId } = useParams<{ formId: string }>();
  const navigate = useNavigate();
  const [formDef, setFormDef] = useState<FormDefinition | null>(null);
  const [formData, setFormData] = useState<FormSubmissionData>({});
  const [visibleFields, setVisibleFields] = useState<FormFieldDefinition[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [submitMessage, setSubmitMessage] = useState<string | null>(null);
  const [isSubmitting, setIsSubmitting] = useState(false);

  const getInitialValueForField = useCallback((field: FormFieldDefinition) => {
    if (field.type === FieldType.CHECKBOX) return false;
    if (field.type === FieldType.CHECKBOX_GROUP) return [];
    if (field.type === FieldType.FILE) return null;
    if (field.type === FieldType.RANGE) return field.min ?? 0;
    if (field.type === FieldType.TIME) return '';
    if (field.type === FieldType.SELECT || field.type === FieldType.RADIO_GROUP) {
        // If it has dynamic options, default to empty, otherwise handle static options if needed (though usually handled by select placeholder)
        return '';
    }
    return '';
  }, []);

  const initializeFormData = useCallback((definition: FormDefinition, currentData: FormSubmissionData = {}) => {
    const initialData: FormSubmissionData = {};
    definition.fields.forEach(field => {
      initialData[field.id] = currentData[field.id] ?? getInitialValueForField(field);
    });
    setFormData(initialData);
    return initialData;
  }, [getInitialValueForField]);
  
  const calculateVisibleFields = useCallback((definition: FormDefinition, currentData: FormSubmissionData): FormFieldDefinition[] => {
    const newlyVisibleFields: FormFieldDefinition[] = [];
    const currentVisibilityMap: { [key: string]: boolean } = {};
    let changedInThisIteration: boolean;

    do {
      changedInThisIteration = false;
      for (const field of definition.fields) {
        let shouldBeVisible = true; 
        if (field.condition) {
          const sourceField = definition.fields.find(f => f.id === field.condition.sourceFieldId);
          const sourceFieldIsCurrentlyVisible = sourceField ? (currentVisibilityMap[sourceField.id] ?? !sourceField.condition) : false;
          
          if (!sourceField || !sourceFieldIsCurrentlyVisible) {
            shouldBeVisible = false;
          } else {
            const sourceFieldValue = currentData[field.condition.sourceFieldId];
            if (!field.condition.triggerOptionValues.includes(sourceFieldValue as string)) {
              shouldBeVisible = false;
            }
          }
        }

        if (currentVisibilityMap[field.id] !== shouldBeVisible) {
          currentVisibilityMap[field.id] = shouldBeVisible;
          changedInThisIteration = true;
        }
      }
    } while (changedInThisIteration); 

    definition.fields.forEach(field => {
        if(currentVisibilityMap[field.id]) {
            newlyVisibleFields.push(field);
        }
    });
    return newlyVisibleFields;
  }, []);


  useEffect(() => {
    const loadForm = async () => {
      if (formId) {
        setIsLoading(true);
        setError(null);
        try {
          const definition = await getFormDefinitionById(formId);
          if (definition) {
            setFormDef(definition);
            const initialData = initializeFormData(definition, {});
            setVisibleFields(calculateVisibleFields(definition, initialData));
          } else {
            // This case should ideally not be hit if getFormDefinitionById throws on 404
            setError("لم يتم العثور على النموذج أو تعذر تحميله.");
          }
        } catch (catchedError: any) {
          const userFacingErrorMessage404 = `تعذر العثور على النموذج. قد يكون الرابط غير صحيح أو تم حذف النموذج.`;
          if (catchedError.status === 404) {
            console.error(
              `Backend API returned 404 (Not Found) for form ID ${formId}.`,
              `User is being shown: "${userFacingErrorMessage404}". Original error details:`,
              catchedError.errorData || catchedError.message 
            );
            setError(userFacingErrorMessage404);
          } else {
            const displayMessage = catchedError.message || "فشل في تحميل النموذج. يرجى المحاولة مرة أخرى لاحقاً.";
             console.error(
              `Error loading form definition for ID ${formId}.`,
              `User is being shown: "${displayMessage}". Original error details:`,
              catchedError.errorData || catchedError.message
            );
            setError(displayMessage);
          }
        } finally {
          setIsLoading(false);
        }
      }
    };
    loadForm();
  }, [formId, initializeFormData, calculateVisibleFields]);

  const handleInputChange = (fieldId: string, value: any) => {
    setFormData(prevData => {
      const newData = { ...prevData, [fieldId]: value };
      
      if (formDef) {
        // Check if this changed field is a source for any dynamic options fields
        formDef.fields.forEach(dependentField => {
          if (dependentField.dynamicOptionsConfig?.dependsOnFieldId === fieldId) {
            // If the source (fieldId) changed, reset the dependent field's value
            // because its options list might have changed.
            if (newData[dependentField.id] !== getInitialValueForField(dependentField)) {
                newData[dependentField.id] = getInitialValueForField(dependentField);
            }
          }
        });

        const newVisibleFields = calculateVisibleFields(formDef, newData);
        const previouslyVisibleFieldIds = new Set(visibleFields.map(vf => vf.id));
        const newlyVisibleFieldIds = new Set(newVisibleFields.map(vf => vf.id));

        formDef.fields.forEach(f => {
          if (previouslyVisibleFieldIds.has(f.id) && !newlyVisibleFieldIds.has(f.id)) {
             newData[f.id] = getInitialValueForField(f); 
          }
        });
        setVisibleFields(newVisibleFields); 
      }
      return newData;
    });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!formDef || !formId || isSubmitting) return;

    setIsSubmitting(true);
    setError(null);
    setSubmitMessage(null);

    // Filter formData to include only visible fields for validation and submission
    const dataToValidateAndSubmit: FormSubmissionData = {};
    visibleFields.forEach(vf => {
        dataToValidateAndSubmit[vf.id] = formData[vf.id];
    });


    for (const field of visibleFields) {
      if (field.required) {
        const value = dataToValidateAndSubmit[field.id]; // Use filtered data for validation
        let isEmpty = false;
        if (field.type === FieldType.CHECKBOX_GROUP) isEmpty = !value || (Array.isArray(value) && value.length === 0);
        else if (field.type === FieldType.FILE) isEmpty = !value;
        else if (typeof value === 'boolean') isEmpty = !value; 
        else isEmpty = !value && value !== 0 && value !== false;

        if (isEmpty) {
          setError(`حقل "${field.label}" مطلوب.`);
          setIsSubmitting(false);
          return;
        }
      }

      if (field.type === FieldType.TEXT && field.blockedWords && field.blockedWords.length > 0) {
        const value = dataToValidateAndSubmit[field.id];
        if (typeof value === 'string' && value.trim() !== '') {
          for (const blockedWord of field.blockedWords) {
            const regex = new RegExp(`\\b${blockedWord.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&')}\\b`, 'i');
            if (regex.test(value)) {
              setError(`حقل "${field.label}" يحتوي على كلمة محظورة: "${blockedWord}". الرجاء إزالتها.`);
              setIsSubmitting(false);
              return;
            }
          }
        }
      }
    }
    
    let submissionLocationData: { latitude?: number; longitude?: number } = {};
    let locationFetchUserMessage = "";

    if (navigator.geolocation) {
      try {
        setSubmitMessage("جاري جلب الموقع..."); 
        const position = await new Promise<GeolocationPosition>((resolve, reject) => {
          navigator.geolocation.getCurrentPosition(resolve, reject, { enableHighAccuracy: true, timeout: 10000, maximumAge: 0 });
        });
        submissionLocationData.latitude = position.coords.latitude;
        submissionLocationData.longitude = position.coords.longitude;
        locationFetchUserMessage = "تم التقاط الموقع بنجاح.";
      } catch (geoError: any) {
        console.warn("Geolocation error:", geoError);
        let specificErrorMsg = geoError.message || "حدث خطأ غير معروف.";
        if (geoError.code) {
          switch (geoError.code) {
            case 1: specificErrorMsg = "تم رفض إذن الوصول إلى الموقع."; break;
            case 2: specificErrorMsg = "معلومات الموقع غير متوفرة."; break;
            case 3: specificErrorMsg = "انتهت مهلة طلب الموقع."; break;
          }
        }
        locationFetchUserMessage = `تعذر استرداد الموقع: ${specificErrorMsg}`;
      }
    } else {
      locationFetchUserMessage = "خاصية تحديد الموقع الجغرافي غير مدعومة.";
    }

    const currentSubmissionTime = new Date().toISOString();
    const submission: FormSubmission = {
      id: `sub-${Date.now()}`,
      formId: formId,
      data: dataToValidateAndSubmit, 
      submittedAt: currentSubmissionTime,
      ...submissionLocationData,
    };

    try {
        await saveFormSubmission(submission);
        
        const whatsappMessage = formatSubmissionForWhatsApp(
        formDef.name,
        currentSubmissionTime,
        dataToValidateAndSubmit, // Send only visible fields' data
        formDef.fields,
        submissionLocationData
        );
        const whatsappNumber = "9647866866324";
        const whatsappUrl = `https://wa.me/${whatsappNumber}?text=${encodeURIComponent(whatsappMessage)}`;

        setSubmitMessage(`تم إرسال النموذج بنجاح! ${locationFetchUserMessage} جاري محاولة فتح WhatsApp...`.trim());
        window.open(whatsappUrl, '_blank');
        
        // Reset form fields to their initial state after submission
        if (formDef) {
            const freshInitialData = initializeFormData(formDef); 
            setVisibleFields(calculateVisibleFields(formDef, freshInitialData)); 
        }
    } catch (saveError: any) {
        console.error("Error saving submission:", saveError);
        setError(saveError.message || "فشل حفظ الرد. حاول مرة أخرى.");
        setSubmitMessage(null); // Clear any positive message
    } finally {
        setIsSubmitting(false);
    }
  };

  if (isLoading) return <div className="flex justify-center items-center h-64"><LoadingSpinner text="Loading form..." /></div>;
  if (error && !formDef && !isLoading) return <p className="text-red-400 text-center text-xl">{error}</p>;
  if (!formDef && !isLoading && !error) return <p className="text-gray-400 text-center text-xl">Form definition not available.</p>; // Show if no error but formDef still null
  if (!formDef) return <div className="flex justify-center items-center h-64"><LoadingSpinner text="Preparing form..." /></div>; // Fallback if still loading or formDef is null for other reasons

  const currentVisibleFieldsToRender = formDef.fields.filter(field => visibleFields.some(vf => vf.id === field.id));

  return (
    <div className="max-w-2xl mx-auto bg-slate-800 p-6 sm:p-8 rounded-lg shadow-2xl">
      <h1 className="text-3xl font-bold text-sky-400 mb-2">{formDef.name}</h1>
      {formDef.description && <p className="text-slate-300 mb-6">{formDef.description}</p>}
      
      {submitMessage && !error && <div className={`mb-4 p-3 rounded-md ${submitMessage.includes("بنجاح") ? 'bg-green-500' : 'bg-sky-600'} text-white text-center`}>{submitMessage}</div>}
      {error && <div className="mb-4 p-3 rounded-md bg-red-500 text-white text-center">{error}</div>}

      <form onSubmit={handleSubmit} className="space-y-6">
        {currentVisibleFieldsToRender.map(field => (
          <div key={field.id}>
            {(field.type !== FieldType.CHECKBOX && field.type !== FieldType.RADIO_GROUP && field.type !== FieldType.CHECKBOX_GROUP) && (
                 <label htmlFor={field.id} className="block text-sm font-medium text-gray-300 mb-1">
                {field.label} {field.required && <span className="text-red-400">*</span>}
                </label>
            )}
             {(field.type === FieldType.RADIO_GROUP || field.type === FieldType.CHECKBOX_GROUP) && (
                 <p className="block text-sm font-medium text-gray-300 mb-1">
                {field.label} {field.required && <span className="text-red-400">*</span>}
                </p>
            )}
            <FormFieldInput
              field={field}
              value={formData[field.id]}
              onChange={(value) => handleInputChange(field.id, value)}
              allFormData={formData} // Pass all form data for dynamic options
            />
          </div>
        ))}
        <div className="flex justify-end pt-4">
          <button
            type="submit"
            disabled={isSubmitting}
            className="px-8 py-3 text-base font-medium text-white bg-sky-500 hover:bg-sky-600 rounded-md focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-slate-800 focus:ring-sky-400 transition-colors disabled:opacity-70 flex items-center justify-center min-w-[150px]"
          >
            {isSubmitting && !submitMessage?.includes("بنجاح") ? <LoadingSpinner size="sm" color="text-white"/> : 'إرسال النموذج'}
          </button>
        </div>
      </form>
    </div>
  );
};

export default FillFormPage;
