import React, { useState, useEffect, useCallback, useMemo, useRef } from 'react';
import { useParams, Link } from 'react-router-dom';
import { FormDefinition, FormSubmission, FieldType, FileValue, FormGroup, FormFieldDefinition as SingleFormFieldDefinition } from '../types';
import { getFormDefinitionById, getFormSubmissions, getFormGroups } from '../services/formService';
import LoadingSpinner from '../components/LoadingSpinner';
import * as XLSX from 'xlsx';
import Chart, { ChartTypeRegistry } from 'chart.js/auto';
import jsPDF from 'jspdf';
import 'jspdf-autotable';
import { CHART_PALETTE } from '../constants';

interface AugmentedFormSubmission extends FormSubmission {
  formName: string;
  originalFormId: string; 
}

interface SortConfig {
  key: string; 
  direction: 'ascending' | 'descending';
}

interface DynamicColumn {
  normalizedLabel: string; 
  displayLabel: string;    
}

type SelectableChartType = 'bar' | 'line' | 'pie' | 'doughnut';


const getFieldDefFromOriginalFormByNormalizedLabel = (
    submission: AugmentedFormSubmission, 
    normalizedColumnLabel: string,
    formsInGroupRef: FormDefinition[] 
): SingleFormFieldDefinition | undefined => {
    const originalForm = formsInGroupRef.find(f => f.id === submission.originalFormId);
    if (!originalForm) return undefined;
    return originalForm.fields.find(f => f.label.trim().toLowerCase() === normalizedColumnLabel);
};

const getFilterableStringForGroup = (value: any, fieldDef?: SingleFormFieldDefinition | null ): string => {
  if (value === null || value === undefined) return "";
  
  if (fieldDef) {
    switch (fieldDef.type) {
      case FieldType.CHECKBOX:
        return value ? "محدد" : "غير محدد";
      case FieldType.CHECKBOX_GROUP:
        return Array.isArray(value) ? value.join(", ").toLowerCase() : "";
      case FieldType.FILE:
        const fileMeta = value as FileValue;
        return (fileMeta?.name || "").toLowerCase();
      case FieldType.DATE:
        try { return value ? new Date(value).toLocaleDateString('ar-EG').toLowerCase() : ""; } 
        catch (e) { return String(value).toLowerCase(); }
      case FieldType.TIME: 
        return String(value).toLowerCase();
      case FieldType.DATETIME_LOCAL:
        try { return value ? new Date(value).toLocaleString('ar-EG').toLowerCase() : ""; } 
        catch (e) { return String(value).toLowerCase(); }
      default:
        return String(value).toLowerCase();
    }
  }
  if (value instanceof Date) return value.toLocaleString('ar-EG').toLowerCase();
  if (typeof value === 'object' && value.name && typeof value.size === 'number') { 
      const fileMeta = value as FileValue;
      return (fileMeta.name || "").toLowerCase();
  }
  return String(value).toLowerCase();
};


const ViewGroupSubmissionsPage: React.FC = () => {
  const { groupId } = useParams<{ groupId: string }>();
  const [groupDef, setGroupDef] = useState<FormGroup | null>(null);
  const [formsInGroup, setFormsInGroup] = useState<FormDefinition[]>([]);
  const [allSubmissions, setAllSubmissions] = useState<AugmentedFormSubmission[]>([]);
  
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const [searchTerm, setSearchTerm] = useState('');
  const [debouncedSearchTerm, setDebouncedSearchTerm] = useState('');
  const [columnFilters, setColumnFilters] = useState<Record<string, any>>({});
  const [debouncedColumnFilters, setDebouncedColumnFilters] = useState<Record<string, any>>({});
  const [sortConfig, setSortConfig] = useState<SortConfig | null>(null);

  const [selectedChartType, setSelectedChartType] = useState<SelectableChartType>('pie');
  const pieChartRef = useRef<HTMLCanvasElement>(null);
  const chartInstanceRef = useRef<Chart | null>(null);

  const dynamicColumns = useMemo((): DynamicColumn[] => {
    const columnsMap = new Map<string, { normalizedLabel: string; displayLabel: string }>();
    formsInGroup.forEach(form => {
      form.fields.forEach(field => {
        const normalizedLabel = field.label.trim().toLowerCase();
        if (!columnsMap.has(normalizedLabel)) {
          columnsMap.set(normalizedLabel, { normalizedLabel, displayLabel: field.label });
        }
      });
    });
    return Array.from(columnsMap.values()).sort((a,b) => a.displayLabel.localeCompare(b.displayLabel, 'ar'));
  }, [formsInGroup]);


  useEffect(() => {
    const handler = setTimeout(() => setDebouncedSearchTerm(searchTerm), 300);
    return () => clearTimeout(handler);
  }, [searchTerm]);

  useEffect(() => {
    const handler = setTimeout(() => setDebouncedColumnFilters(columnFilters), 300);
    return () => clearTimeout(handler);
  }, [columnFilters]);

  const loadData = useCallback(async () => {
    if (!groupId) {
      setError("معرف المجموعة مفقود.");
      setIsLoading(false);
      return;
    }
    setIsLoading(true);
    setError(null);
    try {
      const allGroups = await getFormGroups();
      const currentGroup = allGroups.find(g => g.id === groupId);
      if (!currentGroup) {
        setError("المجموعة غير موجودة.");
        setIsLoading(false);
        return;
      }
      setGroupDef(currentGroup);

      const groupFormDefPromises = currentGroup.formIds.map(fid => getFormDefinitionById(fid));
      const groupFormDefsResults = await Promise.all(groupFormDefPromises);
      const validGroupFormDefs = groupFormDefsResults.filter(f => f !== undefined) as FormDefinition[];
      setFormsInGroup(validGroupFormDefs);

      let submissionsFromGroup: AugmentedFormSubmission[] = [];
      for (const formDef of validGroupFormDefs) {
        const formSubmissions = await getFormSubmissions(formDef.id);
        submissionsFromGroup.push(
          ...formSubmissions.map(sub => ({
            ...sub,
            formName: formDef.name,
            originalFormId: formDef.id,
          }))
        );
      }
      setAllSubmissions(submissionsFromGroup);
    } catch (e) {
      console.error("Error loading group submissions:", e);
      setError("فشل في تحميل ردود المجموعة.");
    } finally {
      setIsLoading(false);
    }
  }, [groupId]);

  useEffect(() => {
    loadData();
  }, [loadData]);
  

  const processedSubmissions = useMemo(() => {
    let filtered = [...allSubmissions];

    if (debouncedSearchTerm) {
      const lowerSearchTerm = debouncedSearchTerm.toLowerCase();
      filtered = filtered.filter(submission => {
        if (getFilterableStringForGroup(new Date(submission.submittedAt)).includes(lowerSearchTerm)) return true;
        if (submission.formName.toLowerCase().includes(lowerSearchTerm)) return true;
        if (submission.latitude && getFilterableStringForGroup(submission.latitude).includes(lowerSearchTerm)) return true;
        if (submission.longitude && getFilterableStringForGroup(submission.longitude).includes(lowerSearchTerm)) return true;
        
        return dynamicColumns.some(col => {
           const fieldDef = getFieldDefFromOriginalFormByNormalizedLabel(submission, col.normalizedLabel, formsInGroup);
           const value = fieldDef ? submission.data[fieldDef.id] : undefined;
           return getFilterableStringForGroup(value, fieldDef).includes(lowerSearchTerm);
        });
      });
    }

    Object.entries(debouncedColumnFilters).forEach(([columnKey, filterValue]) => { 
      if (!filterValue && typeof filterValue !== 'boolean' && filterValue !== 0) return;
      const lowerFilterValue = typeof filterValue === 'string' ? filterValue.toLowerCase() : '';
      
      filtered = filtered.filter(submission => {
        let valueToTest: any;
        let fieldDefinition: SingleFormFieldDefinition | undefined = undefined;

        if (columnKey === '_submittedAt') valueToTest = new Date(submission.submittedAt);
        else if (columnKey === '_formName') { 
             if (!lowerFilterValue) return true; // If filter is empty string, don't filter by formName here
             // This filter is applied from the dropdown, an exact match is expected
             return submission.formName === filterValue; 
        }
        else if (columnKey === '_formName_text_filter') { // This is for the text input under formName column
            if (!lowerFilterValue) return true;
            return submission.formName.toLowerCase().includes(lowerFilterValue);
        }
        else if (columnKey === '_latitude') valueToTest = submission.latitude;
        else if (columnKey === '_longitude') valueToTest = submission.longitude;
        else { 
            fieldDefinition = getFieldDefFromOriginalFormByNormalizedLabel(submission, columnKey, formsInGroup);
            valueToTest = fieldDefinition ? submission.data[fieldDefinition.id] : undefined;
        }
        
        if (columnKey === '_submittedAt' || (fieldDefinition && (fieldDefinition.type === FieldType.DATE || fieldDefinition.type === FieldType.DATETIME_LOCAL))) {
             if (!valueToTest) return false; 
             const dateValue = new Date(valueToTest);
             const { start, end } = filterValue as { start?: string; end?: string };
             if (start && new Date(start) > dateValue) return false;
             if (end) {
                const endDate = new Date(end);
                endDate.setHours(23, 59, 59, 999); 
                if (endDate < dateValue) return false;
             }
             return true;
        }
        if (columnKey !== '_formName' && columnKey !== '_formName_text_filter') {
           return getFilterableStringForGroup(valueToTest, fieldDefinition).includes(lowerFilterValue);
        }
        return true; 
      });
    });
    
    if (sortConfig !== null) {
      filtered.sort((a, b) => {
        let valA: any;
        let valB: any;

        if (sortConfig.key === '_submittedAt') { valA = new Date(a.submittedAt); valB = new Date(b.submittedAt); }
        else if (sortConfig.key === '_formName') { valA = a.formName; valB = b.formName; }
        else if (sortConfig.key === '_latitude') { valA = a.latitude; valB = b.latitude; }
        else if (sortConfig.key === '_longitude') { valA = a.longitude; valB = b.longitude; }
        else { 
          const fieldDefA = getFieldDefFromOriginalFormByNormalizedLabel(a, sortConfig.key, formsInGroup);
          valA = fieldDefA ? a.data[fieldDefA.id] : undefined;
          const fieldDefB = getFieldDefFromOriginalFormByNormalizedLabel(b, sortConfig.key, formsInGroup);
          valB = fieldDefB ? b.data[fieldDefB.id] : undefined;
        }

        if (valA === undefined || valA === null) return 1;
        if (valB === undefined || valB === null) return -1;
        
        const commonFieldDefForSort = getFieldDefFromOriginalFormByNormalizedLabel(a, sortConfig.key, formsInGroup) || getFieldDefFromOriginalFormByNormalizedLabel(b, sortConfig.key, formsInGroup);

        if (commonFieldDefForSort?.type === FieldType.NUMBER || commonFieldDefForSort?.type === FieldType.RANGE || typeof valA === 'number' && typeof valB === 'number') {
            return sortConfig.direction === 'ascending' ? Number(valA) - Number(valB) : Number(valB) - Number(valA);
        }
        if (commonFieldDefForSort?.type === FieldType.DATE || commonFieldDefForSort?.type === FieldType.DATETIME_LOCAL || valA instanceof Date && valB instanceof Date) {
            const dateA = new Date(valA);
            const dateB = new Date(valB);
            return sortConfig.direction === 'ascending' ? dateA.getTime() - dateB.getTime() : dateB.getTime() - dateA.getTime();
        }

        const strA = String(valA).toLowerCase();
        const strB = String(valB).toLowerCase();
        if (strA < strB) return sortConfig.direction === 'ascending' ? -1 : 1;
        if (strA > strB) return sortConfig.direction === 'ascending' ? 1 : -1;
        return 0;
      });
    } else {
        filtered.sort((a,b) => new Date(b.submittedAt).getTime() - new Date(a.submittedAt).getTime());
    }
    return filtered;
  }, [allSubmissions, dynamicColumns, debouncedSearchTerm, debouncedColumnFilters, sortConfig, formsInGroup]);

  const requestSort = (key: string) => { 
    let direction: 'ascending' | 'descending' = 'ascending';
    if (sortConfig && sortConfig.key === key && sortConfig.direction === 'ascending') {
      direction = 'descending';
    }
    setSortConfig({ key, direction });
  };

  const getSortIcon = (key: string) => {
    if (!sortConfig || sortConfig.key !== key) return <span className="text-slate-500">↕</span>;
    return sortConfig.direction === 'ascending' ? <span className="text-sky-300">↑</span> : <span className="text-sky-300">↓</span>;
  };

  const handleColumnFilterChange = (columnKey: string, value: string | { start?: string; end?: string }) => { 
    setColumnFilters(prev => ({ ...prev, [columnKey]: value }));
  };
  
  const handleSelectedFormFilterChange = (formName: string) => { // This is for the main dropdown
    handleColumnFilterChange('_formName', formName); // Store in columnFilters with _formName key
  };

  const clearAllFilters = () => {
    setSearchTerm('');
    setColumnFilters({}); 
  };
  
  const renderFieldValueForDisplay = (submission: AugmentedFormSubmission, normalizedColumnLabel: string): React.ReactNode => {
    const fieldDef = getFieldDefFromOriginalFormByNormalizedLabel(submission, normalizedColumnLabel, formsInGroup);
    const value = fieldDef ? submission.data[fieldDef.id] : undefined;

    if (value === null || value === undefined || (Array.isArray(value) && value.length === 0)) {
        return <span className="italic text-slate-500">غير متوفر</span>;
    }
    if (!fieldDef) return String(value); 

    switch (fieldDef.type) {
        case FieldType.PASSWORD: return '********';
        case FieldType.CHECKBOX: return value ? 'محدد' : 'غير محدد';
        case FieldType.CHECKBOX_GROUP: return Array.isArray(value) ? value.join(', ') : <span className="italic text-slate-500">غير صالح</span>;
        case FieldType.FILE:
            const fileMeta = value as FileValue;
            return fileMeta?.name ? `${fileMeta.name} (${(fileMeta.size / 1024).toFixed(2)} ك.ب)` : <span className="italic text-slate-500">لا يوجد ملف</span>;
        case FieldType.DATE: return value ? new Date(value).toLocaleDateString('ar-EG') : <span className="italic text-slate-500">غير متوفر</span>;
        case FieldType.TIME: return String(value); 
        case FieldType.DATETIME_LOCAL: return value ? new Date(value).toLocaleString('ar-EG') : <span className="italic text-slate-500">غير متوفر</span>;
        default: return String(value);
    }
  };
  
  const getPlainStringForExport = (submission: AugmentedFormSubmission, columnKey: string, isMetaColumn: boolean = false): string => {
    let value: any;
    let fieldDef: SingleFormFieldDefinition | undefined;

    if (isMetaColumn) {
        if (columnKey === '_submittedAt') value = new Date(submission.submittedAt);
        else if (columnKey === '_formName') value = submission.formName;
        else if (columnKey === '_latitude') value = submission.latitude;
        else if (columnKey === '_longitude') value = submission.longitude;
    } else { 
        fieldDef = getFieldDefFromOriginalFormByNormalizedLabel(submission, columnKey, formsInGroup);
        value = fieldDef ? submission.data[fieldDef.id] : undefined;
    }
    
    if (value === null || value === undefined || (Array.isArray(value) && value.length === 0)) return "غير متوفر";
    
    if (value instanceof Date) return value.toISOString();

    if (isMetaColumn) return String(value); 
    if (!fieldDef && !isMetaColumn) return String(value); 

    switch (fieldDef?.type) {
        case FieldType.PASSWORD: return '********';
        case FieldType.CHECKBOX: return value ? 'محدد' : 'غير محدد';
        case FieldType.CHECKBOX_GROUP: return Array.isArray(value) ? value.join(', ') : "بيانات غير صالحة";
        case FieldType.FILE:
            const fileMeta = value as FileValue;
            return fileMeta?.name ? `${fileMeta.name} (${(fileMeta.size / 1024).toFixed(2)} ك.ب)` : "لا يوجد ملف";
        case FieldType.DATE: return value ? new Date(value).toISOString().split('T')[0] : "غير متوفر";
        case FieldType.TIME: return String(value); 
        case FieldType.DATETIME_LOCAL: return value ? new Date(value).toISOString() : "غير متوفر";
        default: return String(value);
    }
    return String(value); 
  };

  const handleExport = (format: 'json' | 'excel' | 'pdf') => {
    if (!groupDef) return;

    const exportHeaders: string[] = ['وقت التقديم', 'اسم النموذج', ...dynamicColumns.map(col => col.displayLabel), 'خط العرض', 'خط الطول'];
    const exportBody = processedSubmissions.map(sub => {
      const row: any[] = [
        getPlainStringForExport(sub, '_submittedAt', true),
        getPlainStringForExport(sub, '_formName', true),
        ...dynamicColumns.map(col => getPlainStringForExport(sub, col.normalizedLabel)),
        getPlainStringForExport(sub, '_latitude', true),
        getPlainStringForExport(sub, '_longitude', true),
      ];
      return row;
    });
    
    const filenameBase = `${groupDef.name.replace(/\s+/g, '_')}_ردود_المجموعة`;

    if (format === 'json') {
      const dataToExport = {
        groupName: groupDef.name, groupId: groupDef.id, exportedAt: new Date().toISOString(),
        submissions: processedSubmissions 
      };
      const jsonString = JSON.stringify(dataToExport, null, 2);
      const blob = new Blob([jsonString], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a'); a.href = url; a.download = `${filenameBase}.json`;
      document.body.appendChild(a); a.click(); document.body.removeChild(a); URL.revokeObjectURL(url);
    } else if (format === 'excel') {
      const wsData = [exportHeaders, ...exportBody.map(row => row.map(cell => cell ?? "غير متوفر"))];
      const worksheet = XLSX.utils.aoa_to_sheet(wsData);
      const workbook = XLSX.utils.book_new();
      XLSX.utils.book_append_sheet(workbook, worksheet, "الردود");
      XLSX.writeFile(workbook, `${filenameBase}.xlsx`);
    } else if (format === 'pdf') {
        const doc = new jsPDF({ orientation: 'landscape' });
        (doc as any).autoTable({
            head: [exportHeaders],
            body: exportBody,
            styles: { font: 'helvetica' }, 
            headStyles: { fillColor: [22, 160, 133] },
        });
        doc.save(`${filenameBase}.pdf`);
    }
  };

  useEffect(() => {
    if (chartInstanceRef.current) {
        chartInstanceRef.current.destroy();
        chartInstanceRef.current = null;
    }
    if (pieChartRef.current && groupDef && processedSubmissions.length > 0) {
        const counts: Record<string, number> = {};
        const selectedFormNameForFilter = columnFilters['_formName'] as string | undefined;
        let chartTitle = `توزيع الردود لمجموعة: ${groupDef.name}`;
        
        formsInGroup.forEach(form => counts[form.name] = 0);

        processedSubmissions.forEach(sub => {
            if (sub.formName && counts.hasOwnProperty(sub.formName)) {
                counts[sub.formName]++;
            }
        });
        
        const activeLabels = Object.keys(counts).filter(key => counts[key] > 0);
        const data = activeLabels.map(label => counts[label]);

        if (data.length === 0) {
             if (pieChartRef.current.getContext('2d')) { 
                const ctx = pieChartRef.current.getContext('2d')!;
                ctx.clearRect(0, 0, pieChartRef.current.width, pieChartRef.current.height);
                ctx.font = "16px Arial";
                ctx.fillStyle = "#94a3b8"; 
                ctx.textAlign = "center";
                ctx.fillText("لا توجد بيانات تطابق الفلترة لعرض الرسم البياني.", pieChartRef.current.width / 2, pieChartRef.current.height / 2);
            }
            return;
        }

        const chartOptions: any = {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'top', labels: { color: '#e2e8f0' } },
                title: { display: true, text: chartTitle, color: '#e2e8f0' }
            }
        };

        const datasetConfig: any = {
            label: "عدد الردود",
            data: data,
        };

        if (selectedChartType === 'line' || selectedChartType === 'bar') {
            chartOptions.scales = {
                x: { ticks: { color: '#94a3b8' } },
                y: { ticks: { color: '#94a3b8' }, beginAtZero: true, suggestedMax: Math.max(...data, 0) + 1 }
            };
            if (selectedChartType === 'line') {
                datasetConfig.borderColor = CHART_PALETTE[0];
                datasetConfig.backgroundColor = CHART_PALETTE[0].replace('0.8', '0.1'); // Lighter fill for line
                datasetConfig.tension = 0.1;
                datasetConfig.fill = true;
            } else { // bar
                datasetConfig.backgroundColor = data.map((_, i) => CHART_PALETTE[i % CHART_PALETTE.length]);
            }
        } else { // pie or doughnut
            datasetConfig.backgroundColor = data.map((_, i) => CHART_PALETTE[i % CHART_PALETTE.length]);
            delete chartOptions.scales;
        }

        chartInstanceRef.current = new Chart(pieChartRef.current, {
            type: selectedChartType as keyof ChartTypeRegistry,
            data: {
                labels: activeLabels,
                datasets: [datasetConfig]
            },
            options: chartOptions
        });
    } else if (pieChartRef.current && pieChartRef.current.getContext('2d')) {
        const ctx = pieChartRef.current.getContext('2d')!;
        ctx.clearRect(0, 0, pieChartRef.current.width, pieChartRef.current.height);
         if (processedSubmissions.length === 0 && allSubmissions.length > 0) {
            ctx.font = "16px Arial";
            ctx.fillStyle = "#94a3b8"; 
            ctx.textAlign = "center";
            ctx.fillText("لا توجد بيانات تطابق الفلترة لعرض الرسم البياني.", pieChartRef.current.width / 2, pieChartRef.current.height / 2);
        }
    }
  }, [processedSubmissions, groupDef, formsInGroup, columnFilters, selectedChartType]); 


  if (isLoading) return <div className="flex justify-center items-center h-64"><LoadingSpinner text="جاري تحميل ردود المجموعة..." /></div>;
  if (error) return <p className="text-red-400 text-center text-xl">{error}</p>;
  if (!groupDef) return <p className="text-gray-400 text-center text-xl">تعريف المجموعة غير متوفر.</p>;

  const hasActiveFilters = searchTerm || Object.values(columnFilters).some(f => {
    if (typeof f === 'string' && f !== '') return true;
    if (typeof f === 'object' && f !== null && ((f as any).start || (f as any).end)) return true;
    return false;
  });

  return (
    <div className="bg-slate-800 p-6 sm:p-8 rounded-lg shadow-2xl space-y-8">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
        <div>
            <h1 className="text-3xl font-bold text-sky-400">ردود المجموعة: {groupDef.name}</h1>
            <p className="text-slate-400">عرض {processedSubmissions.length} من {allSubmissions.length} إجمالي الردود في المجموعة</p>
        </div>
        <div className="flex flex-wrap gap-2">
            {allSubmissions.length > 0 && (<>
              <button onClick={() => handleExport('json')} className="bg-indigo-500 hover:bg-indigo-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تصدير JSON</button>
              <button onClick={() => handleExport('excel')} className="bg-emerald-500 hover:bg-emerald-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تصدير Excel</button>
              <button onClick={() => handleExport('pdf')} className="bg-rose-500 hover:bg-rose-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تصدير PDF</button>
            </>)}
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-3 items-end">
        <div className="md:col-span-1">
            <label htmlFor="formViewSelect" className="block text-sm font-medium text-gray-300 mb-1">عرض ردود لـ:</label>
            <select 
                id="formViewSelect"
                value={columnFilters['_formName'] || ''} 
                onChange={(e) => handleSelectedFormFilterChange(e.target.value)}
                className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200"
            >
                <option value="">جميع النماذج في المجموعة</option>
                {formsInGroup.map(form => <option key={form.id} value={form.name}>{form.name}</option>)}
            </select>
        </div>
        <div className="md:col-span-2">
             <label htmlFor="globalSearchGroup" className="block text-sm font-medium text-gray-300 mb-1">بحث عام في الردود:</label>
            <input 
                id="globalSearchGroup"
                type="text" 
                placeholder="بحث..." 
                value={searchTerm} 
                onChange={(e) => setSearchTerm(e.target.value)} 
                className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200"
            />
        </div>
         {hasActiveFilters && (
            <div className="md:col-span-3 flex justify-end mt-2 md:mt-0">
                 <button onClick={clearAllFilters} className="px-4 py-2 text-sm font-medium text-amber-300 bg-slate-700 hover:bg-slate-600 rounded-md border border-amber-400/50">مسح جميع الفلاتر والبحث</button>
            </div>
        )}
      </div>


      {processedSubmissions.length === 0 && allSubmissions.length > 0 ? (
         <div className="text-center py-10 border-2 border-dashed border-slate-700 rounded-lg"><p className="text-xl text-slate-400">لا توجد ردود تطابق معايير الفلترة الحالية.</p></div>
      ) : allSubmissions.length === 0 ? (
        <div className="text-center py-10 border-2 border-dashed border-slate-700 rounded-lg">
          <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-12 h-12 mx-auto text-slate-500 mb-3"> <path strokeLinecap="round" strokeLinejoin="round" d="M20.25 6.375c0 2.278-3.694 4.125-8.25 4.125S3.75 8.653 3.75 6.375m16.5 0c0-2.278-3.694-4.125-8.25-4.125S3.75 4.097 3.75 6.375m16.5 0v11.25c0 2.278-3.694 4.125-8.25 4.125s-8.25-1.847-8.25-4.125V6.375m16.5 0v3.75m-16.5-3.75v3.75m16.5 0v3.75C20.25 16.153 16.556 18 12 18s-8.25-1.847-8.25-4.125v-3.75m16.5 0c0 2.278-3.694 4.125-8.25 4.125s-8.25-1.847-8.25-4.125" /></svg>
          <p className="text-xl text-slate-400">لا توجد ردود في هذه المجموعة حتى الآن.</p>
        </div>
      ) : (
        <div className="overflow-x-auto shadow-md rounded-lg">
          <table className="min-w-full divide-y divide-slate-700">
            <thead className="bg-slate-700/50">
              <tr>
                <th scope="col" onClick={() => requestSort('_submittedAt')} className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider sticky left-0 bg-slate-700/80 backdrop-blur-sm z-10 cursor-pointer">
                  وقت التقديم {getSortIcon('_submittedAt')}
                </th>
                <th scope="col" onClick={() => requestSort('_formName')} className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                  اسم النموذج {getSortIcon('_formName')}
                </th>
                {dynamicColumns.map(col => (
                  <th key={col.normalizedLabel} scope="col" onClick={() => requestSort(col.normalizedLabel)} className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                    {col.displayLabel} {getSortIcon(col.normalizedLabel)}
                  </th>
                ))}
                <th scope="col" onClick={() => requestSort('_latitude')} className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                  خط العرض {getSortIcon('_latitude')}
                </th>
                <th scope="col" onClick={() => requestSort('_longitude')} className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                  خط الطول {getSortIcon('_longitude')}
                </th>
              </tr>
              <tr className="bg-slate-700/30">
                <td className="px-2 py-1 sticky left-0 bg-slate-700/60 backdrop-blur-sm z-10">
                  <div className="flex space-x-1">
                    <input type="date" placeholder="من" value={columnFilters['_submittedAt']?.start || ''} onChange={(e) => handleColumnFilterChange('_submittedAt', {...(columnFilters['_submittedAt'] || {}), start: e.target.value})} className="w-1/2 text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200"/>
                    <input type="date" placeholder="إلى" value={columnFilters['_submittedAt']?.end || ''} onChange={(e) => handleColumnFilterChange('_submittedAt', {...(columnFilters['_submittedAt'] || {}), end: e.target.value})} className="w-1/2 text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200"/>
                  </div>
                </td>
                <td className="px-2 py-1">
                   <input type="text" placeholder="فلترة بالاسم..." value={columnFilters['_formName_text_filter'] || ''} onChange={(e) => handleColumnFilterChange('_formName_text_filter', e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" title="يتم التحكم في اختيار النموذج الرئيسي من القائمة أعلاه. هذا للفلترة النصية الإضافية إذا لزم الأمر."/>
                </td>
                {dynamicColumns.map(col => (
                  <td key={`${col.normalizedLabel}-filter`} className="px-2 py-1">
                    <input type="text" placeholder={`فلترة ${col.displayLabel}...`} value={columnFilters[col.normalizedLabel] || ''} onChange={(e) => handleColumnFilterChange(col.normalizedLabel, e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" />
                  </td>
                ))}
                <td className="px-2 py-1"><input type="text" placeholder="فلترة..." value={columnFilters['_latitude'] || ''} onChange={(e) => handleColumnFilterChange('_latitude', e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" /></td>
                <td className="px-2 py-1"><input type="text" placeholder="فلترة..." value={columnFilters['_longitude'] || ''} onChange={(e) => handleColumnFilterChange('_longitude', e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" /></td>
              </tr>
            </thead>
            <tbody className="bg-slate-800 divide-y divide-slate-700">
              {processedSubmissions.map(submission => (
                <tr key={submission.id} className="hover:bg-slate-700/40 transition-colors">
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300 sticky left-0 bg-slate-800 hover:bg-slate-700/40 z-0">{new Date(submission.submittedAt).toLocaleString('ar-EG')}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300">{submission.formName}</td>
                  {dynamicColumns.map(col => (
                    <td key={`${submission.id}-${col.normalizedLabel}`} className="px-6 py-4 whitespace-normal break-words text-sm text-slate-300 max-w-xs">
                        {renderFieldValueForDisplay(submission, col.normalizedLabel)}
                    </td>
                  ))}
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300">{submission.latitude !== undefined ? submission.latitude.toFixed(5) : <span className="italic text-slate-500">غير متوفر</span>}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300">{submission.longitude !== undefined ? submission.longitude.toFixed(5) : <span className="italic text-slate-500">غير متوفر</span>}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

       {formsInGroup.length > 0 && allSubmissions.length > 0 && (
        <div className="mt-8 p-6 bg-slate-700/30 rounded-lg border border-slate-600">
            <div className="flex flex-col sm:flex-row justify-between items-center mb-4 gap-4">
                <h3 className="text-xl font-semibold text-sky-400">الرسوم البيانية لردود المجموعة</h3>
                <div>
                    <label htmlFor="chartTypeSelectGroupPage" className="block text-sm font-medium text-gray-300 mb-1">نوع الرسم البياني:</label>
                    <select 
                        id="chartTypeSelectGroupPage" 
                        value={selectedChartType} 
                        onChange={(e) => setSelectedChartType(e.target.value as SelectableChartType)}
                        className="w-full sm:w-auto px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 text-gray-200"
                    >
                        <option value="pie">Pie Chart (دائري)</option>
                        <option value="doughnut">Doughnut Chart (دونات)</option>
                        <option value="bar">Bar Chart (أعمدة)</option>
                        <option value="line">Line Chart (خطي)</option>
                    </select>
                </div>
            </div>
            <div className="relative h-96 w-full">
                 <canvas ref={pieChartRef}></canvas>
            </div>
        </div>
      )}

       <div className="mt-8 p-4 bg-slate-700/30 rounded-lg border border-slate-600">
        <h3 className="text-md font-semibold text-sky-400 mb-1">ملاحظة حول عرض ردود المجموعة</h3>
        <p className="text-sm text-slate-400">تعرض هذه الصفحة جميع الردود من كافة النماذج داخل المجموعة المحددة. يتم الآن دمج الأعمدة التي لها نفس التسمية (مثل "التاريخ") من نماذج مختلفة في عمود واحد. استخدم القائمة المنسدلة أعلاه للتركيز على نموذج معين أو عرض جميع النماذج. قد تختلف الحقول بين النماذج؛ يتم عرض "غير متوفر" للحقول غير الموجودة في رد معين. التصدير يشمل عمود "اسم النموذج" للتمييز.</p>
      </div>
    </div>
  );
};

export default ViewGroupSubmissionsPage;
