import React, { useState, useEffect, useCallback, useMemo, useRef } from 'react';
import { useParams, Link } from 'react-router-dom';
import { FormDefinition, FormSubmission, FieldType, FileValue, FormFieldDefinition } from '../types';
import { getFormDefinitionById, getFormSubmissions } from '../services/formService';
import LoadingSpinner from '../components/LoadingSpinner';
import * as XLSX from 'xlsx';
import Chart, { ChartTypeRegistry } from 'chart.js/auto';
import jsPDF from 'jspdf';
import 'jspdf-autotable'; 
import { CHART_PALETTE } from '../constants';
import L, { Map as LeafletMap, Marker as LeafletMarker, TileLayer, FeatureGroup } from 'leaflet';


interface SortConfig {
  key: string;
  direction: 'ascending' | 'descending';
}

interface ActiveDragState {
  columnKey: string;
  startX: number;
  startWidth: number;
}

type SelectableChartType = 'bar' | 'line' | 'pie' | 'doughnut';
const MIN_COLUMN_WIDTH = 75; // Minimum width in pixels for a column

const getFilterableString = (fieldDef: FormFieldDefinition | null, value: any): string => {
  if (value === null || value === undefined) return "";
  
  if (fieldDef) {
    switch (fieldDef.type) {
      case FieldType.CHECKBOX:
        return value ? "محدد" : "غير محدد"; 
      case FieldType.CHECKBOX_GROUP:
        return Array.isArray(value) ? value.join(", ").toLowerCase() : "";
      case FieldType.FILE:
        const fileMeta = value as FileValue;
        return (fileMeta?.name || "").toLowerCase();
      case FieldType.DATE:
        try {
          return value ? new Date(value).toLocaleDateString('ar-EG').toLowerCase() : ""; 
        } catch (e) { return String(value).toLowerCase(); }
      case FieldType.TIME: 
        return String(value).toLowerCase();
      case FieldType.DATETIME_LOCAL:
        try {
          return value ? new Date(value).toLocaleString('ar-EG').toLowerCase() : ""; 
        } catch (e) { return String(value).toLowerCase(); }
      default:
        return String(value).toLowerCase();
    }
  }
   if (value instanceof Date) return value.toLocaleString('ar-EG').toLowerCase();
   if (typeof value === 'object' && value.name && typeof value.size === 'number') { 
      const fileMeta = value as FileValue;
      return (fileMeta.name || "").toLowerCase();
  }
  return String(value).toLowerCase();
};


const ViewSubmissionsPage: React.FC = () => {
  const { formId } = useParams<{ formId: string }>();
  const [formDef, setFormDef] = useState<FormDefinition | null>(null);
  const [submissions, setSubmissions] = useState<FormSubmission[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const [searchTerm, setSearchTerm] = useState('');
  const [debouncedSearchTerm, setDebouncedSearchTerm] = useState('');
  const [columnFilters, setColumnFilters] = useState<Record<string, any>>({}); 
  const [debouncedColumnFilters, setDebouncedColumnFilters] = useState<Record<string, any>>({});
  const [sortConfig, setSortConfig] = useState<SortConfig | null>(null);

  const [chartFieldId, setChartFieldId] = useState<string>('');
  const [selectedChartType, setSelectedChartType] = useState<SelectableChartType>('pie');
  const pieChartRef = useRef<HTMLCanvasElement>(null);
  const chartInstanceRef = useRef<Chart | null>(null);

  // Map state
  const [showMapView, setShowMapView] = useState<boolean>(false);
  const [selectedSubmissionOnMap, setSelectedSubmissionOnMap] = useState<FormSubmission | null>(null);
  const mapContainerRef = useRef<HTMLDivElement>(null);
  const leafletMapInstanceRef = useRef<LeafletMap | null>(null);
  const markersRef = useRef<LeafletMarker[]>([]);

  // Column Resizing State
  const [columnWidths, setColumnWidths] = useState<Record<string, number>>({});
  const [activeDrag, setActiveDrag] = useState<ActiveDragState | null>(null);
  const tableRef = useRef<HTMLTableElement>(null);


  useEffect(() => {
    const handler = setTimeout(() => setDebouncedSearchTerm(searchTerm), 300);
    return () => clearTimeout(handler);
  }, [searchTerm]);

  useEffect(() => {
    const handler = setTimeout(() => setDebouncedColumnFilters(columnFilters), 300);
    return () => clearTimeout(handler);
  }, [columnFilters]);

  const getLocalStorageColumnWidthsKey = useCallback(() => {
    return formId ? `columnWidths_viewSubmissions_${formId}` : null;
  }, [formId]);

  const loadData = useCallback(async () => {
    if (formId) {
      setIsLoading(true);
      setError(null);
      try {
        const definition = await getFormDefinitionById(formId);
        if (definition) {
          setFormDef(definition);
          const submissionsData = await getFormSubmissions(formId);
          setSubmissions(submissionsData); 
          const firstCategoricalField = definition.fields.find(f => (f.type === FieldType.SELECT || f.type === FieldType.RADIO_GROUP) && f.options && f.options.length > 0);
          if (firstCategoricalField) {
              setChartFieldId(firstCategoricalField.id);
          } else {
              setChartFieldId(''); 
          }
          const storageKey = getLocalStorageColumnWidthsKey();
          if (storageKey) {
              const storedWidths = localStorage.getItem(storageKey);
              if (storedWidths) {
                  try {
                      setColumnWidths(JSON.parse(storedWidths));
                  } catch (e) {
                      console.error("Failed to parse stored column widths:", e);
                  }
              }
          }
        } else {
          setError("لم يتم العثور على النموذج.");
        }
      } catch (err) {
        console.error("Failed to load submissions data:", err);
        setError("فشل في تحميل الردود.");
      } finally {
        setIsLoading(false);
      }
    } else {
        setError("معرف النموذج مفقود.");
        setIsLoading(false);
    }
  }, [formId, getLocalStorageColumnWidthsKey]);

  useEffect(() => {
    loadData();
  }, [loadData]);

  const processedSubmissions = useMemo(() => {
    if (!formDef) return [];
    let filtered = [...submissions];

    if (debouncedSearchTerm) {
      const lowerSearchTerm = debouncedSearchTerm.toLowerCase();
      filtered = filtered.filter(submission => {
        if (getFilterableString(null, new Date(submission.submittedAt)).includes(lowerSearchTerm)) return true;
        if (submission.latitude && getFilterableString(null, submission.latitude).includes(lowerSearchTerm)) return true;
        if (submission.longitude && getFilterableString(null, submission.longitude).includes(lowerSearchTerm)) return true;
        return formDef.fields.some(field => getFilterableString(field, submission.data[field.id]).includes(lowerSearchTerm));
      });
    }

    Object.entries(debouncedColumnFilters).forEach(([columnId, filterValue]) => {
      if (!filterValue && typeof filterValue !== 'boolean' && filterValue !== 0) return; 
      const lowerFilterValue = typeof filterValue === 'string' ? filterValue.toLowerCase() : '';
      
      filtered = filtered.filter(submission => {
        let valueToTest: any;
        let fieldDefinition: FormFieldDefinition | null = null;

        if (columnId === '_submittedAt') valueToTest = new Date(submission.submittedAt);
        else if (columnId === '_latitude') valueToTest = submission.latitude;
        else if (columnId === '_longitude') valueToTest = submission.longitude;
        else {
            fieldDefinition = formDef.fields.find(f => f.id === columnId) || null;
            valueToTest = submission.data[columnId];
        }

        if (fieldDefinition && (fieldDefinition.type === FieldType.DATE || fieldDefinition.type === FieldType.DATETIME_LOCAL) || columnId === '_submittedAt') {
            if (!valueToTest) return false;
            const dateValue = new Date(valueToTest);
            const { start, end } = filterValue as { start?: string; end?: string };
            if (start && new Date(start) > dateValue) return false;
            if (end) {
                const endDate = new Date(end);
                endDate.setHours(23, 59, 59, 999); 
                if (endDate < dateValue) return false;
            }
            return true;
        }
        if (fieldDefinition && fieldDefinition.type === FieldType.CHECKBOX) {
             const filterBool = lowerFilterValue === 'محدد' ? true : (lowerFilterValue === 'غير محدد' ? false : null);
             if (filterBool === null && lowerFilterValue !== '') return false; 
             if (filterBool !== null) return valueToTest === filterBool;
             return true; 
        }
        return getFilterableString(fieldDefinition, valueToTest).includes(lowerFilterValue);
      });
    });
    
    if (sortConfig !== null) {
      filtered.sort((a, b) => {
        let valA: any;
        let valB: any;

        if (sortConfig.key === '_submittedAt') { valA = new Date(a.submittedAt); valB = new Date(b.submittedAt); }
        else if (sortConfig.key === '_latitude') { valA = a.latitude; valB = b.latitude; }
        else if (sortConfig.key === '_longitude') { valA = a.longitude; valB = b.longitude; }
        else { valA = a.data[sortConfig.key]; valB = b.data[sortConfig.key]; }

        if (valA === undefined || valA === null) return 1;
        if (valB === undefined || valB === null) return -1;
        
        const fieldForSort = formDef.fields.find(f => f.id === sortConfig.key);
        if (fieldForSort?.type === FieldType.NUMBER || fieldForSort?.type === FieldType.RANGE || (typeof valA === 'number' && typeof valB === 'number')) {
            return sortConfig.direction === 'ascending' ? Number(valA) - Number(valB) : Number(valB) - Number(valA);
        }
        if (fieldForSort?.type === FieldType.DATE || fieldForSort?.type === FieldType.DATETIME_LOCAL || valA instanceof Date && valB instanceof Date) {
             const dateA = new Date(valA);
             const dateB = new Date(valB);
            return sortConfig.direction === 'ascending' ? dateA.getTime() - dateB.getTime() : dateB.getTime() - dateA.getTime();
        }

        const strA = String(valA).toLowerCase();
        const strB = String(valB).toLowerCase();
        if (strA < strB) return sortConfig.direction === 'ascending' ? -1 : 1;
        if (strA > strB) return sortConfig.direction === 'ascending' ? 1 : -1;
        return 0;
      });
    } else { 
        filtered.sort((a,b) => new Date(b.submittedAt).getTime() - new Date(a.submittedAt).getTime());
    }

    return filtered;
  }, [submissions, formDef, debouncedSearchTerm, debouncedColumnFilters, sortConfig]);

  const requestSort = (key: string) => {
    let direction: 'ascending' | 'descending' = 'ascending';
    if (sortConfig && sortConfig.key === key && sortConfig.direction === 'ascending') {
      direction = 'descending';
    }
    setSortConfig({ key, direction });
  };

  const getSortIcon = (key: string) => {
    if (!sortConfig || sortConfig.key !== key) return <span className="text-slate-500">↕</span>;
    return sortConfig.direction === 'ascending' ? <span className="text-sky-300">↑</span> : <span className="text-sky-300">↓</span>;
  };

  const handleColumnFilterChange = (columnId: string, value: string | { start?: string; end?: string }) => {
    setColumnFilters(prev => ({ ...prev, [columnId]: value }));
  };

  const clearAllFilters = () => {
    setSearchTerm('');
    setColumnFilters({});
  };

  const renderFieldValueForDisplay = (field: FormFieldDefinition, value: any): React.ReactNode => {
    if (value === null || value === undefined || (Array.isArray(value) && value.length === 0)) {
        return <span className="italic text-slate-500">غير متوفر</span>;
    }
    switch (field.type) {
        case FieldType.PASSWORD: return '********';
        case FieldType.CHECKBOX: return value ? 'محدد' : 'غير محدد';
        case FieldType.CHECKBOX_GROUP: return Array.isArray(value) ? value.join(', ') : <span className="italic text-slate-500">غير صالح</span>;
        case FieldType.FILE:
            const fileMeta = value as FileValue;
            return fileMeta?.name ? `${fileMeta.name} (${(fileMeta.size / 1024).toFixed(2)} ك.ب)` : <span className="italic text-slate-500">لا يوجد ملف</span>;
        case FieldType.DATE: return value ? new Date(value).toLocaleDateString('ar-EG') : <span className="italic text-slate-500">غير متوفر</span>;
        case FieldType.TIME: return String(value); 
        case FieldType.DATETIME_LOCAL: return value ? new Date(value).toLocaleString('ar-EG') : <span className="italic text-slate-500">غير متوفر</span>;
        default: return value.toString();
    }
  };
  
  const getPlainStringForExport = (field: FormFieldDefinition | null, value: any): string => {
    if (value === null || value === undefined || (Array.isArray(value) && value.length === 0)) return "غير متوفر";
    if (!field) {
      if (value instanceof Date) return value.toISOString();
      return String(value);
    }
    switch (field.type) {
        case FieldType.PASSWORD: return '********';
        case FieldType.CHECKBOX: return value ? 'محدد' : 'غير محدد';
        case FieldType.CHECKBOX_GROUP: return Array.isArray(value) ? value.join(', ') : "بيانات غير صالحة";
        case FieldType.FILE:
            const fileMeta = value as FileValue;
            return fileMeta?.name ? `${fileMeta.name} (${(fileMeta.size / 1024).toFixed(2)} ك.ب)` : "لا يوجد ملف";
        case FieldType.DATE: return value ? new Date(value).toISOString().split('T')[0] : "غير متوفر";
        case FieldType.TIME: return String(value); 
        case FieldType.DATETIME_LOCAL: return value ? new Date(value).toISOString() : "غير متوفر";
        default: return String(value);
    }
  };

  const handleExport = (format: 'json' | 'excel' | 'pdf') => {
    if (!formDef) return;

    const exportHeaders: string[] = ['وقت التقديم', ...formDef.fields.map(f => f.label), 'خط العرض', 'خط الطول'];
    const exportBody = processedSubmissions.map(sub => {
      const row: any[] = [
        getPlainStringForExport(null, new Date(sub.submittedAt)),
        ...formDef.fields.map(field => getPlainStringForExport(field, sub.data[field.id])),
        getPlainStringForExport(null, sub.latitude),
        getPlainStringForExport(null, sub.longitude),
      ];
      return row;
    });
    
    const filenameBase = `${formDef.name.replace(/\s+/g, '_')}_التقديمات`;

    if (format === 'json') {
      const dataToExport = {
        formName: formDef.name, formId: formDef.id, exportedAt: new Date().toISOString(),
        submissions: processedSubmissions.map(sub => ({
          submittedAt: sub.submittedAt, data: sub.data, latitude: sub.latitude, longitude: sub.longitude,
        }))
      };
      const jsonString = JSON.stringify(dataToExport, null, 2);
      const blob = new Blob([jsonString], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a'); a.href = url; a.download = `${filenameBase}.json`;
      document.body.appendChild(a); a.click(); document.body.removeChild(a); URL.revokeObjectURL(url);
    } else if (format === 'excel') {
      const wsData = [exportHeaders, ...exportBody.map(row => row.map(cell => cell ?? "غير متوفر"))]; 
      const worksheet = XLSX.utils.aoa_to_sheet(wsData);
      const workbook = XLSX.utils.book_new();
      XLSX.utils.book_append_sheet(workbook, worksheet, "التقديمات");
      XLSX.writeFile(workbook, `${filenameBase}.xlsx`);
    } else if (format === 'pdf') {
        const doc = new jsPDF({ orientation: 'landscape' });
        (doc as any).autoTable({
            head: [exportHeaders],
            body: exportBody,
            styles: { font: 'helvetica' }, 
            headStyles: { fillColor: [22, 160, 133] },
        });
        doc.save(`${filenameBase}.pdf`);
    }
  };

  // Chart Effect
  useEffect(() => {
    if (chartInstanceRef.current) {
        chartInstanceRef.current.destroy();
        chartInstanceRef.current = null;
    }
    if (showMapView) return; // Don't render chart if map is shown

    if (pieChartRef.current && formDef && chartFieldId && processedSubmissions.length > 0) {
        const selectedField = formDef.fields.find(f => f.id === chartFieldId);
        if (!selectedField || !(selectedField.type === FieldType.SELECT || selectedField.type === FieldType.RADIO_GROUP || selectedField.type === FieldType.CHECKBOX) || ( (selectedField.type === FieldType.SELECT || selectedField.type === FieldType.RADIO_GROUP) && (!selectedField.options || selectedField.options.length ===0))) {
             if (pieChartRef.current.getContext('2d')) { 
                const ctx = pieChartRef.current.getContext('2d')!;
                ctx.clearRect(0, 0, pieChartRef.current.width, pieChartRef.current.height);
                ctx.font = "16px Arial";
                ctx.fillStyle = "#94a3b8"; 
                ctx.textAlign = "center";
                ctx.fillText("الحقل المختار غير مناسب للرسم البياني أو لا يحتوي على خيارات.", pieChartRef.current.width / 2, pieChartRef.current.height / 2);
            }
            return;
        }

        const counts: Record<string, number> = {};
        let chartLabels: string[] = [];

        if (selectedField.type === FieldType.CHECKBOX) {
            chartLabels = ['محدد', 'غير محدد'];
            chartLabels.forEach(opt => counts[opt] = 0);
            processedSubmissions.forEach(sub => {
                const value = sub.data[chartFieldId];
                if (value === true) counts['محدد']++;
                else counts['غير محدد']++;
            });
        } else if (selectedField.options) { 
            chartLabels = [...selectedField.options, "بدون إجابة"];
            chartLabels.forEach(opt => counts[opt] = 0);
            processedSubmissions.forEach(sub => {
                const value = sub.data[chartFieldId];
                if (value && typeof value === 'string' && selectedField.options?.includes(value)) {
                    counts[value]++;
                } else {
                    counts["بدون إجابة"]++;
                }
            });
        }
        
        const activeLabels = chartLabels.filter(key => counts[key] > 0);
        const data = activeLabels.map(label => counts[label]);

        if (data.length === 0) {
            if (pieChartRef.current.getContext('2d')) {
                const ctx = pieChartRef.current.getContext('2d')!;
                ctx.clearRect(0, 0, pieChartRef.current.width, pieChartRef.current.height);
                ctx.font = "16px Arial";
                ctx.fillStyle = "#94a3b8"; 
                ctx.textAlign = "center";
                ctx.fillText("لا توجد بيانات لعرضها في الرسم البياني للحقل المحدد.", pieChartRef.current.width / 2, pieChartRef.current.height / 2);
            }
            return;
        }

        const chartOptions: any = {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'top', labels:{color: '#e2e8f0'} },
                title: { display: true, text: `توزيع ${selectedField.label}`, color: '#e2e8f0' }
            }
        };

        const datasetConfig: any = {
            label: selectedField.label,
            data: data,
        };

        if (selectedChartType === 'line' || selectedChartType === 'bar') {
            chartOptions.scales = {
                x: { ticks: { color: '#94a3b8' } },
                y: { ticks: { color: '#94a3b8' }, beginAtZero: true, suggestedMax: Math.max(...data, 0) + 1 }
            };
            if (selectedChartType === 'line') {
                datasetConfig.borderColor = CHART_PALETTE[0];
                datasetConfig.backgroundColor = CHART_PALETTE[0].replace('0.8', '0.1');
                datasetConfig.tension = 0.1;
                datasetConfig.fill = true;
            } else { 
                datasetConfig.backgroundColor = data.map((_, i) => CHART_PALETTE[i % CHART_PALETTE.length]); 
            }
        } else { 
            datasetConfig.backgroundColor = data.map((_, i) => CHART_PALETTE[i % CHART_PALETTE.length]);
            delete chartOptions.scales;
        }
        
        chartInstanceRef.current = new Chart(pieChartRef.current, {
            type: selectedChartType as keyof ChartTypeRegistry,
            data: {
                labels: activeLabels,
                datasets: [datasetConfig]
            },
            options: chartOptions
        });
    } else if (pieChartRef.current && pieChartRef.current.getContext('2d') && !showMapView) { 
        const ctx = pieChartRef.current.getContext('2d')!;
        ctx.clearRect(0, 0, pieChartRef.current.width, pieChartRef.current.height);
        if (processedSubmissions.length > 0 && !chartFieldId) {
             ctx.font = "16px Arial";
             ctx.fillStyle = "#94a3b8"; 
             ctx.textAlign = "center";
             ctx.fillText("الرجاء اختيار حقل لعرض الرسم البياني.", pieChartRef.current.width / 2, pieChartRef.current.height / 2);
        }
    }
  }, [processedSubmissions, formDef, chartFieldId, selectedChartType, showMapView]);


  // Map Effects
  useEffect(() => {
    if (showMapView && mapContainerRef.current && !leafletMapInstanceRef.current) {
      const map = L.map(mapContainerRef.current).setView([24.7136, 46.6753], 5); // Default KSA view

      L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
      }).addTo(map);
      leafletMapInstanceRef.current = map;
    }
  }, [showMapView]);

  useEffect(() => {
    if (!showMapView && leafletMapInstanceRef.current) {
      leafletMapInstanceRef.current.remove();
      leafletMapInstanceRef.current = null;
      markersRef.current = []; // Clear marker refs
    }
  }, [showMapView]);

  useEffect(() => {
    if (showMapView && leafletMapInstanceRef.current && formDef) {
      // Clear existing markers from map and from ref array
      markersRef.current.forEach(marker => marker.remove());
      markersRef.current = [];

      const submissionsWithGeo = processedSubmissions.filter(
        sub => sub.latitude != null && sub.longitude != null
      );

      if (submissionsWithGeo.length > 0) {
        const newMarkers: LeafletMarker[] = [];
        submissionsWithGeo.forEach(submission => {
          const marker = L.marker([submission.latitude!, submission.longitude!]);
          marker.on('click', () => {
            setSelectedSubmissionOnMap(submission);
            if(leafletMapInstanceRef.current) { // Pan to marker on click
                leafletMapInstanceRef.current.setView([submission.latitude!, submission.longitude!], leafletMapInstanceRef.current.getZoom());
            }
          });
          marker.bindPopup(`<b>${formDef.name}</b><br/>تقديم: ${new Date(submission.submittedAt).toLocaleTimeString('ar-EG')}`);
          marker.addTo(leafletMapInstanceRef.current!);
          newMarkers.push(marker);
        });
        markersRef.current = newMarkers;

        if (newMarkers.length > 0) {
          const featureGroup = L.featureGroup(newMarkers);
          leafletMapInstanceRef.current.fitBounds(featureGroup.getBounds().pad(0.2));
        }
      } else {
        // No geo-submissions, maybe zoom to default or show message
        leafletMapInstanceRef.current.setView([24.7136, 46.6753], 5);
      }
    }
  }, [showMapView, processedSubmissions, formDef]);

  // Column Resizing Effects & Handlers
  const handleMouseDown = useCallback((e: React.MouseEvent<HTMLDivElement>, columnKey: string) => {
    if (!tableRef.current) return;
    const thElement = (e.target as HTMLElement).closest('th');
    if (!thElement) return;

    document.body.style.cursor = 'col-resize';
    document.body.style.userSelect = 'none';

    setActiveDrag({
      columnKey,
      startX: e.clientX,
      startWidth: thElement.offsetWidth,
    });
  }, []);

  const handleMouseMove = useCallback((e: MouseEvent) => {
    if (!activeDrag || !tableRef.current) return;
    const newWidth = Math.max(MIN_COLUMN_WIDTH, activeDrag.startWidth + (e.clientX - activeDrag.startX));
    setColumnWidths(prev => ({ ...prev, [activeDrag.columnKey]: newWidth }));
  }, [activeDrag]);

  const handleMouseUp = useCallback(() => {
    if (activeDrag) {
      const storageKey = getLocalStorageColumnWidthsKey();
      if (storageKey) {
        localStorage.setItem(storageKey, JSON.stringify(columnWidths));
      }
      setActiveDrag(null);
      document.body.style.cursor = 'default';
      document.body.style.userSelect = 'auto';
    }
  }, [activeDrag, columnWidths, getLocalStorageColumnWidthsKey]);

  useEffect(() => {
    if (activeDrag) {
      document.addEventListener('mousemove', handleMouseMove);
      document.addEventListener('mouseup', handleMouseUp);
    }
    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, [activeDrag, handleMouseMove, handleMouseUp]);


  if (isLoading) return <div className="flex justify-center items-center h-64"><LoadingSpinner text="جاري تحميل التقديمات..." /></div>;
  if (error) return <p className="text-red-400 text-center text-xl">{error}</p>;
  if (!formDef) return <p className="text-gray-400 text-center text-xl">تعريف النموذج غير متوفر.</p>;

  const hasActiveFilters = searchTerm || Object.values(columnFilters).some(f => f && (typeof f === 'string' || (f as any).start || (f as any).end || typeof f === 'boolean'));
  const categoricalFieldsForChart = formDef.fields.filter(f => ((f.type === FieldType.SELECT || f.type === FieldType.RADIO_GROUP) && f.options && f.options.length > 0) || f.type === FieldType.CHECKBOX );
  const submissionsWithGeoCount = submissions.filter(s => s.latitude != null && s.longitude != null).length;
  
  const getColumnWidth = (key: string) => {
    return columnWidths[key] ? `${columnWidths[key]}px` : 'auto';
  };

  return (
    <div className="bg-slate-800 p-6 sm:p-8 rounded-lg shadow-2xl space-y-8">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
        <div>
            <h1 className="text-3xl font-bold text-sky-400">{formDef.name} - التقديمات</h1>
            <p className="text-slate-400">عرض {processedSubmissions.length} من {submissions.length} إجمالي التقديمات</p>
        </div>
        <div className="flex flex-wrap gap-2 items-center">
            {submissionsWithGeoCount > 0 && (
                <button 
                    onClick={() => {
                        setShowMapView(!showMapView);
                        setSelectedSubmissionOnMap(null); // Reset selection when toggling
                        if(!showMapView && leafletMapInstanceRef.current) { // About to show map
                            // Invalidate map size after it becomes visible
                            setTimeout(() => leafletMapInstanceRef.current?.invalidateSize(), 0);
                        }
                    }}
                    className="bg-cyan-500 hover:bg-cyan-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors"
                    aria-pressed={showMapView}
                >
                    {showMapView ? 'عرض الجدول' : `عرض الخريطة (${submissionsWithGeoCount})`}
                </button>
            )}
            <Link to={`/form/${formDef.id}/fill`} className="bg-green-500 hover:bg-green-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تعبئة هذا النموذج</Link>
            {submissions.length > 0 && !showMapView && (<>
              <button onClick={() => handleExport('json')} className="bg-indigo-500 hover:bg-indigo-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تصدير JSON</button>
              <button onClick={() => handleExport('excel')} className="bg-emerald-500 hover:bg-emerald-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تصدير Excel</button>
              <button onClick={() => handleExport('pdf')} className="bg-rose-500 hover:bg-rose-600 text-white font-semibold py-2 px-4 rounded-md shadow-sm transition-colors">تصدير PDF</button>
            </>)}
        </div>
      </div>
    
    {showMapView ? (
        <div className="mt-6 space-y-4">
            <div ref={mapContainerRef} className="h-[500px] md:h-[600px] w-full rounded-lg shadow-md border border-slate-700 bg-slate-700/30">
                {!leafletMapInstanceRef.current && <LoadingSpinner text="جاري تحميل الخريطة..." />}
            </div>
            {selectedSubmissionOnMap && formDef && (
            <div className="p-4 bg-slate-700/50 rounded-md shadow-lg border border-slate-600">
                <div className="flex justify-between items-center mb-3">
                    <h3 className="text-xl font-semibold text-sky-300">تفاصيل الرد المحدد على الخريطة</h3>
                    <button 
                        onClick={() => setSelectedSubmissionOnMap(null)} 
                        className="text-slate-400 hover:text-slate-200"
                        aria-label="إغلاق تفاصيل الرد"
                    >
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-6 h-6">
                         <path strokeLinecap="round" strokeLinejoin="round" d="M6 18L18 6M6 6l12 12" />
                        </svg>
                    </button>
                </div>
                <div className="space-y-2 text-sm text-slate-200 max-h-96 overflow-y-auto pr-2">
                    <p><strong>وقت التقديم:</strong> {new Date(selectedSubmissionOnMap.submittedAt).toLocaleString('ar-EG')}</p>
                    {formDef.fields.map(field => {
                    if (selectedSubmissionOnMap.data.hasOwnProperty(field.id)) {
                        return (
                        <div key={field.id} className="mb-1 py-1 border-b border-slate-600 last:border-b-0">
                            <strong className="text-slate-400">{field.label}:</strong>{' '}
                            <span className="break-all">{renderFieldValueForDisplay(field, selectedSubmissionOnMap.data[field.id])}</span>
                        </div>
                        );
                    }
                    return null;
                    })}
                    {selectedSubmissionOnMap.latitude && selectedSubmissionOnMap.longitude && (
                        <p className="mt-2 pt-2 border-t border-slate-600">
                            <strong className="text-slate-400">الموقع:</strong> {selectedSubmissionOnMap.latitude.toFixed(5)}, {selectedSubmissionOnMap.longitude.toFixed(5)}
                            {' '}
                            <a
                                href={`https://www.google.com/maps?q=${selectedSubmissionOnMap.latitude},${selectedSubmissionOnMap.longitude}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="text-sky-400 hover:text-sky-300 inline-flex items-center"
                            >
                                (فتح في خرائط جوجل)
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-4 h-4 ml-1">
                                <path strokeLinecap="round" strokeLinejoin="round" d="M13.5 6H5.25A2.25 2.25 0 003 8.25v10.5A2.25 2.25 0 005.25 21h10.5A2.25 2.25 0 0018 18.75V10.5m-10.5 6L21 3m0 0h-5.25M21 3v5.25" />
                                </svg>
                            </a>
                        </p>
                    )}
                </div>
            </div>
            )}
            {submissionsWithGeoCount === 0 && (
                <p className="text-center text-slate-400 py-4">لا توجد تقديمات تحتوي على بيانات الموقع لعرضها على الخريطة.</p>
            )}
        </div>
    ) : (
      <>
        <div className="space-y-3">
            <div className="flex flex-col sm:flex-row sm:items-center sm:space-x-3 gap-3">
                <input type="text" placeholder="بحث عام..." value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} className="w-full sm:flex-1 px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200"/>
                {hasActiveFilters && (
                    <button onClick={clearAllFilters} className="px-4 py-2 text-sm font-medium text-amber-300 bg-slate-700 hover:bg-slate-600 rounded-md border border-amber-400/50">مسح جميع الفلاتر</button>
                )}
            </div>
        </div>

        {processedSubmissions.length === 0 && submissions.length > 0 ? (
            <div className="text-center py-10 border-2 border-dashed border-slate-700 rounded-lg"><p className="text-xl text-slate-400">لا توجد تقديمات تطابق معايير الفلترة الحالية.</p></div>
        ) : submissions.length === 0 ? (
            <div className="text-center py-10 border-2 border-dashed border-slate-700 rounded-lg">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-12 h-12 mx-auto text-slate-500 mb-3"><path strokeLinecap="round" strokeLinejoin="round" d="M18.375 12.739l-7.693 7.693a4.5 4.5 0 01-6.364-6.364l10.94-10.94A3.375 3.375 0 0116.5 7.5c0 1.001-.393 1.914-1.052 2.613l-7.693 7.693a1.125 1.125 0 001.59 1.59l7.693-7.693a1.125 1.125 0 000-1.59Z" /></svg>
            <p className="text-xl text-slate-400">لا توجد تقديمات لهذا النموذج حتى الآن.</p>
            </div>
        ) : (
            <div className="overflow-x-auto shadow-md rounded-lg">
            <table ref={tableRef} className="min-w-full divide-y divide-slate-700" style={{ tableLayout: 'fixed' }}>
                <thead className="bg-slate-700/50">
                <tr>
                    <th scope="col" onClick={() => requestSort('_submittedAt')} 
                        style={{ width: getColumnWidth('_submittedAt'), minWidth: `${MIN_COLUMN_WIDTH}px`, position: 'relative' }}
                        className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider sticky left-0 bg-slate-700/80 backdrop-blur-sm z-10 cursor-pointer">
                        وقت التقديم {getSortIcon('_submittedAt')}
                         <div onMouseDown={(e) => handleMouseDown(e, '_submittedAt')} className="absolute top-0 right-0 w-1.5 h-full cursor-col-resize bg-slate-500/30 hover:bg-sky-400/50 opacity-0 hover:opacity-100 transition-opacity" />
                    </th>
                    {formDef.fields.map(field => (
                    <th key={field.id} scope="col" onClick={() => requestSort(field.id)} 
                        style={{ width: getColumnWidth(field.id), minWidth: `${MIN_COLUMN_WIDTH}px`, position: 'relative' }}
                        className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                        {field.label} {getSortIcon(field.id)}
                         <div onMouseDown={(e) => handleMouseDown(e, field.id)} className="absolute top-0 right-0 w-1.5 h-full cursor-col-resize bg-slate-500/30 hover:bg-sky-400/50 opacity-0 hover:opacity-100 transition-opacity" />
                    </th>
                    ))}
                    <th scope="col" onClick={() => requestSort('_latitude')} 
                        style={{ width: getColumnWidth('_latitude'), minWidth: `${MIN_COLUMN_WIDTH}px`, position: 'relative' }}
                        className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                        خط العرض {getSortIcon('_latitude')}
                         <div onMouseDown={(e) => handleMouseDown(e, '_latitude')} className="absolute top-0 right-0 w-1.5 h-full cursor-col-resize bg-slate-500/30 hover:bg-sky-400/50 opacity-0 hover:opacity-100 transition-opacity" />
                    </th>
                    <th scope="col" onClick={() => requestSort('_longitude')} 
                        style={{ width: getColumnWidth('_longitude'), minWidth: `${MIN_COLUMN_WIDTH}px`, position: 'relative' }}
                        className="px-6 py-3 text-left text-xs font-medium text-sky-300 uppercase tracking-wider cursor-pointer">
                        خط الطول {getSortIcon('_longitude')}
                         <div onMouseDown={(e) => handleMouseDown(e, '_longitude')} className="absolute top-0 right-0 w-1.5 h-full cursor-col-resize bg-slate-500/30 hover:bg-sky-400/50 opacity-0 hover:opacity-100 transition-opacity" />
                    </th>
                </tr>
                <tr className="bg-slate-700/30">
                    <td className="px-2 py-1 sticky left-0 bg-slate-700/60 backdrop-blur-sm z-10" style={{ width: getColumnWidth('_submittedAt'), minWidth: `${MIN_COLUMN_WIDTH}px` }}>
                    <div className="flex space-x-1">
                        <input type="date" placeholder="من" value={columnFilters['_submittedAt']?.start || ''} onChange={(e) => handleColumnFilterChange('_submittedAt', {...(columnFilters['_submittedAt'] || {}), start: e.target.value})} className="w-1/2 text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200"/>
                        <input type="date" placeholder="إلى" value={columnFilters['_submittedAt']?.end || ''} onChange={(e) => handleColumnFilterChange('_submittedAt', {...(columnFilters['_submittedAt'] || {}), end: e.target.value})} className="w-1/2 text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200"/>
                    </div>
                    </td>
                    {formDef.fields.map(field => (
                    <td key={`${field.id}-filter`} className="px-2 py-1" style={{ width: getColumnWidth(field.id), minWidth: `${MIN_COLUMN_WIDTH}px` }}>
                        {(field.type === FieldType.DATE || field.type === FieldType.DATETIME_LOCAL) ? (
                        <div className="flex space-x-1">
                            <input type="date" placeholder="من" value={columnFilters[field.id]?.start || ''} onChange={(e) => handleColumnFilterChange(field.id, {...(columnFilters[field.id] || {}), start: e.target.value})} className="w-1/2 text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200"/>
                            <input type="date" placeholder="إلى" value={columnFilters[field.id]?.end || ''} onChange={(e) => handleColumnFilterChange(field.id, {...(columnFilters[field.id] || {}), end: e.target.value})} className="w-1/2 text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200"/>
                        </div>
                        ) : (field.type === FieldType.SELECT || field.type === FieldType.RADIO_GROUP) && field.options ? (
                        <select value={columnFilters[field.id] || ''} onChange={(e) => handleColumnFilterChange(field.id, e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200">
                            <option value="">الكل</option>
                            {field.options.map(opt => <option key={opt} value={opt}>{opt}</option>)}
                        </select>
                        ) : field.type === FieldType.CHECKBOX ? (
                            <select value={columnFilters[field.id] || ''} onChange={(e) => handleColumnFilterChange(field.id, e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200">
                                <option value="">الكل</option>
                                <option value="محدد">محدد</option>
                                <option value="غير محدد">غير محدد</option>
                            </select>
                        ) : (
                        <input type="text" placeholder="فلترة..." value={columnFilters[field.id] || ''} onChange={(e) => handleColumnFilterChange(field.id, e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" />
                        )}
                    </td>
                    ))}
                    <td className="px-2 py-1" style={{ width: getColumnWidth('_latitude'), minWidth: `${MIN_COLUMN_WIDTH}px` }}><input type="text" placeholder="فلترة..." value={columnFilters['_latitude'] || ''} onChange={(e) => handleColumnFilterChange('_latitude', e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" /></td>
                    <td className="px-2 py-1" style={{ width: getColumnWidth('_longitude'), minWidth: `${MIN_COLUMN_WIDTH}px` }}><input type="text" placeholder="فلترة..." value={columnFilters['_longitude'] || ''} onChange={(e) => handleColumnFilterChange('_longitude', e.target.value)} className="w-full text-xs px-2 py-1 bg-slate-600 border-slate-500 rounded focus:ring-sky-400 text-gray-200" /></td>
                </tr>
                </thead>
                <tbody className="bg-slate-800 divide-y divide-slate-700">
                {processedSubmissions.map(submission => (
                    <tr key={submission.id} className="hover:bg-slate-700/40 transition-colors">
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300 sticky left-0 bg-slate-800 hover:bg-slate-700/40 z-0" style={{ width: getColumnWidth('_submittedAt'), minWidth: `${MIN_COLUMN_WIDTH}px` }}>{new Date(submission.submittedAt).toLocaleString('ar-EG')}</td>
                    {formDef.fields.map(field => (
                        <td key={`${submission.id}-${field.id}`} className="px-6 py-4 whitespace-normal break-words text-sm text-slate-300 max-w-xs" style={{ width: getColumnWidth(field.id), minWidth: `${MIN_COLUMN_WIDTH}px` }}>{renderFieldValueForDisplay(field, submission.data[field.id])}</td>
                    ))}
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300" style={{ width: getColumnWidth('_latitude'), minWidth: `${MIN_COLUMN_WIDTH}px` }}>{submission.latitude !== undefined ? submission.latitude.toFixed(5) : <span className="italic text-slate-500">غير متوفر</span>}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-300" style={{ width: getColumnWidth('_longitude'), minWidth: `${MIN_COLUMN_WIDTH}px` }}>{submission.longitude !== undefined ? submission.longitude.toFixed(5) : <span className="italic text-slate-500">غير متوفر</span>}</td>
                    </tr>
                ))}
                </tbody>
            </table>
            </div>
        )}

        {!showMapView && categoricalFieldsForChart.length > 0 && (
            <div className="mt-8 p-6 bg-slate-700/30 rounded-lg border border-slate-600">
                <h3 className="text-xl font-semibold text-sky-400 mb-4">الرسوم البيانية للتقديمات</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                    <div>
                        <label htmlFor="chartFieldSelect" className="block text-sm font-medium text-gray-300 mb-1">اختر حقلاً لعرض توزيعه:</label>
                        <select 
                            id="chartFieldSelect" 
                            value={chartFieldId} 
                            onChange={(e) => setChartFieldId(e.target.value)}
                            className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200"
                        >
                            <option value="">-- اختر حقلاً --</option>
                            {categoricalFieldsForChart.map(field => (
                                <option key={field.id} value={field.id}>{field.label}</option>
                            ))}
                        </select>
                    </div>
                    <div>
                        <label htmlFor="chartTypeSelectPage" className="block text-sm font-medium text-gray-300 mb-1">نوع الرسم البياني:</label>
                        <select 
                            id="chartTypeSelectPage" 
                            value={selectedChartType} 
                            onChange={(e) => setSelectedChartType(e.target.value as SelectableChartType)}
                            className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200"
                        >
                            <option value="pie">Pie Chart (دائري)</option>
                            <option value="doughnut">Doughnut Chart (دونات)</option>
                            <option value="bar">Bar Chart (أعمدة)</option>
                            <option value="line">Line Chart (خطي)</option>
                        </select>
                    </div>
                </div>
                <div className="relative h-96 w-full"> 
                    {processedSubmissions.length > 0 && chartFieldId ? (
                        <canvas ref={pieChartRef}></canvas>
                    ) : (
                        <p className="text-slate-400 text-center pt-10">
                            {submissions.length === 0 ? "لا توجد تقديمات لعرض الرسوم البيانية." : 
                            !chartFieldId ? "الرجاء اختيار حقل لعرض الرسم البياني." : 
                            "لا توجد بيانات تطابق الفلترة الحالية لعرض الرسم البياني."}
                        </p>
                    )}
                </div>
            </div>
        )}

        {!showMapView && (
             <div className="mt-8 p-4 bg-slate-700/30 rounded-lg border border-slate-600">
                <h3 className="text-md font-semibold text-sky-400 mb-1">ملاحظة حول تصدير البيانات</h3>
                <p className="text-sm text-slate-400">يوفر تصدير البيانات بصيغة JSON أو Excel أو PDF طريقة لإخراج بياناتك من هذا التطبيق المحلي. يمكن بعد ذلك استخدام هذه البيانات لاستيرادها إلى أنظمة أخرى أو قاعدة بيانات خلفية. يتم تضمين بيانات تحديد الموقع الجغرافي (خط العرض/خط الطول) إذا تم التقاطها. تتضمن بيانات حقول "الملف" بيانات وصفية (الاسم، النوع، الحجم)، وليس محتوى الملف نفسه. يتم تطبيق معايير الفلترة والبحث على البيانات المصدرة.</p>
            </div>
        )}
      </>
    )}
    </div>
  );
};

export default ViewSubmissionsPage;