
import { FormDefinition, FormSubmission, FormGroup } from '../types';

// The base URL for your backend API. This should be configured appropriately.
// For development, it might be 'http://localhost:YOUR_BACKEND_PORT/api'
const API_BASE_URL = '/api';

// Helper function to handle API responses
async function handleResponse<T>(response: Response): Promise<T> {
  if (!response.ok) {
    let errorData;
    try {
      errorData = await response.json();
    } catch (e) {
      // If response is not JSON, use text
      errorData = { message: await response.text() || response.statusText };
    }
    const error = new Error(errorData?.message || `API error: ${response.status} ${response.statusText}`);
    (error as any).status = response.status;
    (error as any).errorData = errorData;
    console.error('API Error:', errorData, 'Status:', response.status);
    throw error;
  }
  // Check if response has content before trying to parse JSON
  const contentType = response.headers.get("content-type");
  if (contentType && contentType.indexOf("application/json") !== -1) {
    // For DELETE requests that might return 204 with no content but still have application/json header
    if (response.status === 204) {
        return Promise.resolve(undefined as unknown as T);
    }
    return response.json() as Promise<T>;
  }
  // For 204 No Content or other non-JSON success responses (like DELETE often is)
  return Promise.resolve(undefined as unknown as T); 
}

// Form Definition Management
export const getFormDefinitions = async (): Promise<FormDefinition[]> => {
  const response = await fetch(`${API_BASE_URL}/forms`);
  return handleResponse<FormDefinition[]>(response);
};

export const saveFormDefinition = async (form: FormDefinition): Promise<FormDefinition> => {
  const response = await fetch(`${API_BASE_URL}/forms`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(form),
  });
  return handleResponse<FormDefinition>(response);
};

export const getFormDefinitionById = async (id: string): Promise<FormDefinition | undefined> => {
  const response = await fetch(`${API_BASE_URL}/forms/${id}`);
  if (response.status === 404) { 
    const err = new Error(`Form with ID ${id} not found.`);
    (err as any).status = 404;
    (err as any).errorData = { message: `Form with ID ${id} not found.`}; // Add errorData for consistency
    throw err; 
  }
  return handleResponse<FormDefinition>(response);
};

export const deleteFormDefinition = async (id: string): Promise<void> => {
  const response = await fetch(`${API_BASE_URL}/forms/${id}`, { method: 'DELETE' });
  await handleResponse<void>(response); 
};

// Form Submission Management
export const getFormSubmissions = async (formId: string): Promise<FormSubmission[]> => {
  const response = await fetch(`${API_BASE_URL}/forms/${formId}/submissions`);
  return handleResponse<FormSubmission[]>(response);
};

export const saveFormSubmission = async (submission: FormSubmission): Promise<FormSubmission> => {
  const response = await fetch(`${API_BASE_URL}/forms/${submission.formId}/submissions`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ ...submission, submission_id: submission.id }),
  });
  return handleResponse<FormSubmission>(response);
};

export const getAllSubmissions = async (): Promise<FormSubmission[]> => {
  const response = await fetch(`${API_BASE_URL}/submissions/all`); 
  return handleResponse<FormSubmission[]>(response);
};


// Form Group Management
export const getFormGroups = async (): Promise<FormGroup[]> => {
  const response = await fetch(`${API_BASE_URL}/form-groups`);
  return handleResponse<FormGroup[]>(response);
};

export const saveFormGroup = async (group: FormGroup): Promise<FormGroup> => {
  const response = await fetch(`${API_BASE_URL}/form-groups`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(group),
  });
  return handleResponse<FormGroup>(response);
};

export const deleteFormGroup = async (groupId: string): Promise<void> => {
  const response = await fetch(`${API_BASE_URL}/form-groups/${groupId}`, { method: 'DELETE' });
  await handleResponse<void>(response);
};

export const assignFormToGroup = async (formId: string, groupId: string | null): Promise<void> => {
  // Backend endpoint `PUT /api/forms/:formId/assign-group`
  // The body should contain the new groupId, or null to unassign.
  // The backend will handle updating the form_group_forms join table.
  const response = await fetch(`${API_BASE_URL}/forms/${formId}/assign-group`, { 
    method: 'PUT', 
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ groupId: groupId }),
  });
  await handleResponse<void>(response);
};
