<?php
error_log("add_location.php: تم استدعاء الملف."); // <-- تسجيل في البداية تمامًا
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

// --- بداية التعديل: إعدادات الاتصال وإنشاء قاعدة بيانات الجهاز ---
require 'config.php'; // استخدام config.php للحصول على $conn الأولي

// ✅ تأكد من أن الاتصال بقاعدة البيانات الرئيسية يستخدم UTF-8
if ($conn) {
    $conn->set_charset("utf8mb4");
} else {
    // تم التعامل مع خطأ الاتصال في config.php
    exit; // توقف إذا فشل الاتصال الأولي
}

// --- بداية التعديل: دوال لإنشاء جدول لكل جهاز ---

// دالة لتنظيف اسم الجهاز ليكون جزءًا صالحًا من اسم الجدول
function sanitize_table_name_part($name) {
    // إزالة الأحرف غير الأبجدية الرقمية والشرطة السفلية
    $name = preg_replace('/[^a-zA-Z0-9_]/', '_', $name);
    // التأكد من أنه لا يبدأ برقم
    if (preg_match('/^[0-9]/', $name)) {
        $name = '_' . $name;
    }
    // تحديد طول أقصى (اسم الجدول الكلي له حد 64، نترك مساحة للبادئة)
    return substr($name, 0, 50); // تقصير لترك مساحة للبادئة
}

// دالة لإنشاء اسم الجدول الكامل والآمن
function create_safe_table_name($device_name) {
    $sanitized_part = sanitize_table_name_part($device_name);
    // استخدام بادئة لتجنب التعارض مع جداول أخرى وتسهيل التعرف عليها
    return 'device_data_' . $sanitized_part;
}


// دالة للتحقق من وإنشاء الجدول المركزي للأجهزة المسجلة (مع حقل table_name)
function ensure_registered_devices_table($main_conn, $main_db_name) {
    error_log("add_location.php: الدخول إلى ensure_registered_devices_table");
    try {
        if (!$main_conn->query("USE `$main_db_name`")) {
             error_log("add_location.php: فشل في استخدام قاعدة البيانات الرئيسية: " . $main_conn->error);
             return false;
        }
        $check_table_sql = "SHOW TABLES LIKE 'registered_devices'";
        error_log("add_location.php: التحقق من وجود جدول registered_devices...");
        $result = $main_conn->query($check_table_sql);
        if ($result === false) {
             error_log("add_location.php: خطأ في استعلام التحقق من جدول registered_devices: " . $main_conn->error);
             return false;
        }

        if ($result->num_rows == 0) {
            error_log("add_location.php: جدول registered_devices غير موجود، سيتم محاولة إنشائه.");
            // الجدول غير موجود، قم بإنشائه (مع table_name بدلاً من db_name)
            $create_table_sql = "
            CREATE TABLE `registered_devices` (
              `id` INT AUTO_INCREMENT PRIMARY KEY,
              `device_id` VARCHAR(255) NOT NULL UNIQUE,
              `device_name` VARCHAR(255) NOT NULL,
              `table_name` VARCHAR(64) NOT NULL UNIQUE, -- تم التغيير من db_name
              `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
            error_log("add_location.php: تنفيذ أمر إنشاء جدول registered_devices...");
            if (!$main_conn->query($create_table_sql)) {
                error_log("add_location.php: خطأ فادح في إنشاء جدول registered_devices: " . $main_conn->error);
                return false;
            }
            error_log("add_location.php: تم إنشاء جدول registered_devices بنجاح.");
        } else {
             error_log("add_location.php: جدول registered_devices موجود بالفعل.");
             // --- إضافة: التحقق من وجود عمود table_name وإضافته إذا لم يكن موجودًا (للتوافق مع التحديث) ---
             $check_column_sql = "SHOW COLUMNS FROM `registered_devices` LIKE 'table_name'";
             $col_result = $main_conn->query($check_column_sql);
             if ($col_result && $col_result->num_rows == 0) {
                 error_log("add_location.php: عمود 'table_name' غير موجود في registered_devices. محاولة إضافته...");
                 $add_column_sql = "ALTER TABLE `registered_devices` ADD COLUMN `table_name` VARCHAR(64) NULL UNIQUE AFTER `device_name`"; // يسمح بـ NULL مؤقتًا
                 if (!$main_conn->query($add_column_sql)) {
                      error_log("add_location.php: خطأ في إضافة عمود 'table_name': " . $main_conn->error);
                      // لا نعتبره خطأ فادحًا هنا، قد يتم إصلاحه لاحقًا
                 } else {
                      error_log("add_location.php: تم إضافة عمود 'table_name' بنجاح.");
                 }
                 // قد تحتاج لتحديث القيم القديمة إذا كان هناك عمود db_name سابق
                 // $update_sql = "UPDATE registered_devices SET table_name = CONCAT('device_data_', db_name) WHERE table_name IS NULL AND db_name IS NOT NULL";
                 // $main_conn->query($update_sql);
                 // $alter_sql = "ALTER TABLE `registered_devices` MODIFY COLUMN `table_name` VARCHAR(64) NOT NULL UNIQUE"; // جعله NOT NULL بعد التحديث
                 // $main_conn->query($alter_sql);
             }
             // --- نهاية الإضافة ---
        }
        error_log("add_location.php: الخروج من ensure_registered_devices_table بنجاح.");
        return true;
    } catch (Exception $e) {
        error_log("add_location.php: استثناء (Exception) في ensure_registered_devices_table: " . $e->getMessage());
        return false;
    }
}

// دالة للتحقق من وإنشاء جدول الجهاز المحدد داخل القاعدة الرئيسية
function ensure_device_table($main_conn, $main_db_name, $device_table_name) {
    error_log("add_location.php: الدخول إلى ensure_device_table لـ '$device_table_name' في '$main_db_name'");
    try {
        // التأكد من استخدام قاعدة البيانات الصحيحة
        if (!$main_conn->query("USE `$main_db_name`")) {
             error_log("add_location.php: فشل في استخدام قاعدة البيانات '$main_db_name' لإنشاء جدول الجهاز: " . $main_conn->error);
             return false;
        }

        // التحقق من وجود الجدول
        $check_table_sql = "SHOW TABLES LIKE ?"; // استخدام placeholder
        error_log("add_location.php: التحقق من وجود جدول '$device_table_name'...");
        $stmt_check = $main_conn->prepare($check_table_sql);
        if (!$stmt_check) {
            error_log("add_location.php: خطأ في تحضير استعلام التحقق من جدول '$device_table_name': " . $main_conn->error);
            return false;
        }
        $stmt_check->bind_param("s", $device_table_name);
        if (!$stmt_check->execute()) {
             error_log("add_location.php: خطأ في تنفيذ استعلام التحقق من جدول '$device_table_name': " . $stmt_check->error);
             $stmt_check->close();
             return false;
        }
        $result = $stmt_check->get_result();
        $table_exists = $result->num_rows > 0;
        $stmt_check->close();
        error_log("add_location.php: هل جدول '$device_table_name' موجود؟ " . ($table_exists ? 'نعم' : 'لا'));

        if (!$table_exists) {
            error_log("add_location.php: جدول '$device_table_name' غير موجود، محاولة الإنشاء...");
            // إنشاء الجدول (استخدام backticks حول اسم الجدول الديناميكي)
            $create_table_sql = "
            CREATE TABLE `$device_table_name` (
              `id` INT AUTO_INCREMENT PRIMARY KEY,
              `latitude` VARCHAR(50) NOT NULL,
              `longitude` VARCHAR(50) NOT NULL,
              `status` VARCHAR(255) DEFAULT NULL,
              `battery` VARCHAR(10) DEFAULT NULL,
              `timestamp` DATETIME NOT NULL,
               INDEX `idx_timestamp` (`timestamp`) -- إضافة فهرس للوقت لتحسين الاستعلامات
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
            error_log("add_location.php: تنفيذ أمر إنشاء جدول '$device_table_name'...");
            if (!$main_conn->query($create_table_sql)) {
                error_log("add_location.php: خطأ فادح في إنشاء جدول '$device_table_name': " . $main_conn->error);
                return false;
            }
            error_log("add_location.php: تم إنشاء جدول '$device_table_name' بنجاح.");
        }
        error_log("add_location.php: الخروج من ensure_device_table لـ '$device_table_name' بنجاح.");
        return true;

    } catch (Exception $e) {
        error_log("add_location.php: استثناء (Exception) في ensure_device_table لـ '$device_table_name': " . $e->getMessage());
        return false;
    }
}

// --- نهاية التعديل: دوال لإنشاء جدول لكل جهاز ---

// استقبال البيانات من الطلب (يجب أن يكون بعد الدوال)
$data = json_decode(file_get_contents("php://input"), true);

// --- بداية منطق المعالجة الرئيسي ---
if (!$data) {
    error_log("add_location.php: لم يتم استلام أي بيانات JSON.");
    echo json_encode(["status" => "error", "message" => "❌ لم يتم استلام أي بيانات JSON"]);
    if ($conn) $conn->close();
    exit;
}

error_log("add_location.php: البيانات المستقبلة: " . print_r($data, true));

// التحقق من وجود الحقول المطلوبة
$required_fields = ["device_id", "device_name", "latitude", "longitude", "status"];
$missing_fields = [];
foreach ($required_fields as $field) {
    if (!isset($data[$field]) || empty(trim($data[$field]))) { // التحقق من عدم الفراغ أيضاً
        $missing_fields[] = $field;
    }
}

if (!empty($missing_fields)) {
    $error_message = "❌ بيانات ناقصة أو فارغة: الحقول المطلوبة (" . implode(", ", $missing_fields) . ") غير موجودة أو فارغة";
    error_log("add_location.php: " . $error_message);
    echo json_encode(["status" => "error", "message" => $error_message]);
    if ($conn) $conn->close();
    exit;
}

// --- بداية التعديل: استخدام اسم الجدول بدلاً من اسم قاعدة البيانات ---
// تنظيف اسم الجهاز لإنشاء اسم جدول صالح
$original_device_name = $data["device_name"];
$device_table_name = create_safe_table_name($original_device_name); // إنشاء اسم الجدول الآمن
error_log("add_location.php: اسم الجهاز الأصلي: '$original_device_name', اسم الجدول المُنشأ: '$device_table_name'");
if (empty($device_table_name)) { // التحقق من اسم الجدول
     error_log("add_location.php: فشل في إنشاء اسم جدول صالح من '$original_device_name'. الإيقاف.");
     echo json_encode(["status" => "error", "message" => "❌ اسم الجهاز غير صالح لإنشاء جدول"]);
     if ($conn) $conn->close();
     exit;
}
// --- نهاية التعديل ---

// التأكد من وجود جدول الأجهزة المسجلة في قاعدة البيانات الرئيسية
error_log("add_location.php: استدعاء ensure_registered_devices_table...");
if (!ensure_registered_devices_table($conn, $dbname)) {
    error_log("add_location.php: فشل ensure_registered_devices_table. الإيقاف.");
    echo json_encode(["status" => "error", "message" => "❌ خطأ في إعداد جدول الأجهزة المسجلة"]);
    if ($conn) $conn->close();
    exit;
}
error_log("add_location.php: تم التأكد من جدول registered_devices بنجاح.");

// --- بداية التعديل: التأكد من وجود جدول الجهاز المحدد ---
// التأكد من وجود جدول الجهاز داخل القاعدة الرئيسية
error_log("add_location.php: استدعاء ensure_device_table لـ '$device_table_name'...");
if (!ensure_device_table($conn, $dbname, $device_table_name)) {
     error_log("add_location.php: فشل ensure_device_table لـ '$device_table_name'. الإيقاف.");
    echo json_encode(["status" => "error", "message" => "❌ خطأ في إنشاء أو التحقق من جدول الجهاز '$device_table_name'"]);
    if ($conn) $conn->close();
    exit;
}
error_log("add_location.php: تم التأكد من جدول الجهاز '$device_table_name' بنجاح.");
// --- نهاية التعديل ---

// --- بداية التعديل: تسجيل الجهاز مع اسم الجدول ---
// تسجيل الجهاز في الجدول المركزي إذا لم يكن مسجلاً من قبل
error_log("add_location.php: محاولة تسجيل/التحقق من الجهاز '{$data['device_id']}' في registered_devices...");
try {
    // التأكد من استخدام القاعدة الرئيسية
    if (!$conn->query("USE `$dbname`")) {
         throw new Exception("فشل في استخدام قاعدة البيانات الرئيسية للتسجيل: " . $conn->error);
    }
    // التحقق مما إذا كان الجهاز مسجلًا بالفعل
    $check_reg_sql = "SELECT id, table_name FROM registered_devices WHERE device_id = ?";
    $stmt_check_reg = $conn->prepare($check_reg_sql);
    if (!$stmt_check_reg) { throw new Exception("فشل تحضير استعلام التحقق من التسجيل: " . $conn->error); }
    $stmt_check_reg->bind_param("s", $data["device_id"]);
    if (!$stmt_check_reg->execute()) { throw new Exception("فشل تنفيذ استعلام التحقق من التسجيل: " . $stmt_check_reg->error); }
    $result_reg = $stmt_check_reg->get_result();
    $existing_device = $result_reg->fetch_assoc();
    $stmt_check_reg->close();

    if (!$existing_device) {
        error_log("add_location.php: الجهاز '{$data['device_id']}' غير مسجل في registered_devices. محاولة إضافته مع جدول '$device_table_name'...");
        // الجهاز غير مسجل، قم بتسجيله مع اسم الجدول
        $register_sql = "INSERT INTO registered_devices (device_id, device_name, table_name) VALUES (?, ?, ?)";
        $stmt_register = $conn->prepare($register_sql);
         if (!$stmt_register) {
             error_log("add_location.php: فشل تحضير استعلام التسجيل للجهاز '{$data['device_id']}': " . $conn->error);
             // لا نوقف التنفيذ هنا، قد يكون الجهاز مسجلًا من register_device.php
         } else {
            $stmt_register->bind_param("sss", $data["device_id"], $original_device_name, $device_table_name);
            if (!$stmt_register->execute()) {
                error_log("add_location.php: فشل تنفيذ استعلام التسجيل للجهاز '{$data['device_id']}': " . $stmt_register->error);
                // لا نوقف التنفيذ هنا
            } else {
                error_log("add_location.php: تم تسجيل الجهاز الجديد '{$data['device_id']}' بجدول '$device_table_name' بنجاح من add_location.php.");
            }
            $stmt_register->close();
         }

         // بعد محاولة التسجيل، نعيد التحقق للتأكد من وجود الجهاز الآن
         $check_reg_sql_after = "SELECT id, table_name FROM registered_devices WHERE device_id = ?";
         $stmt_check_reg_after = $conn->prepare($check_reg_sql_after);
         if ($stmt_check_reg_after) {
             $stmt_check_reg_after->bind_param("s", $data["device_id"]);
             $stmt_check_reg_after->execute();
             $result_reg_after = $stmt_check_reg_after->get_result();
             $existing_device = $result_reg_after->fetch_assoc(); // تحديث $existing_device
             $stmt_check_reg_after->close();
         } else {
              error_log("add_location.php: فشل تحضير استعلام التحقق الثاني من التسجيل للجهاز '{$data['device_id']}': " . $conn->error);
         }

         if (!$existing_device) {
             error_log("add_location.php: خطأ فادح! الجهاز '{$data['device_id']}' لا يزال غير مسجل في registered_devices بعد محاولة الإضافة. لا يمكن تحديد جدول الإدراج.");
             echo json_encode(["status" => "error", "message" => "❌ خطأ داخلي: الجهاز غير مسجل ولا يمكن تحديد جدول البيانات"]);
             if ($conn) $conn->close();
             exit; // توقف هنا لأننا لا نعرف أين ندرج البيانات
         } else {
              error_log("add_location.php: تم العثور على الجهاز '{$data['device_id']}' بعد محاولة الإضافة. اسم الجدول: '{$existing_device['table_name']}'.");
              $device_table_name = $existing_device['table_name']; // استخدم الاسم المسجل
         }

    } else {
         error_log("add_location.php: الجهاز '{$data['device_id']}' مسجل بالفعل بجدول '{$existing_device['table_name']}'.");
         // اختياري: التحقق مما إذا كان اسم الجدول المخزن يطابق الاسم الحالي
         if ($existing_device['table_name'] !== $device_table_name) {
             error_log("add_location.php: تحذير! اسم الجدول المسجل ('{$existing_device['table_name']}') يختلف عن الاسم المحسوب ('$device_table_name'). قد يشير هذا إلى تغيير اسم الجهاز. سيتم استخدام الجدول المسجل.");
             $device_table_name = $existing_device['table_name']; // استخدم الاسم المسجل لضمان الإدراج في الجدول الصحيح
         }
         // اختياري: تحديث اسم الجهاز إذا تغير
         // $update_name_sql = "UPDATE registered_devices SET device_name = ? WHERE device_id = ?";
         // ...
    }

} catch (Exception $e) {
    error_log("add_location.php: استثناء (Exception) أثناء تسجيل/التحقق من الجهاز '{$data['device_id']}': " . $e->getMessage());
    // لا نوقف التنفيذ هنا، قد يكون الاستثناء بسبب مشكلة غير فادحة في التسجيل الاحتياطي
}
error_log("add_location.php: الانتهاء من عملية التسجيل/التحقق للجهاز '{$data['device_id']}'. سيتم الإدراج في جدول '$device_table_name'.");
// --- نهاية التعديل ---


// --- بداية التعديل: إدراج البيانات في جدول الجهاز المحدد داخل القاعدة الرئيسية ---
error_log("add_location.php: بدء عملية إدراج البيانات في جدول '$device_table_name'...");

// لا حاجة لاتصال جديد، نستخدم $conn (الاتصال الرئيسي)
// $conn يجب أن يكون لا يزال مفتوحًا ونشطًا هنا
if (!$conn) {
     error_log("add_location.php: خطأ فادح! الاتصال الرئيسي مغلق قبل الإدراج.");
      echo json_encode(["status" => "error", "message" => "❌ خطأ داخلي: فقدان الاتصال بقاعدة البيانات"]);
      exit;
}
// التأكد من استخدام القاعدة الصحيحة (احتياطي)
if (!$conn->query("USE `$dbname`")) {
    error_log("add_location.php: فشل في استخدام قاعدة البيانات '$dbname' للإدراج: " . $conn->error);
    echo json_encode(["status" => "error", "message" => "❌ خطأ داخلي: لا يمكن تحديد قاعدة البيانات"]);
    $conn->close();
    exit;
}

// تحضير البيانات للإدراج
$latitude    = $data["latitude"];
$longitude   = $data["longitude"];
$status      = $data["status"];
$battery     = isset($data["battery"]) ? $data["battery"] : null;
date_default_timezone_set("Asia/Baghdad");
$timestamp = date("Y-m-d H:i:s");
error_log("add_location.php: البيانات جاهزة للإدراج في جدول '$device_table_name': lat=$latitude, lon=$longitude, status=$status, battery=$battery, timestamp=$timestamp");

// بناء جملة الإدخال باستخدام اسم الجدول الديناميكي (مع استخدام backticks للحماية)
// **مهم:** يجب التأكد تمامًا أن $device_table_name آمن هنا (تم إنشاؤه من خلال دالتنا)
$insert_sql = "INSERT INTO `$device_table_name` (latitude, longitude, status, battery, timestamp)
               VALUES (?, ?, ?, ?, ?)";

error_log("add_location.php: تحضير استعلام الإدراج في '$device_table_name'...");
$stmt_insert = $conn->prepare($insert_sql); // استخدام $conn (الاتصال الرئيسي)
if (!$stmt_insert) {
     error_log("add_location.php: فشل فادح في تحضير استعلام الإدراج في '$device_table_name': " . $conn->error);
     echo json_encode(["status" => "error", "message" => "⚠️ خطأ في تحضير إدخال البيانات"]);
     $conn->close();
     exit;
}

// ربط المتغيرات
error_log("add_location.php: ربط المتغيرات لاستعلام الإدراج...");
$stmt_insert->bind_param("sssss", $latitude, $longitude, $status, $battery, $timestamp);

// تنفيذ الإدخال
error_log("add_location.php: تنفيذ استعلام الإدراج في '$device_table_name'...");
if ($stmt_insert->execute()) {
    error_log("add_location.php: تم إدراج البيانات بنجاح في '$device_table_name'.");
    echo json_encode(["status" => "success", "message" => "✅ تم حفظ الموقع بنجاح في جدول الجهاز"]);
} else {
    error_log("add_location.php: خطأ فادح في إدراج البيانات في '$device_table_name': " . $stmt_insert->error);
    echo json_encode(["status" => "error", "message" => "⚠️ خطأ في حفظ البيانات: " . $stmt_insert->error]);
}

// إغلاق الاتصالات
error_log("add_location.php: إغلاق اتصال الإدراج والاتصال الرئيسي.");
$stmt_insert->close();
$conn->close();

// --- نهاية التعديل ---
