<?php
header("Content-Type: application/json; charset=UTF-8");
require 'config.php'; // يتضمن $host, $username, $password, $dbname, $conn (الاتصال الرئيسي PDO)

// --- بداية التعديل: استخدام PDO وجلب اسم الجدول ---

if (!$conn) {
    // تم التعامل مع الخطأ في config.php
    exit;
}

// استلام المعاملات من الـ GET
$device_id_filter = $_GET['device_id'] ?? null;
$date_filter = $_GET['date'] ?? null;
$time_from_filter = isset($_GET['time_from']) ? convertTo24Hour($_GET['time_from']) : null;
$time_to_filter = isset($_GET['time_to']) ? convertTo24Hour($_GET['time_to']) : null;

$all_locations = [];

// دالة لتحويل الوقت من صيغة 12 ساعة إلى 24 ساعة
function convertTo24Hour($timeStr) {
    if (!$timeStr) return null;
    $timeStr = trim($timeStr);
    $dateTime = DateTime::createFromFormat('h:i A', $timeStr);
    if ($dateTime) {
        return $dateTime->format('H:i:s');
    }
    // إذا لم يكن بصيغة AM/PM، افترض أنه 24 ساعة وأضف الثواني
    $dateTime24 = DateTime::createFromFormat('H:i', $timeStr);
    if ($dateTime24) {
        return $dateTime24->format('H:i:s');
    }
    // إذا فشل التحويل، أعد القيمة الأصلية مع إضافة ثواني افتراضية
    if (preg_match('/^\d{1,2}:\d{2}$/', $timeStr)) {
        return $timeStr . ':00';
    }
    return null; // أو أعد null إذا كان التنسيق غير صالح تمامًا
}


// دالة لتنفيذ الاستعلام على جدول جهاز معين باستخدام PDO
function fetch_device_data_from_table($pdo_conn, $device_table_name, $device_id, $device_name, $date, $time_from, $time_to) {
    // **مهم:** التأكد من أن اسم الجدول آمن
    if (!preg_match('/^device_data_[a-zA-Z0-9_]+$/', $device_table_name)) {
         error_log("get_locations.php: اسم الجدول '$device_table_name' غير صالح أو غير آمن.");
         return [];
    }

    $query = "
        SELECT
            latitude, longitude, status, battery, timestamp,
            DATE_FORMAT(`timestamp`, '%Y-%m-%d %h:%i:%s %p') AS formatted_time
        FROM
            `$device_table_name` -- استخدام backticks لاسم الجدول الديناميكي
        WHERE 1=1
    ";
    $params = [];

    if ($date) {
        $query .= " AND DATE(timestamp) = :date";
        $params[':date'] = $date;
    }
    if ($time_from) {
        $query .= " AND TIME(timestamp) >= :time_from";
        $params[':time_from'] = $time_from;
    }
    if ($time_to) {
        $query .= " AND TIME(timestamp) <= :time_to";
        $params[':time_to'] = $time_to;
    }
    $query .= " ORDER BY timestamp ASC";

    try {
        $stmt = $pdo_conn->prepare($query);
        if (!$stmt) {
             error_log("get_locations.php: خطأ في تحضير الاستعلام لجدول '$device_table_name': " . print_r($pdo_conn->errorInfo(), true));
             return [];
        }

        if (!$stmt->execute($params)) {
            error_log("get_locations.php: خطأ في تنفيذ الاستعلام لجدول '$device_table_name': " . print_r($stmt->errorInfo(), true));
            $stmt->closeCursor();
            return [];
        }

        $locations = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // إضافة معلومات الجهاز لكل صف
            $row['device_id'] = $device_id;
            $row['device_name'] = $device_name;
            $locations[] = $row;
        }

        $stmt->closeCursor();
        return $locations;

    } catch (PDOException $e) {
        error_log("PDOException في fetch_device_data_from_table لـ '$device_table_name': " . $e->getMessage());
        return [];
    }
}

// --- بداية التعديل: جلب اسم الجدول والاستعلام منه باستخدام PDO ---
try {
    // تحديد الأجهزة التي سيتم جلب بياناتها
    $devices_to_fetch = [];
    if ($device_id_filter) {
        // جلب جهاز واحد محدد (مع اسم الجدول)
        $stmt_get_device = $conn->prepare("SELECT device_id, device_name, table_name FROM registered_devices WHERE device_id = :device_id");
        if ($stmt_get_device) {
            $stmt_get_device->bindParam(':device_id', $device_id_filter, PDO::PARAM_STR);
            if ($stmt_get_device->execute()) {
                $device_info = $stmt_get_device->fetch(PDO::FETCH_ASSOC);
                if ($device_info && !empty($device_info['table_name'])) { // التأكد من وجود اسم الجدول
                    $devices_to_fetch[] = $device_info;
                } else {
                     error_log("get_locations.php: لم يتم العثور على جهاز مسجل أو اسم جدول للجهاز ID: $device_id_filter");
                }
            } else {
                 error_log("get_locations.php: خطأ في تنفيذ استعلام جلب جهاز واحد: " . print_r($stmt_get_device->errorInfo(), true));
            }
            $stmt_get_device->closeCursor();
        } else {
             error_log("get_locations.php: خطأ في تحضير استعلام جلب جهاز واحد: " . print_r($conn->errorInfo(), true));
        }
    } else {
        // جلب جميع الأجهزة المسجلة (مع أسماء جداولها)
        $stmt_all_devices = $conn->query("SELECT device_id, device_name, table_name FROM registered_devices WHERE table_name IS NOT NULL AND table_name != ''");
        if ($stmt_all_devices) {
            $devices_to_fetch = $stmt_all_devices->fetchAll(PDO::FETCH_ASSOC);
            $stmt_all_devices->closeCursor();
        } else {
             error_log("get_locations.php: خطأ في استعلام جلب جميع الأجهزة: " . print_r($conn->errorInfo(), true));
        }
    }

    // جلب البيانات لكل جهاز مطلوب
    if (!empty($devices_to_fetch)) {
        // لا حاجة لـ USE dbname مع PDO إذا تم تحديده في DSN
        foreach ($devices_to_fetch as $device) {
            error_log("get_locations.php: جلب البيانات للجهاز '{$device['device_id']}' من جدول '{$device['table_name']}'...");
            $device_locations = fetch_device_data_from_table(
                $conn, // تمرير الاتصال الرئيسي PDO
                $device['table_name'],
                $device['device_id'],
                $device['device_name'],
                $date_filter,
                $time_from_filter,
                $time_to_filter
            );
            if (!empty($device_locations)) {
                $all_locations = array_merge($all_locations, $device_locations);
            } else {
                 error_log("get_locations.php: لم يتم العثور على بيانات للجهاز '{$device['device_id']}' بالفلترة المحددة.");
            }
        }
    } else {
         error_log("get_locations.php: لا توجد أجهزة مسجلة (بأسماء جداول صالحة) لتلبية الطلب.");
    }

    // فرز النتائج المجمعة حسب الوقت (إذا تم جلب بيانات من أكثر من جهاز)
    if (!$device_id_filter && count($all_locations) > 1) {
         error_log("get_locations.php: فرز النتائج المجمعة حسب الوقت...");
        usort($all_locations, function($a, $b) {
            // افترض أن timestamp هو سلسلة نصية بتنسيق Y-m-d H:i:s أو ما شابه
            $timeA = strtotime($a['timestamp']);
            $timeB = strtotime($b['timestamp']);
            if ($timeA === false || $timeB === false) {
                // التعامل مع أخطاء التحويل إذا لزم الأمر
                return 0;
            }
            return $timeA - $timeB;
        });
    }

} catch (PDOException $e) {
    error_log("PDOException في get_locations.php (الجزء الرئيسي): " . $e->getMessage());
    echo json_encode(["error" => "حدث خطأ في قاعدة البيانات: " . $e->getMessage()], JSON_UNESCAPED_UNICODE);
    exit;
} catch (Exception $e) {
    error_log("Exception في get_locations.php (الجزء الرئيسي): " . $e->getMessage());
    echo json_encode(["error" => "حدث خطأ غير متوقع: " . $e->getMessage()], JSON_UNESCAPED_UNICODE);
    exit;
}

// --- نهاية التعديل ---

// لا حاجة لإغلاق الاتصال صراحةً مع PDO
// error_log("get_locations.php: تم إغلاق الاتصال الرئيسي.");

// إرجاع النتائج كـ JSON
error_log("get_locations.php: إرجاع " . count($all_locations) . " سجلًا.");
echo json_encode($all_locations, JSON_UNESCAPED_UNICODE);

?>
