// Loading Overlay Functions
function showLoading() {
  const overlay = document.getElementById('loadingOverlay');
  if (overlay) overlay.style.display = 'flex';
}

function hideLoading() {
  const overlay = document.getElementById('loadingOverlay');
  if (overlay) overlay.style.display = 'none';
}

// Call hideLoading when page resources are fully loaded
window.addEventListener('load', hideLoading);


//************************************************
//                    CONSTANTS
//************************************************
const accessToken = "pk.eyJ1IjoiaWJyYWhpbWNvbSIsImEiOiJja2YxZnB6OWowc2xxMnFxZGxya2ppdmZxIn0.keECYQuu1Bs0JeWDP8dSFQ";

//************************************************
//       MAP & LAYERS (يدعم 4 أنماط: تضاريس، قمر صناعي، داكن، حركة مرورية)
//************************************************
const outdoorsLayer = L.tileLayer(
  `https://api.mapbox.com/styles/v1/mapbox/outdoors-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
  {
    maxZoom: 20,
    tileSize: 512,
    zoomOffset: -1,
    attribution: 'Map data © OpenStreetMap & Mapbox'
  }
);
const satelliteLayer = L.tileLayer(
  `https://api.mapbox.com/styles/v1/mapbox/satellite-streets-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
  {
    maxZoom: 20,
    tileSize: 512,
    zoomOffset: -1,
    attribution: 'Map data © OpenStreetMap & Mapbox'
  }
);
const darkLayer = L.tileLayer(
  `https://api.mapbox.com/styles/v1/mapbox/dark-v10/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
  {
    maxZoom: 20,
    tileSize: 512,
    zoomOffset: -1,
    attribution: 'Map data © OpenStreetMap & Mapbox'
  }
);
const trafficLayer = L.tileLayer(
  `https://api.mapbox.com/styles/v1/mapbox/traffic-day-v2/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
  {
    maxZoom: 20,
    tileSize: 512,
    zoomOffset: -1,
    attribution: 'Map data © OpenStreetMap & Mapbox'
  }
);

// خريطة Leaflet
var currentLayer = outdoorsLayer; // الافتراضية
var map = L.map('map', {
  center: [30, 0],
  zoom: 2,
  layers: [currentLayer]
});

//---------------------------------------------------
// مجموعة متغيرات عامة
//---------------------------------------------------
var markers  = [];
var circles  = [];
var polylines = [];
var logs     = [];
var markerCounter = 1;
var latestMarker  = null;
var latestLocationTimer = null;
var lastTimestamp = null;

// للأنيميشن
var animationMarker = null;
var animationTimeout = null;
var currentAnimationIndex = 0;
var routeCoordinates = [];
var analyticsGroups  = [];

// أيقونات
var parkIcon = L.icon({
  iconUrl: 'https://almanara.space/gps_tracker/uploads/image.png',
  iconSize: [32, 32],
  iconAnchor: [16, 32],
  popupAnchor: [0, -32]
});
var hiddenIcon = L.icon({
  iconUrl: 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=',
  iconSize: [1, 1],
  iconAnchor: [0, 0],
  popupAnchor: [0, 0]
});
var startIcon = L.icon({
  iconUrl: 'https://almanara.space/gps_tracker/uploads/start.png',
  iconSize: [60, 60],
  iconAnchor: [16, 32]
});
var endIcon = L.icon({
  iconUrl: 'https://almanara.space/gps_tracker/uploads/end.png',
  iconSize: [60, 60],
  iconAnchor: [16, 32]
});
var latestIcon = L.icon({
  iconUrl: 'https://cdn-icons-png.flaticon.com/512/684/684908.png',
  iconSize: [32, 32],
  iconAnchor: [16, 32]
});

//*********************************************
//   دوال UI عامة (Toast رسائل)
//*********************************************
function showToast(message) {
  var container = document.getElementById("toastContainer");
  if (!container) return;
  var toast = document.createElement("div");
  toast.className = "toast";
  toast.textContent = message;
  container.appendChild(toast);
  setTimeout(() => {
    if (container.contains(toast)) {
      toast.style.animation = 'fadeout 0.5s forwards'; // Ensure fadeout completes
      setTimeout(()=> { if (container.contains(toast)) container.removeChild(toast); }, 500);
    }
  }, 3500);
}

//----------------------------------------------------
//   تغيير طبقة الخريطة (Map Style)
//----------------------------------------------------
function changeMapStyle() {
  var styleSelect = document.getElementById("mapStyleSelect");
  var styleValue  = styleSelect.value;
  // حفظ التفضيل في localStorage
  localStorage.setItem('preferredMapStyle', styleValue);

  map.removeLayer(currentLayer);
  if (styleValue === "satellite") {
    currentLayer = satelliteLayer;
  } else if (styleValue === "dark") {
    currentLayer = darkLayer;
  } else if (styleValue === "traffic") {
    currentLayer = trafficLayer;
  } else {
    currentLayer = outdoorsLayer;
  }
  map.addLayer(currentLayer);
}

// استعادة تفضيلات المستخدم من localStorage عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
  var savedStyle = localStorage.getItem('preferredMapStyle');
  const mapStyleSelect = document.getElementById("mapStyleSelect");
  if (savedStyle && mapStyleSelect) {
    mapStyleSelect.value = savedStyle;
    changeMapStyle(); // Apply the saved style
  }
});


//----------------------------------------------------
// دوال تنظيف الخريطة والبيانات
//----------------------------------------------------
function clearMap() {
  markers.forEach(m => map.removeLayer(m));
  markers = [];
  circles.forEach(c => map.removeLayer(c));
  circles = [];
  polylines.forEach(p => map.removeLayer(p));
  polylines = [];
  if (latestMarker) {
    map.removeLayer(latestMarker);
    latestMarker = null;
  }
  if (animationMarker) {
    map.removeLayer(animationMarker);
    animationMarker = null;
  }
  if (latestLocationTimer) {
    clearInterval(latestLocationTimer);
    latestLocationTimer = null;
  }
  if (animationTimeout) {
    clearTimeout(animationTimeout);
    animationTimeout = null;
  }
  logs = [];
  markerCounter = 1;
  routeCoordinates = [];
  analyticsGroups  = [];
  lastTimestamp    = null;

  updateLogsTable();
  updateAnalytics();
  hideLoading(); // Hide loading after clearing
}

function resetFilters() {
  const deviceSelect = document.getElementById("deviceSelect");
  const dateSelect = document.getElementById("dateSelect");
  const timeFrom = document.getElementById("timeFrom");
  const timeTo = document.getElementById("timeTo");
  const selectedDeviceName = document.getElementById("selectedDeviceName");

  if(deviceSelect) deviceSelect.value = "";
  if(dateSelect) dateSelect.value   = "";
  if(timeFrom) timeFrom.value     = "";
  if(timeTo) timeTo.value       = "";
  if(selectedDeviceName) selectedDeviceName.textContent = "لا يوجد جهاز محدد";

  clearMap();
  showToast("تمت إعادة تعيين الفلاتر");
}

//----------------------------------------------------
// دوال جلب البيانات
//----------------------------------------------------
function fetchLocations() {
  showLoading();
  clearMap();
  var deviceSel = document.getElementById("deviceSelect");
  if (!deviceSel) { hideLoading(); return; }

  var deviceId  = deviceSel.value;
  var deviceName = (deviceSel.selectedIndex >= 0 && deviceSel.options[deviceSel.selectedIndex])
                   ? deviceSel.options[deviceSel.selectedIndex].text
                   : "غير محدد";

  const selectedDeviceNameEl = document.getElementById("selectedDeviceName");
  if (selectedDeviceNameEl) selectedDeviceNameEl.textContent = deviceName;


  var date     = document.getElementById("dateSelect") ? document.getElementById("dateSelect").value : "";
  var timeFromEl = document.getElementById("timeFrom");
  var timeToEl = document.getElementById("timeTo");

  var timeFrom = timeFromEl ? convertTo24Hour(timeFromEl.value.trim()) : "";
  var timeTo   = timeToEl ? convertTo24Hour(timeToEl.value.trim()) : "";

  var url = "get_locations.php?";
  if (deviceId) url += "device_id=" + encodeURIComponent(deviceId) + "&";
  if (date)     url += "date=" + encodeURIComponent(date) + "&";
  if (timeFrom) url += "time_from=" + encodeURIComponent(timeFrom) + "&";
  if (timeTo)   url += "time_to=" + encodeURIComponent(timeTo) + "&";
  url = url.slice(0, -1); // Remove last '&' or '?'

  fetch(url)
    .then(res => {
        if (!res.ok) {
            throw new Error(`HTTP error! status: ${res.status}`);
        }
        return res.json();
    })
    .then(data => {
      hideLoading();
      if (Array.isArray(data) && data.length > 0) {
        showToast("تم جلب " + data.length + " نقطة");

        // ترتيب زمني تصاعدي
        let sortedAsc = data.slice().sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
        let startLoc = sortedAsc[0];
        let endLoc   = sortedAsc[sortedAsc.length - 1];

        // أيقونة البداية
        if (startLoc) {
            let startLat = parseFloat(startLoc.latitude);
            let startLng = parseFloat(startLoc.longitude);
            if (!isNaN(startLat) && !isNaN(startLng)) {
              getPlaceName(startLat, startLng).then(place => {
                let markerStart = L.marker([startLat, startLng], { icon: startIcon })
                  .addTo(map)
                  .bindPopup(`
                    <b>بداية المسار</b><br>
                    ${formatTo12Hour(startLoc.timestamp)}<br>
                    بطارية: ${startLoc.battery ? startLoc.battery + '%' : 'غير معروفة'}<br>
                    المكان: ${place}<br>
                    <a href="https://www.google.com/maps/dir/?api=1&destination=${startLat},${startLng}" target="_blank">خرائط جوجل</a> |
                    <a href="https://waze.com/ul?ll=${startLat},${lng}&navigate=yes" target="_blank">Waze</a>
                  `);
                markers.push(markerStart);
              });
            }
        }


        // أيقونة النهاية
        if (endLoc) {
            let endLat = parseFloat(endLoc.latitude);
            let endLng = parseFloat(endLoc.longitude);
            if (!isNaN(endLat) && !isNaN(endLng)) {
              getPlaceName(endLat, endLng).then(place => {
                let markerEnd = L.marker([endLat, endLng], { icon: endIcon })
                  .addTo(map)
                  .bindPopup(`
                    <b>نهاية المسار</b><br>
                    ${formatTo12Hour(endLoc.timestamp)}<br>
                    بطارية: ${endLoc.battery ? endLoc.battery + '%' : 'غير معروفة'}<br>
                    المكان: ${place}<br>
                    <a href="https://www.google.com/maps/dir/?api=1&destination=${endLat},${endLng}" target="_blank">خرائط جوجل</a> |
                    <a href="https://waze.com/ul?ll=${endLat},${endLng}&navigate=yes" target="_blank">Waze</a>
                  `);
                markers.push(markerEnd);
              });
            }
        }


        // تجميع النقاط (بارك/تحرك)
        let grouped = groupLocations(data, 30); // نصف قطر 30م
        grouped.sort((a, b) => new Date(a.entryTime) - new Date(b.entryTime));
        analyticsGroups = grouped;

        var routePoints = [];
        grouped.forEach(g => {
          var lat = g.center[0];
          var lng = g.center[1];
          if (!isNaN(lat) && !isNaN(lng)) {
            var batteryInfo = "";
            if (g.minBattery !== null && g.maxBattery !== null) {
              batteryInfo = (g.minBattery === g.maxBattery)
                ? `بطارية: ${g.minBattery}%<br>`
                : `بطارية: ${g.minBattery}% → ${g.maxBattery}%<br>`;
            }
            var icon = (g.status === "بارك توقف") ? parkIcon : hiddenIcon;
            var mk = L.marker([lat, lng], { icon: icon })
              .addTo(map)
              .bindPopup(`
                <b>النقطة ${markerCounter}</b><br>
                وقت الدخول: ${formatTo12Hour(g.entryTime)}<br>
                وقت الخروج: ${formatTo12Hour(g.exitTime)}<br>
                ${batteryInfo}
                الحالة: ${g.status}<br>
                <a href="https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}" target="_blank">خرائط جوجل</a> |
                <a href="https://waze.com/ul?ll=${lat},${lng}&navigate=yes" target="_blank">Waze</a>
              `);
            markers.push(mk);
            markerCounter++;

            // للجدول
            getPlaceName(lat, lng).then(place => {
              logs.push({
                placeName: place,
                entryTime: g.entryTime,
                exitTime:  g.exitTime,
                center:    [lat, lng],
                status:    g.status,
                batteryRange: (g.minBattery !== null && g.maxBattery !== null)
                             ? (g.minBattery === g.maxBattery ? g.minBattery + '%' : g.minBattery + '% - ' + g.maxBattery + '%')
                             : ''
              });
              updateLogsTable();
            });

            routePoints.push([lat, lng]);
            routeCoordinates.push({
              lat: lat,
              lng: lng,
              time: g.entryTime,
              minBattery: g.minBattery,
              maxBattery: g.maxBattery
            });
          }
        });

        // رسم المسار
        if (routePoints.length > 1) {
          var polyline = L.polyline(routePoints, { color: 'blue', weight: 5 }).addTo(map);
          polylines.push(polyline);
        }
        if (routePoints.length > 0) {
          var bounds = new L.LatLngBounds(routePoints);
          map.fitBounds(bounds);
        }

        updateAnalytics();
      } else {
        showToast("لا توجد بيانات ضمن الشروط");
      }
    })
    .catch(err => {
        console.error("Fetch error: ", err);
        showToast("⚠️ خطأ في جلب البيانات: " + err.message);
        hideLoading();
    });
}

// جلب آخر موقع مباشر
function fetchLatestLocation() {
  showLoading();
  var deviceSel = document.getElementById("deviceSelect");
  if (!deviceSel) { hideLoading(); return; }
  var deviceId  = deviceSel.value;

  if (!deviceId) {
    showToast("⚠️ يرجى اختيار جهاز لجلب أحدث موقع");
    hideLoading();
    return;
  }

  if (latestLocationTimer) {
    clearInterval(latestLocationTimer);
  }

  function getLatest() {
    fetch(`get_latest_location.php?device_id=${deviceId}&t=${Date.now()}`)
      .then(res => {
        if (!res.ok) throw new Error(`HTTP error! status: ${res.status}`);
        return res.json();
      })
      .then(data => {
        hideLoading();
        if (data.status === "success" && data.data) {
          var lat       = parseFloat(data.data.latitude);
          var lng       = parseFloat(data.data.longitude);
          var timestamp = data.data.timestamp;
          var battery   = data.data.battery;
          var deviceName = data.data.device_name || "الجهاز المحدد";

          const selectedDeviceNameEl = document.getElementById("selectedDeviceName");
          if (selectedDeviceNameEl) selectedDeviceNameEl.textContent = deviceName;


          if (!isNaN(lat) && !isNaN(lng)) {
            if (latestMarker) {
              map.removeLayer(latestMarker);
            }
            latestMarker = L.marker([lat, lng], { icon: latestIcon })
              .addTo(map)
              .bindPopup(`
                <b>📡 آخر موقع</b><br>
                الجهاز: ${htmlspecialchars(deviceName)}<br>
                الوقت: ${formatTo12Hour(timestamp)}<br>
                البطارية: ${battery ? battery + '%' : 'غير معروفة'}<br>
                <a href="https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}" target="_blank">خرائط جوجل</a> |
                <a href="https://waze.com/ul?ll=${lat},${lng}&navigate=yes" target="_blank">Waze</a>
              `).openPopup();

            // Clear previous circles before adding a new one
            circles.forEach(c => map.removeLayer(c));
            circles = [];

            var circle = L.circle([lat, lng], {
              radius: 30,
              color: 'blue',
              fillColor: '#00f',
              fillOpacity: 0.5
            }).addTo(map);
            circles.push(circle); // Store the new circle
            map.setView([lat, lng], 15);

            lastTimestamp = timestamp;
          } else {
             showToast("⚠️ بيانات الموقع غير صالحة.");
          }
        } else {
            showToast(data.message || "⚠️ لم يتم العثور على موقع حديث.");
        }
      })
      .catch(err => {
        console.error("خطأ في جلب الموقع:", err);
        showToast("⚠️ خطأ: " + err.message);
        hideLoading();
      });
  }

  getLatest();
  latestLocationTimer = setInterval(getLatest, 10000);
}

//----------------------------------------------------
//   دوال الحركة (أنيميشن)
//----------------------------------------------------
function playAnimation() {
  if (!routeCoordinates.length) {
    showToast("❌ لا يوجد مسار لتشغيل الحركة.");
    return;
  }
  if (animationTimeout) {
    clearTimeout(animationTimeout);
    animationTimeout = null;
  }
  if (currentAnimationIndex >= routeCoordinates.length) {
    currentAnimationIndex = 0; // Restart if at end
  }
  animateStep();
}

function stopAnimation() {
  if (animationTimeout) {
    clearTimeout(animationTimeout);
    animationTimeout = null;
  }
}

function animateStep() {
  if (currentAnimationIndex >= routeCoordinates.length) {
    stopAnimation(); // Stop if past the end
    return;
  }
  var pt = routeCoordinates[currentAnimationIndex];
  var minB = pt.minBattery;
  var maxB = pt.maxBattery;
  var batteryText = "";
  if (minB !== null && maxB !== null) {
    batteryText = (minB === maxB) ? (minB + "%") : (minB + "% → " + maxB + "%");
  }

  if (!animationMarker) {
    animationMarker = L.marker([pt.lat, pt.lng], { icon: createAnimationIcon(pt.time, batteryText) })
      .addTo(map);
  } else {
    animationMarker.setLatLng([pt.lat, pt.lng]);
    animationMarker.setIcon(createAnimationIcon(pt.time, batteryText));
  }
  map.panTo([pt.lat, pt.lng]);
  currentAnimationIndex++;

  var speedEl = document.getElementById("animationSpeed");
  var speed = speedEl ? parseInt(speedEl.value) : 1000;
  if (isNaN(speed) || speed < 100) speed = 1000;

  if (currentAnimationIndex < routeCoordinates.length) {
    animationTimeout = setTimeout(animateStep, speed);
  } else {
      showToast("🏁 انتهت الحركة");
  }
}

function resetAnimationToStart() {
  stopAnimation();
  currentAnimationIndex = 0;
  if (animationMarker) {
    map.removeLayer(animationMarker);
    animationMarker = null;
  }
  if (routeCoordinates.length > 0) {
    var pt = routeCoordinates[0];
    var batteryText = "";
    if (pt.minBattery !== null && pt.maxBattery !== null) {
        batteryText = (pt.minBattery === pt.maxBattery) ? (pt.minBattery + "%") : (pt.minBattery + "% → " + pt.maxBattery + "%");
    }
    animationMarker = L.marker([pt.lat, pt.lng], { icon: createAnimationIcon(pt.time, batteryText) })
      .addTo(map);
    map.panTo([pt.lat, pt.lng]);
    showToast("تم العودة إلى بداية الحركة");
  } else {
    showToast("لا يوجد مسار للعودة إلى بدايته.");
  }
}


function createAnimationIcon(time, battery) {
  var batteryStr = battery ? `<br>بطارية: ${battery}` : "";
  return L.divIcon({
    html: `<div style="position: relative; background: rgba(0,0,0,0.7);
                     padding: 4px 8px; border-radius: 6px; border: 2px solid #fff;
                     display: inline-flex; align-items: center; white-space: nowrap;">
             <img src="https://cdn-icons-png.flaticon.com/512/684/684908.png"
                  style="width:40px;height:40px; flex-shrink: 0;">
             <span style="margin-left: 8px; color: #fff; font-size: 14px;">
               ${formatTo12Hour(time)}${batteryStr}
             </span>
           </div>`,
    iconSize: 'auto', // Let the size be determined by content
    className: '' // Important to prevent default Leaflet divIcon styling
  });
}

//----------------------------------------------------
//    دوال تحليل البيانات
//----------------------------------------------------
let timeChartInstance = null;
let speedChartInstance = null;

function updateAnalytics() {
  var div = document.getElementById("analytics");
  if (!div) return;

  // Clear previous charts if they exist
  if (timeChartInstance) {
    timeChartInstance.destroy();
    timeChartInstance = null;
  }
  if (speedChartInstance) {
    speedChartInstance.destroy();
    speedChartInstance = null;
  }


  // حساب المسافة الإجمالية
  var totalDistance = 0;
  for (var i = 1; i < routeCoordinates.length; i++) {
    totalDistance += getDistance(
      routeCoordinates[i-1].lat, routeCoordinates[i-1].lng,
      routeCoordinates[i].lat,   routeCoordinates[i].lng
    );
  }
  totalDistance /= 1000; // كم
  var movingTime = 0, stoppedTime = 0;
  analyticsGroups.forEach(g => {
    var t1 = new Date(g.entryTime).getTime();
    var t2 = new Date(g.exitTime).getTime();
    var duration = (t2 - t1) / 1000; // seconds
    if (g.status === "تحرك") {
      movingTime += duration;
    } else {
      stoppedTime += duration;
    }
  });
  var movingMins  = movingTime / 60;
  var stoppedMins = stoppedTime / 60;
  var avgSpeed    = (movingTime > 0) // movingTime is in seconds
                    ? totalDistance / (movingTime / 3600) // convert movingTime to hours for km/h
                    : 0;

  div.innerHTML = `
    <h2 id="analyticsTitle">تحليل البيانات</h2>
    <p id="pDistance">المسافة المقطوعة: ${totalDistance.toFixed(2)} كم</p>
    <p id="pMovingTime">وقت التحرك: ${movingMins.toFixed(2)} دقيقة</p>
    <p id="pStoppedTime">وقت التوقف: ${stoppedMins.toFixed(2)} دقيقة</p>
    <p id="pAvgSpeed">متوسط السرعة: ${avgSpeed.toFixed(2)} كم/س (أثناء التحرك)</p>
    <div class="chart-container"><canvas id="timeChart"></canvas></div>
    <div class="chart-container"><canvas id="speedChart"></canvas></div>
  `;
  // Re-apply language if needed after innerHTML change
  if (currentLang === "en") applyEnglishAnalytics();


  // مخطط Doughnut لوقت التحرك والتوقف
  var ctx1 = document.getElementById("timeChart");
  if (ctx1) {
    timeChartInstance = new Chart(ctx1.getContext("2d"), {
        type: 'doughnut',
        data: {
        labels: [
            currentLang === 'ar' ? 'تحرك (دقيقة)' : 'Moving (min)',
            currentLang === 'ar' ? 'توقف (دقيقة)' : 'Stopped (min)'
        ],
        datasets: [{
            data: [movingMins.toFixed(2), stoppedMins.toFixed(2)],
            backgroundColor: ['#36A2EB', '#FF6384'],
            hoverBackgroundColor: ['#36A2EB', '#FF6384']
        }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'bottom', labels: { color: getChartTextColor() } },
                tooltip: { titleColor: getChartTextColor(), bodyColor: getChartTextColor() }
            }
        }
    });
  }


  // مخطط السرعة
  var speeds = [];
  var speedLabels = [];
  for (var i = 1; i < routeCoordinates.length; i++) {
    var d = getDistance(
      routeCoordinates[i-1].lat, routeCoordinates[i-1].lng,
      routeCoordinates[i].lat,   routeCoordinates[i].lng
    ); // distance in meters
    var tDiff = (new Date(routeCoordinates[i].time).getTime() -
                 new Date(routeCoordinates[i-1].time).getTime()) / 3600000; // time diff in hours
    var sp = (tDiff > 0) ? (d/1000)/tDiff : 0; // speed in km/h
    speeds.push(sp.toFixed(2));
    speedLabels.push((currentLang === 'ar' ? "نقطة " : "Point ") + i);
  }
  var ctx2 = document.getElementById("speedChart");
  if (ctx2) {
    speedChartInstance = new Chart(ctx2.getContext("2d"), {
        type: 'line',
        data: {
        labels: speedLabels,
        datasets: [{
            label: currentLang === 'ar' ? 'السرعة (كم/س)' : 'Speed (km/h)',
            data: speeds,
            borderColor: '#4BC0C0',
            backgroundColor: 'rgba(75, 192, 192, 0.2)',
            fill: true,
            tension: 0.1
        }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'bottom', labels: { color: getChartTextColor() } },
                tooltip: { titleColor: getChartTextColor(), bodyColor: getChartTextColor() }
            },
            scales: {
                y: { ticks: { color: getChartTextColor() }, grid: { color: getChartGridColor() } },
                x: { ticks: { color: getChartTextColor() }, grid: { color: getChartGridColor() } }
            }
        }
    });
  }
}

function getChartTextColor() {
    return document.documentElement.getAttribute('data-theme') === 'dark' ? '#e0e0e0' : '#2d3436';
}
function getChartGridColor() {
    return document.documentElement.getAttribute('data-theme') === 'dark' ? 'rgba(224, 224, 224, 0.1)' : 'rgba(45, 52, 54, 0.1)';
}


//----------------------------------------------------
//  دوال مشاركة الموقع
//----------------------------------------------------
function shareLocation() {
  var deviceSel = document.getElementById("deviceSelect");
  if (!deviceSel) return;
  var deviceId  = deviceSel.value;

  if (!deviceId) {
    showToast("❌ اختر جهازاً أولاً.");
    return;
  }
  if (!latestMarker) {
    showToast("❌ لا يوجد موقع حديث للمشاركة. اضغط (آخر موقع مباشر).");
    return;
  }
  var deviceName = (deviceSel.selectedIndex >= 0 && deviceSel.options[deviceSel.selectedIndex])
                   ? deviceSel.options[deviceSel.selectedIndex].text
                   : "جهاز غير محدد";
  var expires = Date.now() + 3600000; // ساعة واحدة
  var shareUrl = window.location.origin + window.location.pathname
               + "?share=1"
               + "&device_id="   + encodeURIComponent(deviceId)
               + "&device_name=" + encodeURIComponent(deviceName)
               + "&expires="     + expires;
  window.open(shareUrl, "_blank");
  showToast("تم فتح رابط المشاركة في نافذة جديدة.");
}

//----------------------------------------------------
//   دوال جدول السجلات (التوقف)
//----------------------------------------------------
function updateLogsTable() {
  var tbody = document.getElementById("logsTable")?.getElementsByTagName("tbody")[0];
  if (!tbody) return;
  tbody.innerHTML = "";

  // فقط توقفات >= 5 دقائق
  var filtered = logs.filter(log => {
    var diff = (new Date(log.exitTime).getTime() - new Date(log.entryTime).getTime()) / 60000; // minutes
    return diff >= 5;
  });
  // ترتيب تصاعدي زمني
  filtered.sort((a, b) => new Date(a.entryTime).getTime() - new Date(b.entryTime).getTime());

  filtered.forEach(log => {
    var row = tbody.insertRow();
    row.insertCell(0).innerHTML = `<a href="#" onclick="event.preventDefault(); focusOnLocation([${log.center}])">${htmlspecialchars(log.placeName)}</a>`;
    row.insertCell(1).textContent = formatTo12Hour(log.entryTime);
    row.insertCell(2).textContent = formatTo12Hour(log.exitTime);
    row.insertCell(3).textContent = htmlspecialchars(log.status);
    row.insertCell(4).textContent = htmlspecialchars(log.batteryRange);
  });
}

// البحث في سجل التوقف
function filterLogsTable() {
  var inputEl = document.getElementById("logsSearchInput");
  if(!inputEl) return;
  var input = inputEl.value.toUpperCase();
  var table = document.getElementById("logsTable");
  if(!table) return;
  var trs   = table.getElementsByTagName("tr");

  for (var i = 1; i < trs.length; i++) { // Start from 1 to skip header row
    var tds = trs[i].getElementsByTagName("td");
    var rowText = "";
    var displayRow = false;
    for (var j = 0; j < tds.length; j++) {
      if (tds[j]) {
        rowText = tds[j].textContent || tds[j].innerText;
        if (rowText.toUpperCase().indexOf(input) > -1) {
          displayRow = true;
          break;
        }
      }
    }
    trs[i].style.display = displayRow ? "" : "none";
  }
}

function resetLogsFilter() {
  const logsSearchInput = document.getElementById("logsSearchInput");
  if (logsSearchInput) {
      logsSearchInput.value = "";
  }
  filterLogsTable();
}

// فرز الجدول بالعمود index
let logsSortDirections = {}; // Object to store sort direction for each column

function sortLogsTable(columnIndex) {
    var table = document.getElementById("logsTable");
    if (!table) return;
    var tbody = table.tBodies[0];
    if (!tbody) return;
    var rows = Array.from(tbody.querySelectorAll("tr"));
    var ths = table.tHead.querySelectorAll("th.sortable");

    // Determine sort direction (toggle or default to ascending)
    const currentDirection = logsSortDirections[columnIndex] || 'asc';
    const newDirection = currentDirection === 'asc' ? 'desc' : 'asc';
    logsSortDirections[columnIndex] = newDirection;

    // Reset other column indicators
    ths.forEach((th, idx) => {
        th.classList.remove('sorted-asc', 'sorted-desc');
        if (idx === columnIndex) {
            th.classList.add(newDirection === 'asc' ? 'sorted-asc' : 'sorted-desc');
        }
    });


    rows.sort((rowA, rowB) => {
        var cellA = rowA.cells[columnIndex].textContent.trim();
        var cellB = rowB.cells[columnIndex].textContent.trim();

        let comparison = 0;
        if (columnIndex === 1 || columnIndex === 2) { // Timestamp columns
            // Attempt to parse dates, fallback to string comparison if parsing fails
            let dateA = new Date(cellA.replace(/(\d{2})\/(\d{2})\/(\d{4})/, '$2/$1/$3')); // Handle dd/mm/yyyy
            let dateB = new Date(cellB.replace(/(\d{2})\/(\d{2})\/(\d{4})/, '$2/$1/$3'));
            if (!isNaN(dateA) && !isNaN(dateB)) {
                 comparison = dateA - dateB;
            } else {
                 comparison = cellA.localeCompare(cellB); // Fallback
            }
        } else { // Text columns
            comparison = cellA.localeCompare(cellB, undefined, {numeric: true, sensitivity: 'base'});
        }
        return newDirection === 'asc' ? comparison : -comparison;
    });

    rows.forEach(row => tbody.appendChild(row));
}


function focusOnLocation(center) {
  if (map && center && center.length === 2 && !isNaN(center[0]) && !isNaN(center[1])) {
      map.setView(center, 15);
  } else {
      console.error("Invalid center coordinates for focusOnLocation:", center);
  }
}

//----------------------------------------------------
//   دوال تجميع المواقع (بارك/تحرك)
//----------------------------------------------------
function groupLocations(locations, radius) {
  if (!locations || !locations.length) return [];
  locations.sort((a, b) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime());

  var groups = [];
  var first  = locations[0];
  if (!first) return [];

  var currentGroup = {
    points: [first],
    center: [parseFloat(first.latitude), parseFloat(first.longitude)],
    entryTime: first.timestamp,
    exitTime:  first.timestamp,
    batteryLevels: [ first.battery ? parseInt(first.battery) : null ]
  };
  var prevTime = new Date(first.timestamp);

  for (var i = 1; i < locations.length; i++) {
    var loc = locations[i];
    var currTime = new Date(loc.timestamp);
    var timeDiff = (currTime.getTime() - prevTime.getTime()) / 60000; // دقائق
    var lastPt   = currentGroup.points[currentGroup.points.length - 1];
    var lastLat  = parseFloat(lastPt.latitude);
    var lastLng  = parseFloat(lastPt.longitude);
    var currLat  = parseFloat(loc.latitude);
    var currLng  = parseFloat(loc.longitude);

    if (isNaN(lastLat) || isNaN(lastLng) || isNaN(currLat) || isNaN(currLng)) continue; // Skip if invalid coords

    var dist     = getDistance(lastLat, lastLng, currLat, currLng); // meters
    var locBattery = loc.battery ? parseInt(loc.battery) : null;

    if (timeDiff <= 5 && dist <= radius) { // Within 5 mins and radius
      currentGroup.points.push(loc);
      currentGroup.exitTime = loc.timestamp;
      if(locBattery !== null) currentGroup.batteryLevels.push(locBattery);
    } else {
      groups.push(finalizeGroup(currentGroup));
      currentGroup = {
        points: [loc],
        center: [currLat, currLng],
        entryTime: loc.timestamp,
        exitTime: loc.timestamp,
        batteryLevels: [locBattery !== null ? locBattery : null]
      };
    }
    prevTime = currTime;
  }
  groups.push(finalizeGroup(currentGroup)); // Push the last group

  return groups;
}

function finalizeGroup(group) {
    // Calculate average center for more accuracy if needed, or just use first point's center
    // For simplicity, we are using the first point's center of a new group.

    var diffMin = (new Date(group.exitTime).getTime() - new Date(group.entryTime).getTime()) / 60000;
    group.status = (diffMin >= 5) ? "بارك توقف" : "تحرك";

    var validB = group.batteryLevels.filter(b => b !== null && !isNaN(b));
    group.minBattery = validB.length > 0 ? Math.min(...validB) : null;
    group.maxBattery = validB.length > 0 ? Math.max(...validB) : null;
    delete group.points; // We don't need to store all points after processing
    return group;
}


//----------------------------------------------------
//   دوال مساعدة
//----------------------------------------------------
function getDistance(lat1, lon1, lat2, lon2) {
  var R = 6371e3; // metres
  var φ1 = lat1 * Math.PI / 180; // φ, λ in radians
  var φ2 = lat2 * Math.PI / 180;
  var Δφ = (lat2 - lat1) * Math.PI / 180;
  var Δλ = (lon2 - lon1) * Math.PI / 180;
  var a = Math.sin(Δφ/2)*Math.sin(Δφ/2) + Math.cos(φ1)*Math.cos(φ2)*Math.sin(Δλ/2)*Math.sin(Δλ/2);
  var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
  return R * c; // in metres
}

function convertTo24Hour(timeStr) {
  if (!timeStr) return null;
  timeStr = timeStr.trim();

  const regex = /^(\d{1,2}):(\d{1,2})(?:\s*([AP]M))?$/i;
  const match = timeStr.match(regex);

  if (!match) {
    console.warn("Invalid time format for conversion:", timeStr);
    return null;
  }

  let hours = parseInt(match[1], 10);
  let minutes = parseInt(match[2], 10);
  let ampm = match[3] ? match[3].toUpperCase() : null;

  if (minutes < 0 || minutes > 59) {
    console.warn("Invalid minutes in time:", timeStr);
    return null;
  }

  if (ampm) {
    if (hours < 1 || hours > 12) {
        console.warn("Invalid hours for AM/PM time:", timeStr);
        return null;
    }
    if (ampm === 'PM' && hours < 12) {
      hours += 12;
    } else if (ampm === 'AM' && hours === 12) { // 12 AM is 00 hours
      hours = 0;
    }
  } else {
    if (hours < 0 || hours > 23) {
        console.warn("Invalid hours for 24-hour time:", timeStr);
        return null;
    }
  }

  const hh = (hours < 10) ? '0' + String(hours) : String(hours);
  const mm = (minutes < 10) ? '0' + String(minutes) : String(minutes);
  return hh + ":" + mm + ":00"; // Return with seconds for consistency if DB expects it
}

function formatTo12Hour(dateStr) {
    if (!dateStr) return '';
    try {
        // Assuming dateStr is in "YYYY-MM-DD HH:MM:SS" or similar SQL format
        const dateObj = new Date(dateStr.replace(' ', 'T')); // More robust parsing
        if (isNaN(dateObj)) return dateStr; // If parsing fails, return original

        let hours = dateObj.getHours();
        const minutes = dateObj.getMinutes();
        const ampm = hours >= 12 ? 'PM' : 'AM';
        hours = hours % 12;
        hours = hours ? hours : 12; // the hour '0' should be '12'
        const minutesStr = minutes < 10 ? '0' + minutes : minutes;

        const day = dateObj.getDate() < 10 ? '0' + dateObj.getDate() : dateObj.getDate();
        const month = (dateObj.getMonth() + 1) < 10 ? '0' + (dateObj.getMonth() + 1) : (dateObj.getMonth() + 1);
        const year = dateObj.getFullYear();

        return `${day}/${month}/${year} ${hours}:${minutesStr} ${ampm}`;
    } catch (e) {
        console.error("Error formatting date:", dateStr, e);
        return dateStr; // Fallback to original string on error
    }
}


function getPlaceName(lat, lng) {
  var url = `https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${accessToken}&language=ar`; // Added language parameter
  return fetch(url)
    .then(res => {
        if (!res.ok) throw new Error(`Geocoding error! status: ${res.status}`);
        return res.json();
    })
    .then(data => {
      if (data.features && data.features.length > 0) {
        return data.features[0].place_name_ar || data.features[0].place_name; // Prefer Arabic name
      }
      return "مكان غير معروف";
    })
    .catch(err => {
        console.error("Geocoding fetch error:", err);
        return "خطأ في تحديد المكان";
    });
}

// Helper to escape HTML for safe insertion
function htmlspecialchars(str) {
    if (typeof str !== 'string') return '';
    return str.replace(/[&<>"']/g, function (match) {
        return {
            '&': '&',
            '<': '<',
            '>': '>',
            '"': '"',
            "'": '''
        }[match];
    });
}


//----------------------------------------------------
//    تنزيل سجلات التوقف بصيغة XLSX
//----------------------------------------------------
function downloadLogsXLSX() {
  var filtered = logs.filter(log => {
    var diff = (new Date(log.exitTime).getTime() - new Date(log.entryTime).getTime()) / 60000;
    return diff >= 5;
  });

  if(filtered.length === 0){
    showToast("لا توجد سجلات للتنزيل.");
    return;
  }

  var ws_data = [
    ["اسم المكان", "وقت الدخول", "وقت الخروج", "الحالة", "البطارية"]
  ];
  filtered.forEach(l => {
    ws_data.push([
      l.placeName,
      formatTo12Hour(l.entryTime), // Use consistent formatting
      formatTo12Hour(l.exitTime),  // Use consistent formatting
      l.status,
      l.batteryRange
    ]);
  });

  var wb = XLSX.utils.book_new();
  var ws = XLSX.utils.aoa_to_sheet(ws_data);
  ws['!cols'] = [ // Column widths
    { wch: 40 }, { wch: 25 }, { wch: 25 }, { wch: 15 }, { wch: 20 }
  ];
  XLSX.utils.book_append_sheet(wb, ws, "سجلات التوقف");
  XLSX.writeFile(wb, "stop_logs.xlsx");
  showToast("تم تنزيل ملف Excel بنجاح");
}

//----------------------------------------------------
//   تصفية الأجهزة + المجموعات في القائمة الجانبية
//----------------------------------------------------
function filterDevices() {
  var inputEl = document.getElementById("searchDevice");
  if (!inputEl) return;
  var input = inputEl.value.toUpperCase();
  var ul    = document.getElementById("deviceList");
  if (!ul) return;
  var lis   = ul.querySelectorAll("li[data-device-id], li.group-header"); // Target specific list items

  for (var i = 0; i < lis.length; i++) {
    var li = lis[i];
    // For group headers, we check the header text.
    // For device items, we check their text content.
    var txt = (li.textContent || li.innerText).toUpperCase();
    if (txt.indexOf(input) > -1) {
        li.style.display = "";
        // If it's a group header and it matches, ensure its child UL is also potentially visible
        // (though individual items within it will be filtered by their own logic if they were li[data-device-id])
        if (li.classList.contains('group-header')) {
            const groupId = li.getAttribute('onclick')?.match(/toggleGroupDevices\((\d+)\)/)?.[1];
            if (groupId) {
                const groupDevicesUl = document.getElementById(`group-devices-${groupId}`);
                if (groupDevicesUl) {
                    // Show the group UL if its header matches, items inside will be filtered if they are also targets
                    // This part is tricky because children filtering happens independently if they are also `li`
                }
            }
        }
    } else {
        li.style.display = "none";
    }
  }
}


// عند الضغط على جهاز في القائمة (الأجهزة غير المضافة لأي مجموعة)
document.addEventListener('DOMContentLoaded', function() {
    const deviceList = document.getElementById("deviceList");
    if (deviceList) {
        deviceList.addEventListener("click", function(e) {
          let targetLi = e.target.closest("li[data-device-id]"); // Find the closest LI with data-device-id
          if (targetLi) {
            var deviceId = targetLi.getAttribute("data-device-id");
            var deviceName = targetLi.childNodes[0].nodeValue.trim(); // Get text content before any buttons

            const deviceSelect = document.getElementById("deviceSelect");
            const selectedDeviceNameEl = document.getElementById("selectedDeviceName");

            if (deviceSelect) deviceSelect.value = deviceId;
            if (selectedDeviceNameEl) selectedDeviceNameEl.textContent = deviceName;

            fetchLatestLocation();
          }
        });
    }
});


// اختيار جهاز من قائمة مجموعة
function selectDeviceFromGroup(deviceId, deviceName) {
  const deviceSelect = document.getElementById("deviceSelect");
  const selectedDeviceNameEl = document.getElementById("selectedDeviceName");

  if (deviceSelect) deviceSelect.value = deviceId;
  if (selectedDeviceNameEl) selectedDeviceNameEl.textContent = htmlspecialchars(deviceName);
  fetchLatestLocation();
}

// توسيع/طي أجهزة المجموعة
function toggleGroupDevices(groupId) {
  var el = document.getElementById("group-devices-" + groupId);
  var headerEl = document.querySelector(`.group-header[onclick*="toggleGroupDevices(${groupId})"]`);
  if (!el) return;

  if (el.style.display === "none" || el.style.display === "") {
    el.style.display = "block"; // Or "table-row-group" or "list-item" depending on UL/LI
    if (headerEl) headerEl.classList.add('active');
  } else {
    el.style.display = "none";
    if (headerEl) headerEl.classList.remove('active');
  }
}

//----------------------------------------------------
//   تبديل اللغة (بسيط)
//----------------------------------------------------
var currentLang = "ar"; // افتراضياً
const translations = {
    en: {
        appTitle: "Almanara - Tracking System",
        deviceListTitle: "Devices & Groups",
        searchDevicePlaceholder: "Search device...",
        lblChooseDevice: "Choose Device:",
        lblDate: "Date:",
        lblTimeFrom: "From (12h):",
        lblTimeTo: "To (12h):",
        btnSearch: "Search",
        btnReset: "Reset",
        btnLatestLoc: "Live Location",
        btnShare: "Share Link",
        lblAnimSpeed: "Animation Speed (ms):",
        btnPlay: "▶️ Play",
        btnStop: "⏹ Stop",
        btnResetAnimation: "↩️ Reset Anim.",
        lblSelectedDeviceNamePrefix: "Selected Device:",
        selectedDeviceNameDefault: "No device selected",
        logsTitle: "Stop Logs (5 mins+)",
        btnDownloadExcel: "📥 Excel (XLSX)",
        lblLogsSearch: "Search in logs:",
        btnResetLogsFilter: "Show All",
        thPlaceName: "Place Name",
        thEntryTime: "Entry Time",
        thExitTime: "Exit Time",
        thStatus: "Status",
        thBattery: "Battery",
        analyticsTitle: "Data Analytics",
        pDistance: "Total Distance: ",
        pMovingTime: "Moving Time: ",
        pStoppedTime: "Stopped Time: ",
        pAvgSpeed: "Average Speed (moving): ",
        logout: "Logout",
        adminPanel: "Admin Panel",
        langToggle: "العربية"
    },
    ar: {
        appTitle: "Almanara - نظام التتبع",
        deviceListTitle: "الأجهزة والمجموعات",
        searchDevicePlaceholder: "🔍 ابحث عن جهاز...",
        lblChooseDevice: "🔽 اختر الجهاز:",
        lblDate: "📅 اختر التاريخ:",
        lblTimeFrom: "⏰ من (12h):",
        lblTimeTo: "⏰ إلى (12h):",
        btnSearch: "🔍 بحث",
        btnReset: "🔄 إعادة تعيين",
        btnLatestLoc: "📡 آخر موقع مباشر",
        btnShare: "🔗 مشاركة الموقع",
        lblAnimSpeed: "⚡ سرعة الحركة (ملّي ثانية):",
        btnPlay: "▶️ تشغيل الحركة",
        btnStop: "⏹ إيقاف الحركة",
        btnResetAnimation: "↩️ رجوع إلى البداية",
        lblSelectedDeviceNamePrefix: "اسم الجهاز المختار:",
        selectedDeviceNameDefault: "لا يوجد جهاز محدد",
        logsTitle: "سجلات التوقف (5 دقائق فأكثر)",
        btnDownloadExcel: "📥 Excel (XLSX)",
        lblLogsSearch: "بحث في السجلات:",
        btnResetLogsFilter: "إعادة الكل",
        thPlaceName: "اسم المكان",
        thEntryTime: "وقت الدخول",
        thExitTime: "وقت الخروج",
        thStatus: "الحالة",
        thBattery: "البطارية",
        analyticsTitle: "تحليل البيانات",
        pDistance: "المسافة المقطوعة: ",
        pMovingTime: "وقت التحرك: ",
        pStoppedTime: "وقت التوقف: ",
        pAvgSpeed: "متوسط السرعة: ",
        logout: "خروج",
        adminPanel: "لوحة الأدمن",
        langToggle: "English"
    }
};

function applyTranslations(lang) {
    const trans = translations[lang];
    document.getElementById("appTitle").textContent = trans.appTitle;
    document.getElementById("deviceListTitle").textContent = trans.deviceListTitle;
    document.getElementById("searchDevice").placeholder = trans.searchDevicePlaceholder;
    document.getElementById("lblChooseDevice").textContent = trans.lblChooseDevice;
    document.getElementById("lblDate").textContent = trans.lblDate;
    document.getElementById("lblTimeFrom").textContent = trans.lblTimeFrom;
    document.getElementById("lblTimeTo").textContent = trans.lblTimeTo;
    document.getElementById("btnSearch").innerHTML = trans.btnSearch; // innerHTML for icons
    document.getElementById("btnReset").innerHTML = trans.btnReset;
    document.getElementById("btnLatestLoc").innerHTML = trans.btnLatestLoc;
    document.getElementById("btnShare").innerHTML = trans.btnShare;
    document.getElementById("lblAnimSpeed").textContent = trans.lblAnimSpeed;
    document.getElementById("btnPlay").innerHTML = trans.btnPlay;
    document.getElementById("btnStop").innerHTML = trans.btnStop;
    document.getElementById("btnResetAnimation").innerHTML = trans.btnResetAnimation;

    const selectedDeviceNameStrong = document.getElementById("lblSelectedDeviceName");
    if(selectedDeviceNameStrong) selectedDeviceNameStrong.textContent = trans.lblSelectedDeviceNamePrefix;
    const selectedDeviceNameSpan = document.getElementById("selectedDeviceName");
    if(selectedDeviceNameSpan && selectedDeviceNameSpan.textContent === translations[currentLang === 'ar' ? 'en' : 'ar'].selectedDeviceNameDefault) { // Only change if it's the default
        selectedDeviceNameSpan.textContent = trans.selectedDeviceNameDefault;
    }


    document.getElementById("logsTitle").textContent = trans.logsTitle;
    document.getElementById("btnDownloadExcel").innerHTML = trans.btnDownloadExcel;
    document.getElementById("lblLogsSearch").textContent = trans.lblLogsSearch;
    document.getElementById("btnResetLogsFilter").textContent = trans.btnResetLogsFilter;
    document.getElementById("thPlaceName").textContent = trans.thPlaceName;
    document.getElementById("thEntryTime").textContent = trans.thEntryTime;
    document.getElementById("thExitTime").textContent = trans.thExitTime;
    document.getElementById("thStatus").textContent = trans.thStatus;
    document.getElementById("thBattery").textContent = trans.thBattery;

    // Analytics text needs careful handling as it's dynamically generated
    applyEnglishAnalytics(); // Call this to update dynamic parts

    document.querySelector(".top-nav .actions button[onclick*='logout.php']").textContent = trans.logout;
    const adminLink = document.querySelector(".top-nav .actions a[href='admin.php']");
    if (adminLink) adminLink.textContent = trans.adminPanel;
    document.getElementById("langLabel").textContent = trans.langToggle;

    document.documentElement.lang = lang;
}
function applyEnglishAnalytics() {
    // This function is called when language is switched to English OR
    // when updateAnalytics is called and language is already English.
    if (currentLang === "en") {
        const analyticsDiv = document.getElementById("analytics");
        if (analyticsDiv) {
            const titleEl = analyticsDiv.querySelector("#analyticsTitle");
            if (titleEl) titleEl.textContent = translations.en.analyticsTitle;

            // For pDistance, pMovingTime, etc., we need to re-append the English prefix
            // This assumes the numeric part is already there from updateAnalytics()
            const pDistanceEl = document.getElementById("pDistance");
            if (pDistanceEl && pDistanceEl.textContent.includes("كم")) { // Check if it's Arabic
                pDistanceEl.textContent = translations.en.pDistance + pDistanceEl.textContent.split(": ")[1].replace("كم", "km");
            }
            const pMovingTimeEl = document.getElementById("pMovingTime");
            if (pMovingTimeEl && pMovingTimeEl.textContent.includes("دقيقة")) {
                pMovingTimeEl.textContent = translations.en.pMovingTime + pMovingTimeEl.textContent.split(": ")[1].replace("دقيقة", "min");
            }
            const pStoppedTimeEl = document.getElementById("pStoppedTime");
            if (pStoppedTimeEl && pStoppedTimeEl.textContent.includes("دقيقة")) {
                pStoppedTimeEl.textContent = translations.en.pStoppedTime + pStoppedTimeEl.textContent.split(": ")[1].replace("دقيقة", "min");
            }
            const pAvgSpeedEl = document.getElementById("pAvgSpeed");
            if (pAvgSpeedEl && pAvgSpeedEl.textContent.includes("كم/س")) {
                 pAvgSpeedEl.textContent = translations.en.pAvgSpeed + pAvgSpeedEl.textContent.split(": ")[1].replace("كم/س", "km/h");
            }
        }
        // Update chart labels if charts exist
        if (timeChartInstance) {
            timeChartInstance.data.labels = [translations.en.pMovingTime.replace(': ', ' (min)'), translations.en.pStoppedTime.replace(': ', ' (min)')];
            timeChartInstance.update();
        }
        if (speedChartInstance) {
            speedChartInstance.data.datasets[0].label = translations.en.pAvgSpeed.replace(': ', ' (km/h)');
            speedChartInstance.data.labels = speedChartInstance.data.labels.map((l,i) => "Point " + (i+1));
            speedChartInstance.update();
        }
    } else {
        // If currentLang is 'ar', updateAnalytics() will handle Arabic text.
        // We might need to call updateAnalytics() again if lang switched from EN to AR
        // to regenerate with Arabic labels if data hasn't changed.
        // For simplicity, let's assume updateAnalytics() will be called if data changes.
    }
}


function toggleLanguage() {
  currentLang = (currentLang === "ar") ? "en" : "ar";
  localStorage.setItem('preferredLang', currentLang);
  applyTranslations(currentLang);
  // If charts exist, their labels/titles might need updating directly or by re-calling updateAnalytics
  updateAnalytics(); // Re-render analytics which will pick up currentLang
}

// Load preferred language on page load
document.addEventListener('DOMContentLoaded', function() {
    const preferredLang = localStorage.getItem('preferredLang');
    if (preferredLang && preferredLang !== currentLang) { // if stored lang is different from default
        currentLang = preferredLang;
    }
    applyTranslations(currentLang); // Apply initial language (either default 'ar' or stored)
    updateAnalytics(); // Initial analytics rendering with correct language
});