<?php
session_start();
require 'config.php';  // ملف الاتصال بقاعة البيانات

// ضبط الترميز UTF-8 للتأكد من إدخال وقراءة النصوص العربية بشكل صحيح
header('Content-Type: text/html; charset=utf-8');
$conn->exec("SET NAMES 'utf8mb4'");

//------------------------------------------------------
// التحقق من تسجيل الدخول
//------------------------------------------------------
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id   = $_SESSION['user_id'];
$user_role = $_SESSION['role'] ?? 'user'; // قد يكون 'admin' أو 'user'

// تهيئة متغيرات التخزين المؤقت للبيانات إذا لم تكن موجودة
if (!isset($_SESSION['device_data_needs_refresh'])) {
    $_SESSION['device_data_needs_refresh'] = true; // تحتاج إلى تحديث في المرة الأولى
}
if (!isset($_SESSION['device_data_cache'])) {
    $_SESSION['device_data_cache'] = [];
    $_SESSION['device_data_needs_refresh'] = true; // تحتاج إلى تحديث إذا كان الكاش فارغًا
}


//----------------------------------------------------------------------------------
// [1] منطق إنشاء المجموعات وإدارتها (Admin فقط)
//----------------------------------------------------------------------------------
if ($user_role === 'admin') {

    // إنشاء مجموعة جديدة
    if (isset($_POST['create_group']) && !empty($_POST['group_name'])) {
        $group_name = trim($_POST['group_name']);
        if ($group_name !== "") {
            $stmt = $conn->prepare("INSERT INTO device_groups (group_name) VALUES (?)");
            $stmt->execute([$group_name]);
            $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        }
    }

    // تعديل اسم المجموعة
    if (isset($_POST['rename_group']) && !empty($_POST['new_group_name']) && !empty($_POST['group_id'])) {
        $newName = trim($_POST['new_group_name']);
        $gid     = (int) $_POST['group_id'];

        if ($newName !== "") {
            $stmt = $conn->prepare("UPDATE device_groups SET group_name = ? WHERE id = ?");
            $stmt->execute([$newName, $gid]);
            $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش
        }
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }

    // إضافة جهاز إلى مجموعة
    if (isset($_POST['add_device_to_group']) && !empty($_POST['selected_group']) && !empty($_POST['selected_device_for_group'])) {
        $group_id  = $_POST['selected_group'];
        $device_id = $_POST['selected_device_for_group'];

        // التحقق إن كان الجهاز مضافًا مسبقًا في نفس المجموعة لتفادي التكرار
        $checkStmt = $conn->prepare("SELECT * FROM device_group_members WHERE group_id = ? AND device_id = ?");
        $checkStmt->execute([$group_id, $device_id]);
        if ($checkStmt->rowCount() == 0) {
            $stmt = $conn->prepare("INSERT INTO device_group_members (group_id, device_id) VALUES (?, ?)");
            $stmt->execute([$group_id, $device_id]);
            $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش
        }
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }

    // حذف جهاز من داخل المجموعة
    if (isset($_GET['remove_device_from_group']) && isset($_GET['group_id'])) {
        $gid = (int) $_GET['group_id'];
        $did = $_GET['remove_device_from_group'];

        $stmt = $conn->prepare("DELETE FROM device_group_members WHERE group_id = ? AND device_id = ?");
        $stmt->execute([$gid, $did]);
        $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش

        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }

    // حذف مجموعة كاملة
    if (isset($_GET['delete_group'])) {
        $delete_group_id = (int) $_GET['delete_group'];
        try {
            // استخدام المعاملات لحذف آمن
            $conn->beginTransaction();

            // حذف الأجهزة المرتبطة بالمجموعة
            $stmtDelMembers = $conn->prepare("DELETE FROM device_group_members WHERE group_id = ?");
            $stmtDelMembers->execute([$delete_group_id]);

            // حذف المجموعة نفسها
            $stmtDelGroup = $conn->prepare("DELETE FROM device_groups WHERE id = ?");
            $stmtDelGroup->execute([$delete_group_id]);

            $conn->commit();
            $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش

            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        } catch (Exception $e) {
            $conn->rollBack();
            echo "خطأ في حذف المجموعة: " . $e->getMessage();
            exit;
        }
    }

    // نقل جهاز من مجموعة إلى أخرى
    if (isset($_POST['move_device']) 
        && !empty($_POST['move_device_from_group']) 
        && !empty($_POST['move_device_to_group']) 
        && !empty($_POST['device_id_to_move'])) 
    {
        $fromGroup = $_POST['move_device_from_group'];
        $toGroup   = $_POST['move_device_to_group'];
        $deviceId  = $_POST['device_id_to_move'];

        // أولاً: حذف الجهاز من المجموعة القديمة
        $stmt = $conn->prepare("DELETE FROM device_group_members WHERE group_id = ? AND device_id = ?");
        $stmt->execute([$fromGroup, $deviceId]);

        // ثانيًا: إضافة الجهاز إلى المجموعة الجديدة (إذا لم يكن مضافًا)
        $checkStmt = $conn->prepare("SELECT * FROM device_group_members WHERE group_id = ? AND device_id = ?");
        $checkStmt->execute([$toGroup, $deviceId]);
        if ($checkStmt->rowCount() == 0) {
            $stmt = $conn->prepare("INSERT INTO device_group_members (group_id, device_id) VALUES (?, ?)");
            $stmt->execute([$toGroup, $deviceId]);
        }
        $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش

        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
}


//------------------------------------------------------
// حذف بيانات جهاز (للمسؤول فقط)
//------------------------------------------------------
if (isset($_GET['delete_device']) && $user_role === 'admin') {
    $device_id = $_GET['delete_device'];
    // حذف جميع سجلات الجهاز من جدول المواقع
    $stmt = $conn->prepare("DELETE FROM locations WHERE device_id = ?");
    $stmt->execute([$device_id]);
    // قد نحتاج أيضًا لحذف الجهاز من جدول الأجهزة الرئيسي إذا كان موجودًا ومستقلاً
    // $stmt_delete_device = $conn->prepare("DELETE FROM devices WHERE device_id = ?");
    // $stmt_delete_device->execute([$device_id]);
    $_SESSION['device_data_needs_refresh'] = true; // تحديث الكاش
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}


//----------------------------------------------------------------------------------
// [2] جلب الأجهزة والمجموعات من قاعدة البيانات أو من الكاش
//----------------------------------------------------------------------------------
if ($_SESSION['device_data_needs_refresh'] === true || empty($_SESSION['device_data_cache'])) {
    // جلب فقط الأجهزة غير المضافة لأي مجموعة لعرضها في القائمة الجانبية
    $stmt = $conn->query("
        SELECT DISTINCT l.device_id, l.device_name
        FROM locations l
        WHERE l.device_id NOT IN (
            SELECT DISTINCT device_id FROM device_group_members
        )
        ORDER BY l.device_name ASC
    ");
    $ungroupedDevices = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // جلب كل الأجهزة لاستخدامها في صندوق الاختيار الرئيسي وصندوق "نقل جهاز" و"إضافة جهاز"
    $stmtAll = $conn->query("
        SELECT DISTINCT device_id, device_name 
        FROM locations 
        ORDER BY device_name ASC
    ");
    $allDevices = $stmtAll->fetchAll(PDO::FETCH_ASSOC);

    // جلب المجموعات
    $stmtGroups = $conn->query("SELECT * FROM device_groups ORDER BY group_name ASC");
    $groups = $stmtGroups->fetchAll(PDO::FETCH_ASSOC);

    // جلب الأجهزة المتعلقة بكل مجموعة
    $groupedDevices = [];
    foreach ($groups as $grp) {
        $gid = $grp['id'];
        $stmtG = $conn->prepare("
            SELECT DISTINCT l.device_id, l.device_name
            FROM locations l
            JOIN device_group_members gm ON l.device_id = gm.device_id
            WHERE gm.group_id = ?
            ORDER BY l.device_name ASC
        ");
        $stmtG->execute([$gid]);
        $groupedDevices[$gid] = $stmtG->fetchAll(PDO::FETCH_ASSOC);
    }

    // تخزين البيانات في الكاش
    $_SESSION['device_data_cache'] = [
        'ungroupedDevices' => $ungroupedDevices,
        'allDevices'       => $allDevices,
        'groups'           => $groups,
        'groupedDevices'   => $groupedDevices
    ];
    $_SESSION['device_data_needs_refresh'] = false; // تم التحديث

} else {
    // استخدام البيانات من الكاش
    $ungroupedDevices = $_SESSION['device_data_cache']['ungroupedDevices'];
    $allDevices       = $_SESSION['device_data_cache']['allDevices'];
    $groups           = $_SESSION['device_data_cache']['groups'];
    $groupedDevices   = $_SESSION['device_data_cache']['groupedDevices'];
}


//----------------------------------------------------------------------------------
// [3] منطق مشاركة الموقع إذا تم الطلب عبر GET
//----------------------------------------------------------------------------------
if (isset($_GET['share']) && isset($_GET['device_id']) && isset($_GET['expires']) && isset($_GET['device_name'])) {
    $device_id   = $_GET['device_id'];
    $device_name = $_GET['device_name'];
    $expires     = $_GET['expires'];
    ?>
    <!DOCTYPE html>
    <html lang="ar">
    <head>
      <meta charset="UTF-8">
      <meta name="viewport" content="width=device-width, initial-scale=1.0">
      <title>عرض الموقع المشترك</title>
      <!-- Leaflet -->
      <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.3/dist/leaflet.css"/>
      <script src="https://unpkg.com/leaflet@1.9.3/dist/leaflet.js"></script>
      <style>
        body {
          font-family: Arial, sans-serif;
          margin: 0; 
          padding: 0;
          background-color: #121212; 
          color: #e0e0e0;
          text-align: center;
        }
        .header {
          background: #007BFF;
          padding: 15px;
          font-size: 20px;
          font-weight: bold;
          color: #fff;
        }
        #sharedMap {
          height: 600px;
          width: 100%;
          margin-top: 20px;
        }
      </style>
    </head>
    <body>
      <div class="header">
        عرض الموقع المشترك للجهاز: <?php echo htmlspecialchars($device_name, ENT_QUOTES, 'UTF-8'); ?>
      </div>
      <div id="sharedMap"></div>
      <script>
        var shareExpires = <?php echo json_encode($expires); ?>;
        var deviceId     = <?php echo json_encode($device_id); ?>;
        
        // إنشاء الخريطة
        var map = L.map('sharedMap', { center: [30, 0], zoom: 2 });
        L.tileLayer(
          "https://api.mapbox.com/styles/v1/mapbox/outdoors-v11/tiles/{z}/{x}/{y}?access_token=pk.eyJ1IjoiaWJyYWhpbWNvbSIsImEiOiJja2YxZnB6OWowc2xxMnFxZGxya2ppdmZxIn0.keECYQuu1Bs0JeWDP8dSFQ", 
          {
            maxZoom: 20,
            tileSize: 512,
            zoomOffset: -1,
            attribution: '&copy; <a href="https://www.openstreetmap.org/">OSM</a> & <a href="https://www.mapbox.com/">Mapbox</a>'
          }
        ).addTo(map);

        var sharedMarker = null;

        function fetchSharedLocation() {
            if (Date.now() > shareExpires) {
                alert("انتهت صلاحية مشاركة الموقع.");
                return;
            }
            fetch("get_latest_location.php?device_id=" + deviceId + "&t=" + Date.now())
                .then(res => res.json())
                .then(data => {
                    if (data.status === "success") {
                        var lat = parseFloat(data.data.latitude);
                        var lng = parseFloat(data.data.longitude);
                        var battery = data.data.battery;
                        if (!isNaN(lat) && !isNaN(lng)) {
                            if (!sharedMarker) {
                                sharedMarker = L.marker([lat, lng]).addTo(map);
                            } else {
                                sharedMarker.setLatLng([lat, lng]);
                            }
                            sharedMarker.bindPopup(`
                                <b>موقع الجهاز: <?php echo htmlspecialchars($device_name, ENT_QUOTES, 'UTF-8'); ?></b><br>
                                البطارية: ${battery ? battery + '%' : 'غير معروفة'}<br>
                                <a href="https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}" target="_blank">خرائط جوجل</a> |
                                <a href="https://waze.com/ul?ll=${lat},${lng}&navigate=yes" target="_blank">Waze</a>
                            `).openPopup();
                            map.setView([lat, lng], 15);
                        }
                    }
                })
                .catch(err => console.error("⚠️ خطأ في جلب الموقع:", err));
        }

        // جلب الموقع فوراً ثم تحديثه كل 10 ثوانٍ
        fetchSharedLocation();
        setInterval(fetchSharedLocation, 10000);
      </script>
    </body>
    </html>
    <?php
    exit;
}
?>
<!DOCTYPE html>
<html lang="ar">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Almanara - نظام التتبع المطوَّر</title>

  <!-- Leaflet -->
  <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.3/dist/leaflet.css"/>
  <script src="https://unpkg.com/leaflet@1.9.3/dist/leaflet.js"></script>

  <!-- Mapbox GL (للجيوكودينغ) -->
  <script src="https://api.mapbox.com/mapbox-gl-js/v2.6.1/mapbox-gl.js"></script>
  <link href="https://api.mapbox.com/mapbox-gl-js/v2.6.1/mapbox-gl.css" rel="stylesheet"/>

  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

  <!-- SheetJS -->
  <script src="https://cdn.sheetjs.com/xlsx-latest/package/dist/xlsx.full.min.js"></script>

  <!-- Loading Overlay -->
  <div id="loadingOverlay">
    <div class="spinner"></div>
  </div>

  <style>
    /* CSS Variables for Theme */
    :root {
      --bg-primary: #f5f7fa;
      --bg-secondary: #ffffff;
      --text-primary: #2d3436;
      --text-secondary: #636e72;
      --accent-color: #0396FF;
      --accent-color-dark: #0D47A1;
      --border-color: #e1e8ed;
      --shadow-color: rgba(0,0,0,0.1);
      --hover-bg: #f8f9fa;
    }

    /* Dark Mode Variables */
    [data-theme="dark"] {
      --bg-primary: #121212;
      --bg-secondary: #1e1e1e;
      --text-primary: #e0e0e0;
      --text-secondary: #a0a0a0;
      --accent-color: #2196F3;
      --border-color: #2d2d2d;
      --shadow-color: rgba(0,0,0,0.3);
      --hover-bg: #2d2d2d;
    }

    /* RESET & BASE STYLES */
    * { box-sizing: border-box; }
    body {
      font-family: 'Segoe UI', 'Helvetica Neue', Arial, sans-serif;
      margin: 0; 
      padding: 0;
      background-color: var(--bg-primary); 
      color: var(--text-primary);
      display: flex; 
      flex-direction: column; 
      min-height: 100vh;
      line-height: 1.6;
      transition: background-color 0.3s ease, color 0.3s ease;
    }
    /* TOP NAV */
    .top-nav {
      width: 100%; 
      background: linear-gradient(135deg, var(--accent-color), var(--accent-color-dark)); 
      padding: 15px 25px;
      box-shadow: 0 4px 12px var(--shadow-color);
      display: flex; 
      justify-content: space-between; 
      align-items: center;
      flex-wrap: wrap;
      position: sticky;
      top: 0;
      z-index: 1000;
      transition: background 0.3s ease, box-shadow 0.3s ease;
    }
    .top-nav .logo { 
      font-size: 22px; 
      color: #fff; 
      font-weight: 600;
      letter-spacing: 0.5px;
      text-shadow: 1px 1px 2px rgba(0,0,0,0.1);
    }
    .top-nav .actions { display: flex; gap: 12px; }
    .top-nav .actions a,
    .top-nav .actions button {
      color: #fff; 
      background: rgba(255,255,255,0.15); 
      border: 1px solid rgba(255,255,255,0.2);
      font-size: 15px; 
      border-radius: 8px; 
      padding: 10px 16px;
      cursor: pointer; 
      transition: all 0.3s ease;
      text-decoration: none;
      backdrop-filter: blur(5px);
      display: flex;
      align-items: center;
      gap: 6px;
    }
    .top-nav .actions a:hover,
    .top-nav .actions button:hover { 
      background: rgba(255,255,255,0.25);
      transform: translateY(-1px);
      box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    }
    /* CONTAINER LAYOUT */
    .container { display: flex; flex: 1; overflow: hidden; }
    /* SIDEBAR */
    .sidebar {
      width: 320px; 
      background: var(--bg-secondary); 
      padding: 20px; 
      overflow-y: auto;
      flex-shrink: 0;
      border-right: 1px solid var(--border-color);
      box-shadow: 2px 0 10px var(--shadow-color);
      transition: background-color 0.3s ease, border-color 0.3s ease, box-shadow 0.3s ease;
    }
    .sidebar h2 { 
      text-align: center; 
      margin-top: 0;
      color: #2d3436;
      font-weight: 600;
      font-size: 1.4em;
      margin-bottom: 20px;
    }
    .sidebar input[type="text"] {
      width: 100%; 
      padding: 12px 15px;
      margin-bottom: 15px;
      border-radius: 10px; 
      border: 1px solid #e1e8ed;
      background: #f8f9fa; 
      color: #2d3436;
      font-size: 14px;
      transition: all 0.3s ease;
    }
    .sidebar input[type="text"]:focus {
      border-color: #0396FF;
      box-shadow: 0 0 0 3px rgba(3,150,255,0.1);
      outline: none;
    }
    .sidebar ul { 
      list-style: none; 
      padding: 0; 
      margin: 0;
      border-radius: 12px;
      overflow: hidden;
    }
    .sidebar ul li {
      padding: 12px 15px;
      border-bottom: 1px solid var(--border-color);
      cursor: pointer; 
      transition: all 0.3s ease;
      position: relative;
      background: var(--bg-secondary);
      color: var(--text-primary);
      display: flex;
      align-items: center;
      justify-content: space-between;
    }
    .sidebar ul li:hover { background: #333; }
    /* زر الحذف للأجهزة (للمسؤول فقط) */
    .delete-btn {
      color: #fff;
      background: #d9534f;
      border: none;
      border-radius: 3px;
      padding: 2px 5px;
      margin-left: 10px;
      cursor: pointer;
      font-size: 12px;
    }
    .edit-btn {
      color: #fff;
      background: #f0ad4e;
      border: none;
      border-radius: 3px;
      padding: 2px 5px;
      margin-left: 10px;
      cursor: pointer;
      font-size: 12px;
    }
    /* MAIN CONTENT */
    .main-content {
      flex: 1; 
      padding: 15px; 
      background: var(--bg-secondary); 
      overflow-y: auto;
      display: flex; 
      flex-direction: column;
      transition: background-color 0.3s ease;
    }
    /* CONTROLS */
    .controls {
      background: #fff;
      padding: 20px;
      display: flex;
      flex-wrap: wrap;
      gap: 15px;
      align-items: center;
      justify-content: center;
      border-radius: 15px;
      box-shadow: 0 8px 16px rgba(0,0,0,0.1);
      position: sticky;
      top: 20px;
      z-index: 1000;
      margin: 0 20px 20px 20px;
      backdrop-filter: blur(10px);
      border: 1px solid rgba(255,255,255,0.1);
    }
    .controls select, .controls input {
      padding: 12px 15px;
      font-size: 15px;
      border-radius: 10px;
      border: 1px solid #e1e8ed;
      background: #f8f9fa;
      color: #2d3436;
      max-width: 200px;
      transition: all 0.3s ease;
    }
    .controls select:focus, .controls input:focus {
      border-color: #0396FF;
      box-shadow: 0 0 0 3px rgba(3,150,255,0.1);
      outline: none;
    }
    .controls button {
      padding: 12px 20px;
      font-size: 15px;
      border-radius: 10px;
      background: linear-gradient(135deg, #0396FF, #0D47A1);
      border: none;
      font-weight: 600;
      cursor: pointer;
      transition: all 0.3s ease;
      color: #fff;
      display: flex;
      align-items: center;
      gap: 8px;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    .controls button:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 12px rgba(0,0,0,0.15);
    }
    #selectedDeviceContainer { color: #fff; margin: 10px; text-align: center; }
    /* MAP SECTION */
    #map {
      height: 600px; width: 90%; margin: 0 auto 20px auto;
      border-radius: 10px; box-shadow: 0 0 10px rgba(0,0,0,0.5);
    }
    /* LOGS SECTION */
    .logs {
      width: 90%;
      margin: 30px auto;
      padding: 25px;
      background: #fff;
      border-radius: 15px;
      box-shadow: 0 10px 20px rgba(0,0,0,0.1);
    }
    .logs h2 {
      margin: 0 0 20px 0;
      text-align: center;
      color: #2d3436;
      font-weight: 600;
      font-size: 1.5em;
    }
    .logs-controls {
      text-align: center;
      margin-bottom: 20px;
      display: flex;
      flex-wrap: wrap;
      justify-content: center;
      gap: 15px;
      background: #f8f9fa;
      padding: 15px;
      border-radius: 10px;
    }
    .logs-controls label {
      display: inline-flex;
      align-items: center;
      font-weight: 500;
      color: #2d3436;
    }
    .logs-controls input[type="text"] {
      width: 220px;
      padding: 12px 15px;
      border-radius: 10px;
      border: 1px solid #e1e8ed;
      background: #fff;
      color: #2d3436;
      transition: all 0.3s ease;
    }
    .logs-controls input[type="text"]:focus {
      border-color: #0396FF;
      box-shadow: 0 0 0 3px rgba(3,150,255,0.1);
      outline: none;
    }
    .logs table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
      margin-top: 15px;
    }
    .logs th {
      background: #f8f9fa;
      color: #2d3436;
      font-weight: 600;
      padding: 15px;
      text-align: center;
      border-bottom: 2px solid #e1e8ed;
      position: sticky;
      top: 0;
    }
    .logs td {
      padding: 12px 15px;
      text-align: center;
      border-bottom: 1px solid #e1e8ed;
      color: #2d3436;
      transition: background 0.3s ease;
    }
    .logs tr:hover td {
      background: #f8f9fa;
    }
    .logs th.sortable {
      cursor: pointer;
      transition: all 0.3s ease;
    }
    .logs th.sortable:hover {
      background: #e1e8ed;
      color: #0396FF;
    }
    .logs th.sortable::after {
      content: '⌄';
      display: inline-block;
      margin-left: 5px;
      transition: transform 0.3s ease;
    }
    .logs th.sortable.sorted-asc::after {
      transform: rotate(180deg);
    }
    /* ANALYTICS SECTION */
    #analytics {
      width: 90%;
      max-width: 1200px;
      margin: 30px auto;
      background: #fff;
      padding: 25px;
      border-radius: 15px;
      box-shadow: 0 10px 20px rgba(0,0,0,0.1);
      text-align: center;
    }
    #analytics h2 {
      margin: 0 0 20px 0;
      color: #2d3436;
      font-weight: 600;
      font-size: 1.5em;
    }
    #analytics .chart-container {
      background: #f8f9fa;
      padding: 15px;
      border-radius: 10px;
      margin-top: 20px;
    }
    /* TOAST NOTIFICATIONS */
    #toastContainer {
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 9999;
      display: flex;
      flex-direction: column;
      gap: 12px;
    }
    .toast {
      background: linear-gradient(135deg, #0396FF, #0D47A1);
      color: #fff;
      padding: 12px 20px;
      border-radius: 10px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      animation: fadein 0.5s, fadeout 0.5s 3.5s;
      display: flex;
      align-items: center;
      gap: 10px;
      font-weight: 500;
      backdrop-filter: blur(5px);
      border: 1px solid rgba(255,255,255,0.1);
    }
    .toast::before {
      content: '🔔';
      font-size: 1.2em;
    }
    @keyframes fadein { from { opacity: 0; } to { opacity: 1; } }
    @keyframes fadeout { from { opacity: 1; } to { opacity: 0; } }
    /* SMALL SCREENS */
    @media (max-width: 768px) {
      .container { flex-direction: column; }
      .sidebar { width: 100%; order: 2; max-height: 200px; }
      .main-content { width: 100%; order: 1; }
      .controls { position: static; margin-bottom: 20px; }
      #map { width: 100%; }
    }
    /* ستايل خاص لعنصر المجموعة */
    .group-header {
      background-color: #f8f9fa;
      font-weight: 600;
      cursor: pointer;
      position: relative;
      padding: 15px;
      border-radius: 10px;
      margin-bottom: 5px;
      transition: all 0.3s ease;
      display: flex;
      align-items: center;
      justify-content: space-between;
      border: 1px solid #e1e8ed;
    }
    .group-header:hover {
      background-color: #e9ecef;
      transform: translateX(-2px);
    }
    .group-header:after {
      content: '▼';
      font-size: 14px;
      color: #0396FF;
      transition: transform 0.3s ease;
    }
    .group-header.active:after {
      transform: rotate(180deg);
    }
    .group-devices {
      display: none;
      background-color: #fff;
      border-radius: 8px;
      margin: 5px 0 10px 0;
      overflow: hidden;
      box-shadow: inset 0 2px 4px rgba(0,0,0,0.05);
    }
    .group-devices li {
      border-bottom: 1px solid #e1e8ed;
      cursor: pointer;
      padding: 12px 15px;
      transition: all 0.3s ease;
      display: flex;
      align-items: center;
      justify-content: space-between;
    }
    .group-devices li:hover {
      background-color: #f8f9fa;
      padding-left: 20px;
    }
    .group-devices li:last-child {
      border-bottom: none;
    }
    .delete-btn, .edit-btn {
      padding: 6px 12px;
      border-radius: 6px;
      font-size: 13px;
      font-weight: 500;
      transition: all 0.3s ease;
      border: none;
      display: inline-flex;
      align-items: center;
      gap: 4px;
    }
    .delete-btn {
      background: #ff4757;
      color: #fff;
    }
    .edit-btn {
      background: #ffa502;
      color: #fff;
    }
    .delete-btn:hover, .edit-btn:hover {
      transform: translateY(-1px);
      box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
  </style>
  <script>
    // Loading Overlay Functions
    function showLoading() {
      document.getElementById('loadingOverlay').style.display = 'flex';
    }
    
    function hideLoading() {
      document.getElementById('loadingOverlay').style.display = 'none';
    }
    
    // Add loading state to all forms
    document.addEventListener('DOMContentLoaded', function() {
      const forms = document.querySelectorAll('form');
      forms.forEach(form => {
        form.addEventListener('submit', function() {
          showLoading();
        });
      });
    });
    
    // التحقق من وجود تفضيل مسبق للوضع
    const currentTheme = localStorage.getItem('theme') || 'light';
    document.documentElement.setAttribute('data-theme', currentTheme);

    // تبديل الوضع بين الداكن والفاتح
    function toggleTheme() {
      const currentTheme = document.documentElement.getAttribute('data-theme');
      const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
      document.documentElement.setAttribute('data-theme', newTheme);
      localStorage.setItem('theme', newTheme);
      
      // تحديث أيقونة الزر
      const themeToggle = document.getElementById('themeToggle');
      themeToggle.textContent = newTheme === 'dark' ? '🌙' : '☀️';
    }

    // تحديث أيقونة الزر عند تحميل الصفحة
    window.addEventListener('load', () => {
      const themeToggle = document.getElementById('themeToggle');
      themeToggle.textContent = currentTheme === 'dark' ? '🌙' : '☀️';
    });
  </script>
</head>
<body>

  <!-- شريط علوي بسيط -->
  <nav class="top-nav">
    <div class="logo" id="appTitle">Almanara - نظام التتبع</div>
    <div class="actions">
      <?php if ($user_role === 'admin'): ?>
        <a href="admin.php">لوحة الأدمن</a>
      <?php endif; ?>
      <button id="themeToggle" onclick="toggleTheme()">🌙</button>
      <button id="langLabel" onclick="toggleLanguage()">English</button>
      <button onclick="window.location='logout.php'">خروج</button>
    </div>
  </nav>

  <div class="container">
    <!-- القائمة الجانبية للأجهزة + المجموعات -->
    <div class="sidebar">
      <h2 id="deviceListTitle">الأجهزة والمجموعات</h2>
      <input type="text" id="searchDevice" placeholder="🔍 ابحث عن جهاز..." oninput="filterDevices()">

      <ul id="deviceList">
        <!-- عرض جميع الأجهزة غير المضافة لأي مجموعة -->
        <?php foreach ($ungroupedDevices as $device): ?>
          <li data-device-id="<?= htmlspecialchars($device['device_id'], ENT_QUOTES, 'UTF-8') ?>">
            <?= htmlspecialchars($device['device_name'], ENT_QUOTES, 'UTF-8') ?>
            <?php if ($user_role === 'admin'): ?>
              <a href="?delete_device=<?= htmlspecialchars($device['device_id'], ENT_QUOTES, 'UTF-8') ?>" 
                 class="delete-btn" 
                 onclick="return confirm('هل أنت متأكد من حذف معلومات هذا الجهاز؟')">
                حذف
              </a>
            <?php endif; ?>
          </li>
        <?php endforeach; ?>

        <!-- عرض المجموعات بشكل قابل للطيّ -->
        <?php foreach ($groups as $grp): ?>
          <li class="group-header" onclick="toggleGroupDevices(<?= $grp['id'] ?>)">
            <?php
              // إذا طلب المستخدم تحرير اسم هذه المجموعة، نظهر فورم التعديل بدل الاسم العادي
              if (isset($_GET['edit_group']) && $_GET['edit_group'] == $grp['id'] && $user_role === 'admin'):
            ?>
              <form method="post" style="display:inline;" onclick="event.stopPropagation();">
                <input type="text" name="new_group_name" value="<?= htmlspecialchars($grp['group_name'], ENT_QUOTES, 'UTF-8') ?>" required>
                <input type="hidden" name="group_id" value="<?= $grp['id'] ?>">
                <button type="submit" name="rename_group">حفظ</button>
              </form>
            <?php else: ?>
              <?= htmlspecialchars($grp['group_name'], ENT_QUOTES, 'UTF-8') ?>
              <?php if ($user_role === 'admin'): ?>
                <a href="?edit_group=<?= $grp['id'] ?>" class="edit-btn" style="float:left;margin-left:5px;"
                   onclick="event.stopPropagation();">
                  تعديل
                </a>
                <a href="?delete_group=<?= $grp['id'] ?>" 
                   class="delete-btn" 
                   style="float: left; margin-left: 5px;"
                   onclick="event.stopPropagation();return confirm('هل أنت متأكد من حذف هذه المجموعة؟')">
                  حذف
                </a>
              <?php endif; ?>
            <?php endif; ?>
          </li>

          <ul id="group-devices-<?= $grp['id'] ?>" class="group-devices">
            <?php if (!empty($groupedDevices[$grp['id']])): ?>
              <?php foreach ($groupedDevices[$grp['id']] as $dvc): ?>
                <li onclick="selectDeviceFromGroup('<?= htmlspecialchars($dvc['device_id'], ENT_QUOTES, 'UTF-8') ?>','<?= htmlspecialchars($dvc['device_name'], ENT_QUOTES, 'UTF-8') ?>')">
                  <?= htmlspecialchars($dvc['device_name'], ENT_QUOTES, 'UTF-8') ?>
                  <?php if ($user_role === 'admin'): ?>
                    <a href="?remove_device_from_group=<?= htmlspecialchars($dvc['device_id'], ENT_QUOTES, 'UTF-8') ?>&group_id=<?= $grp['id'] ?>" 
                       class="delete-btn" 
                       style="float:right;"
                       onclick="event.stopPropagation();return confirm('هل تريد إزالة هذا الجهاز من المجموعة؟')">
                      حذف
                    </a>
                  <?php endif; ?>
                </li>
              <?php endforeach; ?>
            <?php else: ?>
              <li style="cursor: default; color: #bbb;">لا توجد أجهزة في هذه المجموعة</li>
            <?php endif; ?>
          </ul>
        <?php endforeach; ?>
      </ul>

      <!-- قسم خاص بإضافة المجموعات وأجهزة للمجموعة (للمسؤول فقط) -->
      <?php if ($user_role === 'admin'): ?>
        <hr style="margin: 20px 0;">
        <h3 style="text-align:center;">إدارة المجموعات</h3>
        <!-- نموذج إنشاء مجموعة جديدة -->
        <form method="post" style="margin-top: 10px;">
          <input type="text" name="group_name" placeholder="اسم المجموعة الجديد..." required>
          <button type="submit" name="create_group">إنشاء</button>
        </form>

        <hr style="margin: 20px 0;">

        <!-- نموذج إضافة جهاز إلى مجموعة -->
        <form method="post">
          <label>اختر مجموعة:</label><br>
          <select name="selected_group" required style="margin-bottom:8px; width:100%;">
            <option value="">-- اختر --</option>
            <?php foreach ($groups as $grp): ?>
              <option value="<?= $grp['id'] ?>">
                <?= htmlspecialchars($grp['group_name'], ENT_QUOTES, 'UTF-8') ?>
              </option>
            <?php endforeach; ?>
          </select>

          <label>اختر جهاز (يمكن أن يكون في مجموعة أخرى):</label><br>
          <!-- التعديل هنا: استخدام allDevices لعرض جميع الأجهزة -->
          <select name="selected_device_for_group" required style="margin-bottom:8px; width:100%;">
            <option value="">-- اختر --</option>
            <?php foreach ($allDevices as $device): ?>
              <option value="<?= htmlspecialchars($device['device_id'], ENT_QUOTES, 'UTF-8') ?>">
                <?= htmlspecialchars($device['device_name'], ENT_QUOTES, 'UTF-8') ?>
              </option>
            <?php endforeach; ?>
          </select>

          <button type="submit" name="add_device_to_group">إضافة الجهاز للمجموعة</button>
        </form>

        <hr style="margin: 20px 0;">

        <!-- نموذج نقل جهاز من مجموعة لأخرى -->
        <form method="post">
          <h4>نقل جهاز من مجموعة لأخرى</h4>
          <label>من المجموعة:</label><br>
          <select name="move_device_from_group" required style="margin-bottom:8px; width:100%;">
            <option value="">-- اختر --</option>
            <?php foreach ($groups as $grp): ?>
              <option value="<?= $grp['id'] ?>">
                <?= htmlspecialchars($grp['group_name'], ENT_QUOTES, 'UTF-8') ?>
              </option>
            <?php endforeach; ?>
          </select>

          <label>إلى المجموعة:</label><br>
          <select name="move_device_to_group" required style="margin-bottom:8px; width:100%;">
            <option value="">-- اختر --</option>
            <?php foreach ($groups as $grp): ?>
              <option value="<?= $grp['id'] ?>">
                <?= htmlspecialchars($grp['group_name'], ENT_QUOTES, 'UTF-8') ?>
              </option>
            <?php endforeach; ?>
          </select>

          <label>اختر الجهاز:</label><br>
          <!-- كذلك هنا استخدام allDevices -->
          <select name="device_id_to_move" required style="margin-bottom:8px; width:100%;">
            <option value="">-- اختر الجهاز --</option>
            <?php foreach ($allDevices as $dev): ?>
              <option value="<?= htmlspecialchars($dev['device_id'], ENT_QUOTES, 'UTF-8') ?>">
                <?= htmlspecialchars($dev['device_name'], ENT_QUOTES, 'UTF-8') ?>
              </option>
            <?php endforeach; ?>
          </select>

          <button type="submit" name="move_device">نقل الجهاز</button>
        </form>
      <?php endif; ?>
    </div>

    <!-- المحتوى الرئيسي -->
    <div class="main-content">
      <div class="controls">
        <label id="lblChooseDevice">🔽 اختر الجهاز:</label>
        <!-- هذا الصندوق يعرض كل الأجهزة -->
        <select id="deviceSelect">
          <option value="">جميع الأجهزة</option>
          <?php foreach ($allDevices as $device): ?>
            <option value="<?= htmlspecialchars($device['device_id'], ENT_QUOTES, 'UTF-8') ?>">
              <?= htmlspecialchars($device['device_name'], ENT_QUOTES, 'UTF-8') ?>
            </option>
          <?php endforeach; ?>
        </select>

        <label id="lblDate">📅 اختر التاريخ:</label>
        <input type="date" id="dateSelect">

        <label id="lblTimeFrom">⏰ من (12h):</label>
        <input type="text" id="timeFrom" placeholder="مثال: 01:00 PM">

        <label id="lblTimeTo">⏰ إلى (12h):</label>
        <input type="text" id="timeTo" placeholder="مثال: 05:30 PM">

        <button onclick="fetchLocations()" id="btnSearch">🔍 بحث</button>
        <button onclick="resetFilters()"  id="btnReset">🔄 إعادة تعيين</button>
        <button onclick="fetchLatestLocation()" id="btnLatestLoc">📡 آخر موقع مباشر</button>
        <button onclick="shareLocation()"       id="btnShare">🔗 مشاركة الموقع</button>

        <label id="lblAnimSpeed">⚡ سرعة الحركة (ملّي ثانية):</label>
        <input type="number" id="animationSpeed" value="1000" min="100" style="width: 120px;"/>

<button onclick="playAnimation()" id="btnPlay">▶️ تشغيل الحركة</button>
<button onclick="stopAnimation()" id="btnStop">⏹ إيقاف الحركة</button>
<button onclick="resetAnimationToStart()" id="btnResetAnimation">↩️ رجوع إلى البداية</button>

        <!-- اختيار نمط الخريطة -->
        <select id="mapStyleSelect" onchange="changeMapStyle()">
          <option value="outdoors">خريطة التضاريس</option>
          <option value="satellite">خريطة القمر الصناعي</option>
          <option value="dark">خريطة داكنة</option>
          <option value="traffic">خريطة الحركة المرورية</option>
        </select>
      </div>

      <div id="selectedDeviceContainer">
        <strong id="lblSelectedDeviceName">اسم الجهاز المختار:</strong> 
        <span id="selectedDeviceName">لا يوجد جهاز محدد</span>
      </div>

      <div id="map"></div>

      <div class="logs">
        <h2 id="logsTitle">سجلات التوقف (5 دقائق فأكثر)</h2>
        <div class="logs-controls">
          <button onclick="downloadLogsXLSX()" id="btnDownloadExcel">📥 Excel (XLSX)</button>
          <label id="lblLogsSearch">بحث في السجلات:</label>
          <input type="text" id="logsSearchInput" oninput="filterLogsTable()">
          <button onclick="resetLogsFilter()" id="btnResetLogsFilter">إعادة الكل</button>
        </div>

        <table id="logsTable">
          <thead>
            <tr>
              <th class="sortable" onclick="sortLogsTable(0)" id="thPlaceName">اسم المكان</th>
              <th class="sortable" onclick="sortLogsTable(1)" id="thEntryTime">وقت الدخول</th>
              <th class="sortable" onclick="sortLogsTable(2)" id="thExitTime">وقت الخروج</th>
              <th id="thStatus">الحالة</th>
              <th id="thBattery">البطارية</th>
            </tr>
          </thead>
          <tbody></tbody>
        </table>
      </div>

      <div id="analytics"></div>
    </div>
  </div>

  <!-- Toast notifications -->
  <div id="toastContainer"></div>


  <script>
    //************************************************
    //                    CONSTANTS
    //************************************************
    const accessToken = "pk.eyJ1IjoiaWJyYWhpbWNvbSIsImEiOiJja2YxZnB6OWowc2xxMnFxZGxya2ppdmZxIn0.keECYQuu1Bs0JeWDP8dSFQ";

    //************************************************
    //       MAP & LAYERS (يدعم 4 أنماط: تضاريس، قمر صناعي، داكن، حركة مرورية)
    //************************************************
    const outdoorsLayer = L.tileLayer(
      `https://api.mapbox.com/styles/v1/mapbox/outdoors-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
      {
        maxZoom: 20, 
        tileSize: 512, 
        zoomOffset: -1,
        attribution: 'Map data &copy; OpenStreetMap & Mapbox'
      }
    );
    const satelliteLayer = L.tileLayer(
      `https://api.mapbox.com/styles/v1/mapbox/satellite-streets-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
      {
        maxZoom: 20, 
        tileSize: 512, 
        zoomOffset: -1,
        attribution: 'Map data &copy; OpenStreetMap & Mapbox'
      }
    );
    const darkLayer = L.tileLayer(
      `https://api.mapbox.com/styles/v1/mapbox/dark-v10/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
      {
        maxZoom: 20, 
        tileSize: 512, 
        zoomOffset: -1,
        attribution: 'Map data &copy; OpenStreetMap & Mapbox'
      }
    );
    const trafficLayer = L.tileLayer(
      `https://api.mapbox.com/styles/v1/mapbox/traffic-day-v2/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
      {
        maxZoom: 20,
        tileSize: 512,
        zoomOffset: -1,
        attribution: 'Map data &copy; OpenStreetMap & Mapbox'
      }
    );

    // خريطة Leaflet
    var currentLayer = outdoorsLayer; // الافتراضية
    var map = L.map('map', {
      center: [30, 0],
      zoom: 2,
      layers: [currentLayer]
    });

    //---------------------------------------------------
    // مجموعة متغيرات عامة
    //---------------------------------------------------
    var markers  = [];
    var circles  = [];
    var polylines = [];
    var logs     = [];
    var markerCounter = 1;
    var latestMarker  = null;
    var latestLocationTimer = null;
    var lastTimestamp = null;

    // للأنيميشن
    var animationMarker = null;
    var animationTimeout = null;
    var currentAnimationIndex = 0;
    var routeCoordinates = [];
    var analyticsGroups  = [];

    // أيقونات
    var parkIcon = L.icon({
      iconUrl: 'https://almanara.space/gps_tracker/uploads/image.png',
      iconSize: [32, 32], 
      iconAnchor: [16, 32], 
      popupAnchor: [0, -32]
    });
    var hiddenIcon = L.icon({
      iconUrl: 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=',
      iconSize: [1, 1], 
      iconAnchor: [0, 0], 
      popupAnchor: [0, 0]
    });
    var startIcon = L.icon({
      iconUrl: 'https://almanara.space/gps_tracker/uploads/start.png',
      iconSize: [60, 60], 
      iconAnchor: [16, 32]
    });
    var endIcon = L.icon({
      iconUrl: 'https://almanara.space/gps_tracker/uploads/end.png',
      iconSize: [60, 60], 
      iconAnchor: [16, 32]
    });
    var latestIcon = L.icon({
      iconUrl: 'https://cdn-icons-png.flaticon.com/512/684/684908.png',
      iconSize: [32, 32], 
      iconAnchor: [16, 32]
    });

    //*********************************************
    //   دوال UI عامة (Toast رسائل)
    //*********************************************
    function showToast(message) {
      var container = document.getElementById("toastContainer");
      var toast = document.createElement("div");
      toast.className = "toast";
      toast.textContent = message;
      container.appendChild(toast);
      setTimeout(() => {
        if (container.contains(toast)) {
          container.removeChild(toast);
        }
      }, 4000);
    }

    //----------------------------------------------------
    //   تغيير طبقة الخريطة (Map Style)
    //----------------------------------------------------
    function changeMapStyle() {
      var styleSelect = document.getElementById("mapStyleSelect");
      var styleValue  = styleSelect.value;
      // حفظ التفضيل في localStorage
      localStorage.setItem('preferredMapStyle', styleValue);

      map.removeLayer(currentLayer);
      if (styleValue === "satellite") {
        currentLayer = satelliteLayer;
      } else if (styleValue === "dark") {
        currentLayer = darkLayer;
      } else if (styleValue === "traffic") {
        currentLayer = trafficLayer;
      } else {
        currentLayer = outdoorsLayer;
      }
      map.addLayer(currentLayer);
    }

    // استعادة تفضيلات المستخدم من localStorage
    window.onload = function() {
      var savedStyle = localStorage.getItem('preferredMapStyle');
      if (savedStyle) {
        document.getElementById("mapStyleSelect").value = savedStyle;
        changeMapStyle();
      }
    }

    //----------------------------------------------------
    // دوال تنظيف الخريطة والبيانات
    //----------------------------------------------------
    function clearMap() {
      markers.forEach(m => map.removeLayer(m));
      markers = [];
      circles.forEach(c => map.removeLayer(c));
      circles = [];
      polylines.forEach(p => map.removeLayer(p));
      polylines = [];
      if (latestMarker) {
        map.removeLayer(latestMarker);
        latestMarker = null;
      }
      if (animationMarker) {
        map.removeLayer(animationMarker);
        animationMarker = null;
      }
      if (latestLocationTimer) {
        clearInterval(latestLocationTimer);
        latestLocationTimer = null;
      }
      if (animationTimeout) {
        clearTimeout(animationTimeout);
        animationTimeout = null;
      }
      logs = [];
      markerCounter = 1;
      routeCoordinates = [];
      analyticsGroups  = [];
      lastTimestamp    = null;

      updateLogsTable(); 
      updateAnalytics(); 
    }

    function resetFilters() {
      document.getElementById("deviceSelect").value = "";
      document.getElementById("dateSelect").value   = "";
      document.getElementById("timeFrom").value     = "";
      document.getElementById("timeTo").value       = "";
      document.getElementById("selectedDeviceName").textContent = "لا يوجد جهاز محدد";
      clearMap();
      showToast("تمت إعادة تعيين الفلاتر");
    }

    //----------------------------------------------------
    // دوال جلب البيانات
    //----------------------------------------------------
    function fetchLocations() {
      clearMap();
      var deviceSel = document.getElementById("deviceSelect");
      var deviceId  = deviceSel.value;
      var deviceName = (deviceSel.selectedIndex >= 0)
                       ? deviceSel.options[deviceSel.selectedIndex].text
                       : "غير محدد";
      document.getElementById("selectedDeviceName").textContent = deviceName;

      var date     = document.getElementById("dateSelect").value;
      var timeFrom = convertTo24Hour(document.getElementById("timeFrom").value.trim());
      var timeTo   = convertTo24Hour(document.getElementById("timeTo").value.trim());

      var url = "get_locations.php?";
      if (deviceId) url += "device_id=" + encodeURIComponent(deviceId) + "&";
      if (date)     url += "date=" + encodeURIComponent(date) + "&";
      if (timeFrom) url += "time_from=" + encodeURIComponent(timeFrom) + "&";
      if (timeTo)   url += "time_to=" + encodeURIComponent(timeTo) + "&";
      url = url.slice(0, -1);

      fetch(url)
        .then(res => res.json())
        .then(data => {
          if (Array.isArray(data) && data.length > 0) {
            showToast("تم جلب " + data.length + " نقطة");

            // ترتيب زمني تصاعدي
            let sortedAsc = data.slice().sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
            let startLoc = sortedAsc[0];
            let endLoc   = sortedAsc[sortedAsc.length - 1];

            // أيقونة البداية
            let startLat = parseFloat(startLoc.latitude);
            let startLng = parseFloat(startLoc.longitude);
            if (!isNaN(startLat) && !isNaN(startLng)) {
              getPlaceName(startLat, startLng).then(place => {
                let markerStart = L.marker([startLat, startLng], { icon: startIcon })
                  .addTo(map)
                  .bindPopup(`
                    <b>بداية المسار</b><br>
                    ${formatTo12Hour(startLoc.timestamp)}<br>
                    بطارية: ${startLoc.battery ? startLoc.battery + '%' : 'غير معروفة'}<br>
                    المكان: ${place}<br>
                    <a href="https://www.google.com/maps/dir/?api=1&destination=${startLat},${startLng}" target="_blank">خرائط جوجل</a> |
                    <a href="https://waze.com/ul?ll=${startLat},${startLng}&navigate=yes" target="_blank">Waze</a>
                  `);
                markers.push(markerStart);
              });
            }

            // أيقونة النهاية
            let endLat = parseFloat(endLoc.latitude);
            let endLng = parseFloat(endLoc.longitude);
            if (!isNaN(endLat) && !isNaN(endLng)) {
              getPlaceName(endLat, endLng).then(place => {
                let markerEnd = L.marker([endLat, endLng], { icon: endIcon })
                  .addTo(map)
                  .bindPopup(`
                    <b>نهاية المسار</b><br>
                    ${formatTo12Hour(endLoc.timestamp)}<br>
                    بطارية: ${endLoc.battery ? endLoc.battery + '%' : 'غير معروفة'}<br>
                    المكان: ${place}<br>
                    <a href="https://www.google.com/maps/dir/?api=1&destination=${endLat},${endLng}" target="_blank">خرائط جوجل</a> |
                    <a href="https://waze.com/ul?ll=${endLat},${endLng}&navigate=yes" target="_blank">Waze</a>
                  `);
                markers.push(markerEnd);
              });
            }

            // تجميع النقاط (بارك/تحرك)
            let grouped = groupLocations(data, 30); // نصف قطر 30م
            grouped.sort((a, b) => new Date(a.entryTime) - new Date(b.entryTime));
            analyticsGroups = grouped;

            var routePoints = [];
            grouped.forEach(g => {
              var lat = g.center[0];
              var lng = g.center[1];
              if (!isNaN(lat) && !isNaN(lng)) {
                var batteryInfo = "";
                if (g.minBattery !== null && g.maxBattery !== null) {
                  batteryInfo = (g.minBattery === g.maxBattery)
                    ? `بطارية: ${g.minBattery}%<br>` 
                    : `بطارية: ${g.minBattery}% → ${g.maxBattery}%<br>`;
                }
                var icon = (g.status === "بارك توقف") ? parkIcon : hiddenIcon;
                var mk = L.marker([lat, lng], { icon: icon })
                  .addTo(map)
                  .bindPopup(`
                    <b>النقطة ${markerCounter}</b><br>
                    وقت الدخول: ${formatTo12Hour(g.entryTime)}<br>
                    وقت الخروج: ${formatTo12Hour(g.exitTime)}<br>
                    ${batteryInfo}
                    الحالة: ${g.status}<br>
                    <a href="https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}" target="_blank">خرائط جوجل</a> |
                    <a href="https://waze.com/ul?ll=${lat},${lng}&navigate=yes" target="_blank">Waze</a>
                  `);
                markers.push(mk);
                markerCounter++;

                // للجدول
                getPlaceName(lat, lng).then(place => {
                  logs.push({
                    placeName: place,
                    entryTime: g.entryTime,
                    exitTime:  g.exitTime,
                    center:    [lat, lng],
                    status:    g.status,
                    batteryRange: (g.minBattery !== null && g.maxBattery !== null)
                                 ? (g.minBattery === g.maxBattery ? g.minBattery + '%' : g.minBattery + '% - ' + g.maxBattery + '%')
                                 : ''
                  });
                  updateLogsTable();
                });

                routePoints.push([lat, lng]);
                routeCoordinates.push({
                  lat: lat, 
                  lng: lng, 
                  time: g.entryTime,
                  minBattery: g.minBattery,
                  maxBattery: g.maxBattery
                });
              }
            });

            // رسم المسار
            if (routePoints.length > 1) {
              var polyline = L.polyline(routePoints, { color: 'blue', weight: 5 }).addTo(map);
              polylines.push(polyline);
            }
            if (routePoints.length > 0) {
              var bounds = new L.LatLngBounds(routePoints);
              map.fitBounds(bounds);
            }

            updateAnalytics();
          } else {
            showToast("لا توجد بيانات ضمن الشروط");
          }
        })
        .catch(err => console.error(err));
    }

    // جلب آخر موقع مباشر
    function fetchLatestLocation() {
      var deviceId = document.getElementById("deviceSelect").value;
      if (!deviceId) {
        showToast("⚠️ يرجى اختيار جهاز لجلب أحدث موقع");
        return;
      }

      if (latestLocationTimer) {
        clearInterval(latestLocationTimer);
      }

      function getLatest() {
        fetch(`get_latest_location.php?device_id=${deviceId}&t=${Date.now()}`)
          .then(res => res.json())
          .then(data => {
            if (data.status === "success") {
              var lat       = parseFloat(data.data.latitude);
              var lng       = parseFloat(data.data.longitude);
              var timestamp = data.data.timestamp;
              var battery   = data.data.battery;
              if (!isNaN(lat) && !isNaN(lng)) {
                if (latestMarker) {
                  map.removeLayer(latestMarker);
                }
                latestMarker = L.marker([lat, lng], { icon: latestIcon })
                  .addTo(map)
                  .bindPopup(`
                    <b>📡 آخر موقع</b><br>
                    الجهاز: ${data.data.device_name}<br>
                    الوقت: ${formatTo12Hour(timestamp)}<br>
                    البطارية: ${battery ? battery + '%' : 'غير معروفة'}<br>
                    <a href="https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}" target="_blank">خرائط جوجل</a> |
                    <a href="https://waze.com/ul?ll=${lat},${lng}&navigate=yes" target="_blank">Waze</a>
                  `).openPopup();

                var circle = L.circle([lat, lng], {
                  radius: 30, 
                  color: 'blue', 
                  fillColor: '#00f', 
                  fillOpacity: 0.5
                }).addTo(map);
                circles.push(circle);
                map.setView([lat, lng], 15);

                lastTimestamp = timestamp;
              }
            }
          })
          .catch(err => console.error("خطأ في جلب الموقع:", err));
      }

      getLatest();
      latestLocationTimer = setInterval(getLatest, 10000);
    }

    //----------------------------------------------------
    //   دوال الحركة (أنيميشن)
    //----------------------------------------------------
// عند تشغيل الحركة: لا نقوم بإعادة تعيين currentAnimationIndex إذا كانت الحركة متوقفة في وسط المسار
function playAnimation() {
  if (!routeCoordinates.length) {
    showToast("❌ لا يوجد مسار لتشغيل الحركة.");
    return;
  }
  // إلغاء أي تايمر سابق
  if (animationTimeout) {
    clearTimeout(animationTimeout);
    animationTimeout = null;
  }
  // استئناف الحركة من currentAnimationIndex الحالي (إذا كان قد وصل إلى نهاية المسار يمكن إعادة تعيينه)
  if (currentAnimationIndex >= routeCoordinates.length) {
    currentAnimationIndex = 0;
  }
  animateStep();
}

// إيقاف الحركة ببساطة بإيقاف التايمر
function stopAnimation() {
  if (animationTimeout) {
    clearTimeout(animationTimeout);
    animationTimeout = null;
  }
}

// دالة الحركة خطوة بخطوة كما هي مع تحديث currentAnimationIndex
function animateStep() {
  if (currentAnimationIndex >= routeCoordinates.length) {
    return;
  }
  var pt = routeCoordinates[currentAnimationIndex];
  var minB = pt.minBattery;
  var maxB = pt.maxBattery;
  var batteryText = "";
  if (minB !== null && maxB !== null) {
    batteryText = (minB === maxB) ? (minB + "%") : (minB + "% → " + maxB + "%");
  }

  if (!animationMarker) {
    animationMarker = L.marker([pt.lat, pt.lng], { icon: createAnimationIcon(pt.time, batteryText) })
      .addTo(map);
  } else {
    animationMarker.setLatLng([pt.lat, pt.lng]);
    animationMarker.setIcon(createAnimationIcon(pt.time, batteryText));
  }
  map.panTo([pt.lat, pt.lng]);
  currentAnimationIndex++;

  var speed = parseInt(document.getElementById("animationSpeed").value);
  if (isNaN(speed) || speed < 100) speed = 1000;
  if (currentAnimationIndex < routeCoordinates.length) {
    animationTimeout = setTimeout(animateStep, speed);
  }
}

// دالة لإعادة الحركة إلى البداية
function resetAnimationToStart() {
  if (animationTimeout) {
    clearTimeout(animationTimeout);
    animationTimeout = null;
  }
  currentAnimationIndex = 0;
  if (animationMarker) {
    map.removeLayer(animationMarker);
    animationMarker = null;
  }
  if (routeCoordinates.length > 0) {
    var pt = routeCoordinates[0];
    // إعادة رسم المؤشر في نقطة البداية
    animationMarker = L.marker([pt.lat, pt.lng], { icon: createAnimationIcon(pt.time, (pt.minBattery && pt.maxBattery) ? ((pt.minBattery === pt.maxBattery) ? (pt.minBattery + "%") : (pt.minBattery + "% → " + pt.maxBattery + "%")) : "") })
      .addTo(map);
    map.panTo([pt.lat, pt.lng]);
    showToast("تم العودة إلى بداية الحركة");
  }
}


    function createAnimationIcon(time, battery) {
      var batteryStr = battery ? `<br>بطارية: ${battery}` : "";
      return L.divIcon({
        html: `<div style="position: relative; background: rgba(0,0,0,0.7);
                         padding: 4px 8px; border-radius: 6px; border: 2px solid #fff;
                         display: inline-flex; align-items: center;">
                 <img src="https://cdn-icons-png.flaticon.com/512/684/684908.png"
                      style="width:40px;height:40px;">
                 <span style="margin-left: 8px; color: #fff; font-size: 14px;">
                   ${formatTo12Hour(time)}${batteryStr}
                 </span>
               </div>`,
        iconSize: [60, 50],
        className: ''
      });
    }

    //----------------------------------------------------
    //    دوال تحليل البيانات
    //----------------------------------------------------
    function updateAnalytics() {
      var div = document.getElementById("analytics");
      // حساب المسافة الإجمالية
      var totalDistance = 0;
      for (var i = 1; i < routeCoordinates.length; i++) {
        totalDistance += getDistance(
          routeCoordinates[i-1].lat, routeCoordinates[i-1].lng,
          routeCoordinates[i].lat,   routeCoordinates[i].lng
        );
      }
      totalDistance /= 1000; // كم
      var movingTime = 0, stoppedTime = 0;
      analyticsGroups.forEach(g => {
        var t1 = new Date(g.entryTime).getTime();
        var t2 = new Date(g.exitTime).getTime();
        var duration = (t2 - t1) / 1000;
        if (g.status === "تحرك") {
          movingTime += duration;
        } else {
          stoppedTime += duration;
        }
      });
      var movingMins  = movingTime / 60;
      var stoppedMins = stoppedTime / 60;
      var avgSpeed    = (movingTime > 0)
                        ? totalDistance / (movingTime / 3600)
                        : 0;

      div.innerHTML = `
        <h2 id="analyticsTitle">تحليل البيانات</h2>
        <p id="pDistance">المسافة المقطوعة: ${totalDistance.toFixed(2)} كم</p>
        <p id="pMovingTime">وقت التحرك: ${movingMins.toFixed(2)} دقيقة</p>
        <p id="pStoppedTime">وقت التوقف: ${stoppedMins.toFixed(2)} دقيقة</p>
        <p id="pAvgSpeed">متوسط السرعة: ${avgSpeed.toFixed(2)} كم/س</p>
        <canvas id="timeChart"></canvas>
        <canvas id="speedChart"></canvas>
      `;

      // مخطط Doughnut لوقت التحرك والتوقف
      var ctx1 = document.getElementById("timeChart").getContext("2d");
      new Chart(ctx1, {
        type: 'doughnut',
        data: {
          labels: ['تحرك (دقيقة)', 'توقف (دقيقة)'],
          datasets: [{
            data: [movingMins, stoppedMins]
          }]
        },
        options: { responsive: true, plugins: { legend: { position: 'bottom' } } }
      });

      // مخطط السرعة
      var speeds = [];
      var speedLabels = [];
      for (var i = 1; i < routeCoordinates.length; i++) {
        var d = getDistance(
          routeCoordinates[i-1].lat, routeCoordinates[i-1].lng,
          routeCoordinates[i].lat,   routeCoordinates[i].lng
        );
        var tDiff = (new Date(routeCoordinates[i].time).getTime() -
                     new Date(routeCoordinates[i-1].time).getTime()) / 3600000;
        var sp = (tDiff > 0) ? (d/1000)/tDiff : 0;
        speeds.push(sp.toFixed(2));
        speedLabels.push("نقطة " + i);
      }
      var ctx2 = document.getElementById("speedChart").getContext("2d");
      new Chart(ctx2, {
        type: 'line',
        data: {
          labels: speedLabels,
          datasets: [{
            label: 'السرعة (كم/س)',
            data: speeds,
            fill: false,
            tension: 0.1
          }]
        },
        options: { responsive: true, plugins: { legend: { position: 'bottom' } } }
      });
    }

    //----------------------------------------------------
    //  دوال مشاركة الموقع
    //----------------------------------------------------
    function shareLocation() {
      var deviceSel = document.getElementById("deviceSelect");
      var deviceId  = deviceSel.value;
      if (!deviceId) {
        showToast("❌ اختر جهازاً أولاً.");
        return;
      }
      if (!latestMarker) {
        showToast("❌ لا يوجد موقع حديث للمشاركة. اضغط (آخر موقع مباشر).");
        return;
      }
      var deviceName = deviceSel.options[deviceSel.selectedIndex].text;
      var expires = Date.now() + 3600000; // ساعة
      var shareUrl = window.location.origin + window.location.pathname
                   + "?share=1"
                   + "&device_id="   + encodeURIComponent(deviceId)
                   + "&device_name=" + encodeURIComponent(deviceName)
                   + "&expires="     + expires;
      window.open(shareUrl, "_blank");
    }

    //----------------------------------------------------
    //   دوال جدول السجلات (التوقف)
    //----------------------------------------------------
    function updateLogsTable() {
      var tbody = document.getElementById("logsTable").getElementsByTagName("tbody")[0];
      tbody.innerHTML = "";

      // فقط توقفات >= 5 دقائق
      var filtered = logs.filter(log => {
        var diff = (new Date(log.exitTime) - new Date(log.entryTime)) / 60000;
        return diff >= 5;
      });
      // ترتيب تصاعدي زمني
      filtered.sort((a, b) => new Date(a.entryTime) - new Date(b.entryTime));

      filtered.forEach(log => {
        var row = tbody.insertRow();
        row.insertCell(0).innerHTML = `<a href="#" onclick="focusOnLocation([${log.center}])">${log.placeName}</a>`;
        row.insertCell(1).textContent = formatTo12Hour(log.entryTime);
        row.insertCell(2).textContent = formatTo12Hour(log.exitTime);
        row.insertCell(3).textContent = log.status;
        row.insertCell(4).textContent = log.batteryRange;
      });
    }

    // البحث في سجل التوقف
    function filterLogsTable() {
      var input = document.getElementById("logsSearchInput").value.toUpperCase();
      var table = document.getElementById("logsTable");
      var trs   = table.getElementsByTagName("tr");
      for (var i = 1; i < trs.length; i++) {
        var tds = trs[i].getElementsByTagName("td");
        var rowText = "";
        for (var j = 0; j < tds.length; j++) {
          rowText += tds[j].textContent.toUpperCase() + " ";
        }
        trs[i].style.display = (rowText.indexOf(input) > -1) ? "" : "none";
      }
    }

    function resetLogsFilter() {
      document.getElementById("logsSearchInput").value = "";
      filterLogsTable();
    }

    // فرز الجدول بالعمود index
    let logsSortDirection = true; // تصاعدي أو تنازلي
    function sortLogsTable(index) {
      var table = document.getElementById("logsTable");
      var tbody = table.tBodies[0];
      var rows  = Array.from(tbody.querySelectorAll("tr"));
      logsSortDirection = !logsSortDirection;

      rows.sort((rowA, rowB) => {
        var cellA = rowA.cells[index].textContent.trim();
        var cellB = rowB.cells[index].textContent.trim();
        if (index === 1 || index === 2) {
          // أعمدة الوقت (دخول/خروج)
          var dateA = new Date(cellA);
          var dateB = new Date(cellB);
          return logsSortDirection ? (dateA - dateB) : (dateB - dateA);
        } else {
          // نصوص عادية
          return logsSortDirection
            ? cellA.localeCompare(cellB)
            : cellB.localeCompare(cellA);
        }
      });
      rows.forEach(row => tbody.appendChild(row));
    }

    function focusOnLocation(center) {
      map.setView(center, 15);
    }

    //----------------------------------------------------
    //   دوال تجميع المواقع (بارك/تحرك)
    //----------------------------------------------------
    function groupLocations(locations, radius) {
      if (!locations.length) return [];
      locations.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));

      var groups = [];
      var first  = locations[0];
      var currentGroup = {
        points: [first],
        center: [parseFloat(first.latitude), parseFloat(first.longitude)],
        entryTime: first.timestamp,
        exitTime:  first.timestamp,
        batteryLevels: [ parseInt(first.battery) || null ]
      };
      var prevTime = new Date(first.timestamp);

      for (var i = 1; i < locations.length; i++) {
        var loc = locations[i];
        var currTime = new Date(loc.timestamp);
        var timeDiff = (currTime - prevTime) / 60000; // دقائق
        var lastPt   = currentGroup.points[currentGroup.points.length - 1];
        var lastLat  = parseFloat(lastPt.latitude);
        var lastLng  = parseFloat(lastPt.longitude);
        var currLat  = parseFloat(loc.latitude);
        var currLng  = parseFloat(loc.longitude);
        var dist     = getDistance(lastLat, lastLng, currLat, currLng);
        var locBattery = parseInt(loc.battery) || null;

        if (timeDiff <= 5 && dist <= radius) {
          currentGroup.points.push(loc);
          currentGroup.exitTime = loc.timestamp;
          currentGroup.batteryLevels.push(locBattery);
        } else {
          groups.push(currentGroup);
          currentGroup = {
            points: [loc],
            center: [currLat, currLng],
            entryTime: loc.timestamp,
            exitTime: loc.timestamp,
            batteryLevels: [locBattery]
          };
        }
        prevTime = currTime;
      }
      groups.push(currentGroup);

      // تحديد الحالة + حساب البطاريات
      groups.forEach(g => {
        var diffMin = (new Date(g.exitTime) - new Date(g.entryTime)) / 60000;
        g.status = (diffMin >= 5) ? "بارك توقف" : "تحرك";
        var validB = g.batteryLevels.filter(b => b !== null && !isNaN(b));
        g.minBattery = validB.length > 0 ? Math.min(...validB) : null;
        g.maxBattery = validB.length > 0 ? Math.max(...validB) : null;
      });
      return groups;
    }

    //----------------------------------------------------
    //   دوال مساعدة
    //----------------------------------------------------
    function getDistance(lat1, lon1, lat2, lon2) {
      var R = 6371e3;
      var φ1 = lat1 * Math.PI / 180;
      var φ2 = lat2 * Math.PI / 180;
      var Δφ = (lat2 - lat1) * Math.PI / 180;
      var Δλ = (lon2 - lon1) * Math.PI / 180;
      var a = Math.sin(Δφ/2)*Math.sin(Δφ/2) + Math.cos(φ1)*Math.cos(φ2)*Math.sin(Δλ/2)*Math.sin(Δλ/2);
      var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
      return R * c;
    }

    function convertTo24Hour(timeStr) {
      if (!timeStr) return null;
      timeStr = timeStr.trim();

      // Regex to capture hours, minutes, and optional AM/PM
      // Allows 1 or 2 digits for hours/minutes, optional space before AM/PM
      const regex = /^(\d{1,2}):(\d{1,2})(?:\s*([AP]M))?$/i;
      const match = timeStr.match(regex);

      if (!match) {
        return null; // Format doesn't match HH:MM or HH:MM AM/PM
      }

      let hours = parseInt(match[1], 10);
      let minutes = parseInt(match[2], 10);
      let ampm = match[3] ? match[3].toUpperCase() : null;

      // Validate minutes
      if (minutes < 0 || minutes > 59) {
        return null; // Invalid minutes
      }

      if (ampm) {
        // If AM/PM is present, hours must be 1-12
        if (hours < 1 || hours > 12) {
          return null; // Invalid hours for AM/PM format
        }
        if (ampm === 'PM' && hours < 12) {
          hours += 12;
        } else if (ampm === 'AM' && hours === 12) { // 12 AM is 00 hours
          hours = 0;
        }
      } else {
        // No AM/PM, assume 24-hour format. Hours must be 0-23.
        if (hours < 0 || hours > 23) {
          return null; // Invalid hours for 24-hour format
        }
      }

      const hh = (hours < 10) ? '0' + String(hours) : String(hours);
      const mm = (minutes < 10) ? '0' + String(minutes) : String(minutes);
      return hh + ":" + mm + ":00";
    }

    function formatTo12Hour(dateStr) {
      if (!dateStr) return '';
      // Simply return the original date string to display it without any changes
      return dateStr;
    }

    function getPlaceName(lat, lng) {
      var url = `https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${accessToken}`;
      return fetch(url)
        .then(res => res.json())
        .then(data => {
          if (data.features && data.features.length > 0) {
            return data.features[0].place_name;
          }
          return "مكان غير معروف";
        })
        .catch(_ => "مكان غير معروف");
    }

    //----------------------------------------------------
    //    تنزيل سجلات التوقف بصيغة XLSX
    //----------------------------------------------------
    function downloadLogsXLSX() {
      var filtered = logs.filter(log => {
        var diff = (new Date(log.exitTime) - new Date(log.entryTime)) / 60000;
        return diff >= 5;
      });
      var ws_data = [
        ["اسم المكان", "وقت الدخول", "وقت الخروج", "الحالة", "البطارية"]
      ];
      filtered.forEach(l => {
        ws_data.push([
          l.placeName,
          formatTo12Hour(l.entryTime),
          formatTo12Hour(l.exitTime),
          l.status,
          l.batteryRange
        ]);
      });

      var wb = XLSX.utils.book_new();
      var ws = XLSX.utils.aoa_to_sheet(ws_data);
      ws['!cols'] = [
        { wch: 30 }, { wch: 25 }, { wch: 25 }, { wch: 20 }, { wch: 15 }
      ];
      XLSX.utils.book_append_sheet(wb, ws, "سجلات");
      XLSX.writeFile(wb, "logs.xlsx");
      showToast("تم تنزيل ملف Excel بنجاح");
    }

    //----------------------------------------------------
    //   تصفية الأجهزة + المجموعات في القائمة الجانبية
    //----------------------------------------------------
    function filterDevices() {
      var input = document.getElementById("searchDevice").value.toUpperCase();
      var ul    = document.getElementById("deviceList");
      var lis   = ul.getElementsByTagName("li");
      for (var i = 0; i < lis.length; i++) {
        var txt = lis[i].textContent || lis[i].innerText;
        lis[i].style.display = (txt.toUpperCase().indexOf(input) > -1) ? "" : "none";
      }
    }

    // عند الضغط على جهاز في القائمة (الأجهزة غير المضافة لأي مجموعة)
    document.getElementById("deviceList").addEventListener("click", function(e) {
      if (e.target.tagName === "LI" && e.target.hasAttribute("data-device-id")) {
        var deviceId = e.target.getAttribute("data-device-id");
        document.getElementById("deviceSelect").value = deviceId;
        document.getElementById("selectedDeviceName").textContent = e.target.textContent;
        fetchLatestLocation();
      }
    });

    // اختيار جهاز من قائمة مجموعة
    function selectDeviceFromGroup(deviceId, deviceName) {
      document.getElementById("deviceSelect").value = deviceId;
      document.getElementById("selectedDeviceName").textContent = deviceName;
      fetchLatestLocation();
    }

    // توسيع/طي أجهزة المجموعة
    function toggleGroupDevices(groupId) {
      var el = document.getElementById("group-devices-" + groupId);
      if (!el) return;
      if (el.style.display === "none" || el.style.display === "") {
        el.style.display = "block";
      } else {
        el.style.display = "none";
      }
    }

    //----------------------------------------------------
    //   تبديل اللغة (بسيط)
    //----------------------------------------------------
    var currentLang = "ar"; // افتراضياً
    function toggleLanguage() {
      if (currentLang === "ar") {
        currentLang = "en";
        document.getElementById("langLabel").textContent         = "العربية";
        document.getElementById("appTitle").textContent          = "Almanara - Tracking System";
        document.getElementById("deviceListTitle").textContent   = "Devices & Groups";
        document.getElementById("searchDevice").placeholder      = "Search...";
        document.getElementById("lblChooseDevice").textContent   = "Choose Device:";
        document.getElementById("lblDate").textContent           = "Date:";
        document.getElementById("lblTimeFrom").textContent       = "From (12h):";
        document.getElementById("lblTimeTo").textContent         = "To (12h):";
        document.getElementById("btnSearch").textContent         = "Search";
        document.getElementById("btnReset").textContent          = "Reset";
        document.getElementById("btnLatestLoc").textContent      = "Live Location";
        document.getElementById("btnShare").textContent          = "Share";
        document.getElementById("lblAnimSpeed").textContent      = "Animation Speed (ms):";
        document.getElementById("btnPlay").textContent           = "Play";
        document.getElementById("btnStop").textContent           = "Stop";
        document.getElementById("lblSelectedDeviceName").textContent = "Selected Device:";
        document.getElementById("logsTitle").textContent         = "Stop Logs (5 mins+)";
        document.getElementById("btnDownloadExcel").textContent  = "Download Excel";
        document.getElementById("lblLogsSearch").textContent     = "Search in logs:";
        document.getElementById("btnResetLogsFilter").textContent= "Show All";
        document.getElementById("thPlaceName").textContent       = "Place Name";
        document.getElementById("thEntryTime").textContent       = "Entry Time";
        document.getElementById("thExitTime").textContent        = "Exit Time";
        document.getElementById("thStatus").textContent          = "Status";
        document.getElementById("thBattery").textContent         = "Battery";
        document.getElementById("analyticsTitle").textContent    = "Analytics";
        document.getElementById("pDistance").textContent         = "";
        document.getElementById("pMovingTime").textContent       = "";
        document.getElementById("pStoppedTime").textContent      = "";
        document.getElementById("pAvgSpeed").textContent         = "";
      } else {
        currentLang = "ar";
        location.reload(); // إعادة تحميل الصفحة لاستعادة النصوص بالعربية
      }
    }
  </script>
</body>
</html>
