const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const mysql = require('mysql2/promise');
require('dotenv').config();

const app = express();
const port = process.env.PORT || 3001;

app.use(cors());
app.use(bodyParser.json());

const dbConfig = {
    host: process.env.DB_HOST,
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_DATABASE,
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0
};

const pool = mysql.createPool(dbConfig);

// Test DB connection
(async () => {
    try {
        const connection = await pool.getConnection();
        console.log('Connected to the database!');
        connection.release();
    } catch (error) {
        console.error('Error connecting to the database:', error);
    }
})();

app.get('/', (req, res) => {
    res.send('Backend server is running!');
});

// Forms API
app.post('/api/forms', async (req, res) => {
    try {
        const { id, name, description, created_at, user_id, fields } = req.body;
        const connection = await pool.getConnection();
        await connection.beginTransaction();

        const [formResult] = await connection.execute(
            'INSERT INTO forms (id, name, description, created_at, user_id) VALUES (?, ?, ?, ?, ?)',
            [id, name, description, created_at, user_id]
        );

        for (const field of fields) {
            await connection.execute(
                'INSERT INTO form_fields (id, form_id, label, type, options_static, required, placeholder, accept, min_value, max_value, step_value, blocked_words, field_order, conditional_logic_source_field_id, conditional_logic_trigger_option_values, dynamic_options_depends_on_field_id, dynamic_options_map) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
                [field.id, id, field.label, field.type, JSON.stringify(field.options_static), field.required, field.placeholder, field.accept, field.min_value, field.max_value, field.step_value, JSON.stringify(field.blocked_words), field.field_order, field.conditional_logic_source_field_id, JSON.stringify(field.conditional_logic_trigger_option_values), field.dynamic_options_depends_on_field_id, JSON.stringify(field.dynamic_options_map)]
            );
        }

        await connection.commit();
        connection.release();
        res.status(201).send({ message: 'Form created successfully' });
    } catch (error) {
        console.error('Error creating form:', error);
        res.status(500).send({ message: 'Error creating form' });
    }
});

app.get('/api/forms/:id', async (req, res) => {
    try {
        const { id } = req.params;
        const connection = await pool.getConnection();
        const [forms] = await connection.execute('SELECT * FROM forms WHERE id = ?', [id]);
        if (forms.length === 0) {
            return res.status(404).send({ message: 'Form not found' });
        }
        const [fields] = await connection.execute('SELECT * FROM form_fields WHERE form_id = ? ORDER BY field_order', [id]);
        connection.release();
        const form = forms[0];
        form.fields = fields.map(field => ({
            ...field,
            options_static: JSON.parse(field.options_static || 'null'),
            blocked_words: JSON.parse(field.blocked_words || 'null'),
            conditional_logic_trigger_option_values: JSON.parse(field.conditional_logic_trigger_option_values || 'null'),
            dynamic_options_map: JSON.parse(field.dynamic_options_map || 'null'),
        }));
        res.send(form);
    } catch (error) {
        console.error('Error fetching form:', error);
        res.status(500).send({ message: 'Error fetching form' });
    }
});

// Form Submissions API
app.post('/api/forms/:id/submissions', async (req, res) => {
    try {
        const { id } = req.params;
        const { submission_id, latitude, longitude, user_id, data } = req.body;
        const connection = await pool.getConnection();
        await connection.execute(
            'INSERT INTO form_submissions (id, form_id, latitude, longitude, user_id, data) VALUES (?, ?, ?, ?, ?, ?)',
            [submission_id, id, latitude, longitude, user_id, JSON.stringify(data)]
        );
        connection.release();
        res.status(201).send({ message: 'Submission created successfully' });
    } catch (error) {
        console.error('Error creating submission:', error);
        res.status(500).send({ message: 'Error creating submission' });
    }
});

app.get('/api/forms/:id/submissions', async (req, res) => {
    try {
        const { id } = req.params;
        const connection = await pool.getConnection();
        const [submissions] = await connection.execute('SELECT * FROM form_submissions WHERE form_id = ?', [id]);
        connection.release();
        res.send(submissions.map(s => ({...s, data: JSON.parse(s.data)})));
    } catch (error) {
        console.error('Error fetching submissions:', error);
        res.status(500).send({ message: 'Error fetching submissions' });
    }
});

// Form Groups API
app.post('/api/form-groups', async (req, res) => {
    try {
        const { id, name, created_at, user_id, form_ids } = req.body;
        const connection = await pool.getConnection();
        await connection.beginTransaction();

        await connection.execute(
            'INSERT INTO form_groups (id, name, created_at, user_id) VALUES (?, ?, ?, ?)',
            [id, name, created_at, user_id]
        );

        for (const form_id of form_ids) {
            await connection.execute(
                'INSERT INTO form_group_forms (group_id, form_id) VALUES (?, ?)',
                [id, form_id]
            );
        }

        await connection.commit();
        connection.release();
        res.status(201).send({ message: 'Form group created successfully' });
    } catch (error) {
        console.error('Error creating form group:', error);
        res.status(500).send({ message: 'Error creating form group' });
    }
});

app.get('/api/form-groups/:id/submissions', async (req, res) => {
    try {
        const { id } = req.params;
        const connection = await pool.getConnection();
        const [group_forms] = await connection.execute('SELECT form_id FROM form_group_forms WHERE group_id = ?', [id]);
        const form_ids = group_forms.map(gf => gf.form_id);

        if (form_ids.length === 0) {
            return res.send([]);
        }

        const [submissions] = await connection.execute(`SELECT * FROM form_submissions WHERE form_id IN (${form_ids.map(() => '?').join(',')})`, form_ids);
        connection.release();
        res.send(submissions.map(s => ({...s, data: JSON.parse(s.data)})));
    } catch (error) {
        console.error('Error fetching group submissions:', error);
        res.status(500).send({ message: 'Error fetching group submissions' });
    }
});

app.listen(port, () => {
    console.log(`Server is running on port ${port}`);
});