
import React, { useState, useEffect, useMemo } from 'react';
import { FormFieldDefinition, FieldType, FileValue, FormSubmissionData } from '../types';

interface FormFieldInputProps {
  field: FormFieldDefinition;
  value: any;
  onChange: (value: any) => void;
  isDarkMode?: boolean;
  allFormData?: FormSubmissionData; // To access other field values for dynamic options
}

const FormFieldInput: React.FC<FormFieldInputProps> = ({ field, value, onChange, isDarkMode = true, allFormData }) => {
  const commonInputClasses = `w-full px-3 py-2 rounded-md shadow-sm focus:outline-none focus:ring-2 ${
    isDarkMode 
      ? 'bg-slate-700 border-slate-600 text-gray-200 focus:ring-sky-400 focus:border-sky-400' 
      : 'bg-white border-gray-300 text-gray-900 focus:ring-sky-500 focus:border-sky-500'
  } border`;

  const [rangeValue, setRangeValue] = useState(value);
  useEffect(() => {
    if (field.type === FieldType.RANGE) {
      setRangeValue(value ?? field.min ?? 0);
    }
  }, [value, field.type, field.min]);

  const currentOptions: string[] = useMemo(() => {
    if (field.dynamicOptionsConfig && allFormData) {
      const sourceFieldValue = allFormData[field.dynamicOptionsConfig.dependsOnFieldId];
      if (sourceFieldValue && typeof sourceFieldValue === 'string' && field.dynamicOptionsConfig.optionsMap[sourceFieldValue]) {
        return field.dynamicOptionsConfig.optionsMap[sourceFieldValue];
      }
      return []; // No source value or no mapping for current source value
    }
    return field.options || []; // Fallback to static options
  }, [field.options, field.dynamicOptionsConfig, allFormData]);


  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const target = e.target as HTMLInputElement;
    if (field.type === FieldType.CHECKBOX) {
      onChange(target.checked);
    } else if (field.type === FieldType.FILE) {
      if (target.files && target.files.length > 0) {
        const file = target.files[0];
        onChange({ name: file.name, type: file.type, size: file.size } as FileValue);
      } else {
        onChange(null);
      }
    } else if (field.type === FieldType.RANGE) {
        const numValue = parseFloat(target.value);
        setRangeValue(numValue);
        onChange(numValue);
    } else {
      onChange(target.value);
    }
  };
  
  const handleCheckboxGroupChange = (optionValue: string, checked: boolean) => {
    const currentValue = (Array.isArray(value) ? value : []) as string[];
    let newValue: string[];
    if (checked) {
      newValue = [...currentValue, optionValue];
    } else {
      newValue = currentValue.filter(v => v !== optionValue);
    }
    onChange(newValue);
  };

  const renderSelectOptions = () => (
    <>
      <option value="" disabled>{field.placeholder || 'Select an option'}</option>
      {currentOptions.map(option => (
        <option key={option} value={option}>{option}</option>
      ))}
    </>
  );

  const renderRadioGroupOptions = () => (
    currentOptions.map((option, optIndex) => (
      <div key={`${field.id}-opt-${optIndex}`} className="flex items-center">
        <input
          type="radio"
          id={`${field.id}-opt-${optIndex}`}
          name={field.id}
          value={option}
          checked={value === option}
          onChange={(e) => onChange(e.target.value)}
          required={field.required}
          className={`h-4 w-4 ${isDarkMode ? 'text-sky-400 bg-slate-600 border-slate-500 focus:ring-sky-400' : 'text-sky-600 border-gray-300 focus:ring-sky-500'} focus:ring-offset-0`}
        />
        <label htmlFor={`${field.id}-opt-${optIndex}`} className={`ml-2 text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
          {option}
        </label>
      </div>
    ))
  );

  const renderCheckboxGroupOptions = () => (
    currentOptions.map((option, optIndex) => (
      <div key={`${field.id}-opt-${optIndex}`} className="flex items-center">
        <input
          type="checkbox"
          id={`${field.id}-opt-${optIndex}`}
          name={`${field.id}-opt-${optIndex}`} // Name can be unique for checkbox in a group unlike radio
          value={option}
          checked={(Array.isArray(value) ? value : []).includes(option)}
          onChange={(e) => handleCheckboxGroupChange(option, e.target.checked)}
          className={`h-4 w-4 rounded ${isDarkMode ? 'bg-slate-700 border-slate-600 text-sky-400 focus:ring-sky-400' : 'text-sky-600 focus:ring-sky-500 border-gray-300'} focus:ring-offset-0`}
        />
        <label htmlFor={`${field.id}-opt-${optIndex}`} className={`ml-2 text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
          {option}
        </label>
      </div>
    ))
  );
  
  const isDynamicOptionsSourceUnselected = field.dynamicOptionsConfig && allFormData && !allFormData[field.dynamicOptionsConfig.dependsOnFieldId];
  const noDynamicOptionsAvailable = field.dynamicOptionsConfig && currentOptions.length === 0 && (allFormData && allFormData[field.dynamicOptionsConfig.dependsOnFieldId]);


  switch (field.type) {
    case FieldType.TEXT:
    case FieldType.EMAIL:
    case FieldType.DATE:
    case FieldType.TIME:
    case FieldType.DATETIME_LOCAL:
    case FieldType.TEL:
    case FieldType.URL:
    case FieldType.PASSWORD:
      return <input type={field.type} id={field.id} name={field.id} value={value || ''} onChange={handleChange} placeholder={field.placeholder} required={field.required} className={commonInputClasses}/>;
    case FieldType.NUMBER:
      return <input type={field.type} id={field.id} name={field.id} value={value || ''} onChange={handleChange} placeholder={field.placeholder} required={field.required} className={commonInputClasses} min={field.min} max={field.max} step={field.step}/>;
    case FieldType.TEXTAREA:
      return <textarea id={field.id} name={field.id} value={value || ''} onChange={handleChange} placeholder={field.placeholder} required={field.required} rows={4} className={commonInputClasses}/>;
    case FieldType.SELECT:
      return (
        <select id={field.id} name={field.id} value={value || ''} onChange={handleChange} required={field.required} className={commonInputClasses} disabled={isDynamicOptionsSourceUnselected || noDynamicOptionsAvailable}>
          {isDynamicOptionsSourceUnselected ? <option value="">اختر قيمة من الحقل المصدر أولاً</option> :
           noDynamicOptionsAvailable ? <option value="">لا توجد خيارات متاحة للاختيار الحالي</option> :
           renderSelectOptions()}
        </select>
      );
    case FieldType.CHECKBOX:
      return (<div className="flex items-center"><input type="checkbox" id={field.id} name={field.id} checked={!!value} onChange={handleChange} required={field.required} className={`h-5 w-5 rounded ${isDarkMode ? 'bg-slate-700 border-slate-600 text-sky-400 focus:ring-sky-400' : 'text-sky-600 focus:ring-sky-500 border-gray-300'} focus:ring-offset-0`}/><label htmlFor={field.id} className={`ml-2 text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{field.label} {field.required && '*'}</label></div>);
    case FieldType.RADIO_GROUP:
      return (
        <div className="space-y-2">
          {isDynamicOptionsSourceUnselected ? <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>اختر قيمة من الحقل المصدر أولاً لعرض الخيارات.</p> :
           noDynamicOptionsAvailable ? <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>لا توجد خيارات متاحة للاختيار الحالي.</p> :
           renderRadioGroupOptions()}
        </div>
      );
    case FieldType.CHECKBOX_GROUP:
      return (
        <div className="space-y-2">
          {isDynamicOptionsSourceUnselected ? <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>اختر قيمة من الحقل المصدر أولاً لعرض الخيارات.</p> :
           noDynamicOptionsAvailable ? <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>لا توجد خيارات متاحة للاختيار الحالي.</p> :
           renderCheckboxGroupOptions()}
        </div>
      );
    case FieldType.FILE:
      return (<div><input type="file" id={field.id} name={field.id} onChange={handleChange} required={field.required} className={`${commonInputClasses} file:mr-4 file:py-2 file:px-4 file:rounded-md file:border-0 file:text-sm file:font-semibold ${isDarkMode ? 'file:bg-sky-500 file:text-white hover:file:bg-sky-600' : 'file:bg-sky-100 file:text-sky-700 hover:file:bg-sky-200'}`} accept={field.accept}/>{value && typeof value === 'object' && value.name && (<p className={`mt-1 text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>Selected: {value.name} ({(value.size / 1024).toFixed(2)} KB)</p>)}</div>);
    case FieldType.RANGE:
      return (<div className="flex items-center space-x-3"><input type="range" id={field.id} name={field.id} value={rangeValue ?? field.min ?? 0} min={field.min ?? 0} max={field.max ?? 100} step={field.step ?? 1} onChange={handleChange} required={field.required} className={`w-full h-2 rounded-lg appearance-none cursor-pointer ${isDarkMode ? 'bg-slate-600 accent-sky-500' : 'bg-gray-200 accent-sky-600'}`}/><span className={`text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{rangeValue}</span></div>);
    default:
      return <p className="text-red-500">Unsupported field type: {field.type}</p>;
  }
};

export default FormFieldInput;
