
import React, { useState, useEffect, useCallback } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { FormDefinition, FormFieldDefinition, FieldType, ConditionalLogic, DynamicOptionsConfig } from '../types';
import { saveFormDefinition, getFormDefinitionById, getFormDefinitions } from '../services/formService';
import { suggestFormFields } from '../services/geminiService';
import LoadingSpinner from '../components/LoadingSpinner';
import Modal from '../components/Modal';

const fieldTypeDisplayNames: Record<FieldType, string> = {
  [FieldType.TEXT]: 'Text (Short)',
  [FieldType.TEXTAREA]: 'Text Area (Paragraph)',
  [FieldType.NUMBER]: 'Number',
  [FieldType.EMAIL]: 'Email',
  [FieldType.DATE]: 'Date',
  [FieldType.TIME]: 'Time',
  [FieldType.DATETIME_LOCAL]: 'Date and Time',
  [FieldType.SELECT]: 'Dropdown (Select One from Options)',
  [FieldType.CHECKBOX]: 'Single Checkbox (True/False)',
  [FieldType.RADIO_GROUP]: 'Radio Buttons (Select One from Options)',
  [FieldType.CHECKBOX_GROUP]: 'Checkboxes (Select Multiple from Options)',
  [FieldType.FILE]: 'File Upload',
  [FieldType.TEL]: 'Phone Number',
  [FieldType.URL]: 'Website URL',
  [FieldType.PASSWORD]: 'Password',
  [FieldType.RANGE]: 'Range Slider',
};


const CreateFormPage: React.FC = () => {
  const navigate = useNavigate();
  const { formId } = useParams<{ formId?: string }>();
  const isEditing = Boolean(formId);

  const [formName, setFormName] = useState('');
  const [formDescription, setFormDescription] = useState('');
  const [fields, setFields] = useState<FormFieldDefinition[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [geminiLoading, setGeminiLoading] = useState(false);
  const [geminiError, setGeminiError] = useState<string | null>(null);

  const [showImportFieldsModal, setShowImportFieldsModal] = useState(false);
  const [sourceForms, setSourceForms] = useState<FormDefinition[]>([]);
  const [selectedSourceFormId, setSelectedSourceFormId] = useState<string | null>(null);
  const [fieldsFromSourceForm, setFieldsFromSourceForm] = useState<FormFieldDefinition[]>([]);
  const [selectedFieldsToImport, setSelectedFieldsToImport] = useState<Set<string>>(new Set());

  const isValidSourceFieldType = (type: FieldType) => 
    type === FieldType.SELECT || type === FieldType.RADIO_GROUP;

  const getValidSourceFields = useCallback((currentFields: FormFieldDefinition[], currentFieldId?: string) => {
    return currentFields.filter(f => 
      f.id !== currentFieldId &&
      isValidSourceFieldType(f.type) && 
      f.options && 
      f.options.length > 0 && 
      f.options.every(opt => opt.trim() !== '') &&
      !f.dynamicOptionsConfig // Prevent circular dependencies or overly complex chains for now
    );
  }, []);
  
  const cleanUpVisibilityConditions = useCallback((currentFields: FormFieldDefinition[]): FormFieldDefinition[] => {
    return currentFields.map((field) => {
      if (field.condition) { // Only operate if condition object exists
        const sourceField = currentFields.find(f => f.id === field.condition!.sourceFieldId);
        
        if (!sourceField || !isValidSourceFieldType(sourceField.type) || !sourceField.options || sourceField.options.length === 0 || sourceField.options.every(opt => opt.trim() === '')) {
          return { ...field, condition: { ...field.condition, sourceFieldId: field.condition.sourceFieldId || '', triggerOptionValues: [] } };
        }
  
        const validSourceOptions = sourceField.options.filter(opt => opt.trim() !== '');
        const validTriggerValues = field.condition.triggerOptionValues.filter(triggerVal => 
          validSourceOptions.includes(triggerVal)
        );
  
        return { ...field, condition: { ...field.condition, triggerOptionValues: validTriggerValues } };
      }
      return field; 
    });
  }, []); 

  const cleanUpDynamicOptions = useCallback((currentFields: FormFieldDefinition[]): FormFieldDefinition[] => {
    return currentFields.map((field) => {
      if (field.dynamicOptionsConfig) { 
        const sourceField = currentFields.find(f => f.id === field.dynamicOptionsConfig!.dependsOnFieldId);
  
        if (!sourceField || !isValidSourceFieldType(sourceField.type) || !sourceField.options || sourceField.options.length === 0 || sourceField.options.every(opt => opt.trim() === '')) {
          return { ...field, dynamicOptionsConfig: { ...field.dynamicOptionsConfig, dependsOnFieldId: field.dynamicOptionsConfig.dependsOnFieldId || '', optionsMap: {} } };
        }
  
        const validSourceOptions = sourceField.options.filter(opt => opt.trim() !== '');
        const newOptionsMap: { [key: string]: string[] } = {};
  
        for (const sourceOpt of validSourceOptions) {
          const existingDependentOptions = field.dynamicOptionsConfig.optionsMap[sourceOpt];
          if (existingDependentOptions && Array.isArray(existingDependentOptions)) {
            newOptionsMap[sourceOpt] = existingDependentOptions.filter(depOpt => depOpt.trim() !== '');
          } else {
            newOptionsMap[sourceOpt] = []; 
          }
        }
        for (const mappedSourceOptKey in field.dynamicOptionsConfig.optionsMap) {
            if (validSourceOptions.includes(mappedSourceOptKey) && !newOptionsMap[mappedSourceOptKey]) {
                 newOptionsMap[mappedSourceOptKey] = field.dynamicOptionsConfig.optionsMap[mappedSourceOptKey].filter(depOpt => depOpt.trim() !== '');
            } else if (!validSourceOptions.includes(mappedSourceOptKey)) {
                // Key removed from source, so remove from map - this is implicitly handled by only iterating validSourceOptions earlier
            }
        }
        // Ensure only keys present in validSourceOptions exist in the final map
        const finalFilteredOptionsMap: { [key: string]: string[] } = {};
        for(const key in newOptionsMap){
            if(validSourceOptions.includes(key)){
                finalFilteredOptionsMap[key] = newOptionsMap[key];
            }
        }

        return { ...field, dynamicOptionsConfig: { ...field.dynamicOptionsConfig, optionsMap: finalFilteredOptionsMap } };
      }
      return field; 
    });
  }, []); 


  const runAllCleanups = useCallback((currentFields: FormFieldDefinition[]) => {
    let cleaned = cleanUpVisibilityConditions(currentFields);
    cleaned = cleanUpDynamicOptions(cleaned);
    return cleaned;
  }, [cleanUpVisibilityConditions, cleanUpDynamicOptions]);

  useEffect(() => {
    const loadForm = async () => {
      if (isEditing && formId) {
        setIsLoading(true);
        setError(null);
        try {
          const existingForm = await getFormDefinitionById(formId);
          if (existingForm) {
            setFormName(existingForm.name);
            setFormDescription(existingForm.description || '');
            setFields(runAllCleanups(existingForm.fields));
          } else {
            setError("Form not found.");
          }
        } catch (err) {
          console.error("Failed to load form for editing:", err);
          setError("فشل في تحميل النموذج.");
        } finally {
          setIsLoading(false);
        }
      }
    };
    loadForm();
  }, [formId, isEditing, runAllCleanups]);

  const addField = () => {
    setFields(prevFields => runAllCleanups([...prevFields, { 
      id: `field-${Date.now()}`, 
      label: '', 
      type: FieldType.TEXT, 
      required: false,
      options: undefined, 
    }]));
  };

  const updateField = (index: number, updatedProperties: Partial<FormFieldDefinition> & { blockedWordsInput?: string }) => {
    let newFields = [...fields];
    const oldField = newFields[index];
    
    let newBlockedWords: string[] | undefined = oldField.blockedWords;
    if (typeof updatedProperties.blockedWordsInput === 'string') {
      newBlockedWords = updatedProperties.blockedWordsInput.split(',').map(word => word.trim()).filter(word => word.length > 0);
      delete updatedProperties.blockedWordsInput; 
    }

    newFields[index] = { ...oldField, ...updatedProperties };
    if (newBlockedWords !== undefined) {
      newFields[index].blockedWords = newBlockedWords;
    }

    if (updatedProperties.type && updatedProperties.type !== oldField.type) {
        const newType = updatedProperties.type;
        if (newType === FieldType.SELECT || newType === FieldType.RADIO_GROUP || newType === FieldType.CHECKBOX_GROUP) {
            if (!newFields[index].dynamicOptionsConfig) { 
                if (!newFields[index].options || newFields[index].options?.length === 0) {
                     newFields[index].options = ['Option 1']; 
                }
            }
        } else {
            delete newFields[index].options;
            // If type changes away from one that supports dynamic options, remove the config
            // but only if the user hasn't explicitly enabled it (which implies newFields[index].dynamicOptionsConfig would exist)
            if (!newFields[index].dynamicOptionsConfig?.dependsOnFieldId) { // A more specific check might be if checkbox for enable is unticked
                 delete newFields[index].dynamicOptionsConfig;
            }
        }

        if (newType !== FieldType.FILE) delete newFields[index].accept;
        if (newType !== FieldType.RANGE) {
            delete newFields[index].min; delete newFields[index].max; delete newFields[index].step;
        } else { 
            newFields[index].min = newFields[index].min ?? 0;
            newFields[index].max = newFields[index].max ?? 100;
            newFields[index].step = newFields[index].step ?? 1;
        }
        if (newType !== FieldType.TEXT) {
            delete newFields[index].blockedWords;
        } else if (newBlockedWords !== undefined) { 
            newFields[index].blockedWords = newBlockedWords;
        }
    }
     
    if (updatedProperties.dynamicOptionsConfig && oldField.dynamicOptionsConfig &&
        updatedProperties.dynamicOptionsConfig.dependsOnFieldId !== oldField.dynamicOptionsConfig.dependsOnFieldId) {
        newFields[index].dynamicOptionsConfig!.optionsMap = {};
    }
    if (updatedProperties.condition && oldField.condition &&
        updatedProperties.condition.sourceFieldId !== oldField.condition.sourceFieldId) {
        newFields[index].condition!.triggerOptionValues = [];
    }
    
    setFields(runAllCleanups(newFields));
  };

  const removeField = (index: number) => {
    const fieldToRemove = fields[index];
    let newFields = fields.filter((_, i) => i !== index);
    
    newFields = newFields.map(f => {
        let modifiedField = { ...f };
        if (modifiedField.condition && modifiedField.condition.sourceFieldId === fieldToRemove.id) {
            modifiedField.condition = { sourceFieldId: '', triggerOptionValues: [] };
        }
        if (modifiedField.dynamicOptionsConfig && modifiedField.dynamicOptionsConfig.dependsOnFieldId === fieldToRemove.id) {
            modifiedField.dynamicOptionsConfig = { dependsOnFieldId: '', optionsMap: {} };
        }
        return modifiedField;
    });
    setFields(runAllCleanups(newFields)); 
  };

  const moveField = (index: number, direction: 'up' | 'down') => {
    const newFields = [...fields];
    const fieldToMove = newFields[index];
    const swapIndex = direction === 'up' ? index - 1 : index + 1;

    if (swapIndex < 0 || swapIndex >= newFields.length) return;

    newFields[index] = newFields[swapIndex];
    newFields[swapIndex] = fieldToMove;
    setFields(runAllCleanups(newFields)); 
  };

  const duplicateField = (index: number) => {
    const originalField = fields[index];
    const newId = `field-${Date.now()}-${Math.random().toString(16).slice(2)}`;
    
    const duplicatedField = JSON.parse(JSON.stringify(originalField)) as FormFieldDefinition;
    
    duplicatedField.id = newId;
    duplicatedField.label = `${originalField.label} (نسخة)`;
    
    delete duplicatedField.condition;
    delete duplicatedField.dynamicOptionsConfig;


    const newFields = [...fields];
    newFields.splice(index + 1, 0, duplicatedField);
    setFields(runAllCleanups(newFields));
  };
  
  const addStaticOption = (fieldIndex: number) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.options) {
      field.options.push(`Option ${field.options.length + 1}`);
    } else {
      field.options = ['Option 1'];
    }
    setFields(runAllCleanups(newFields));
  };

  const updateStaticOption = (fieldIndex: number, optionIndex: number, value: string) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.options) {
      field.options[optionIndex] = value;
      setFields(runAllCleanups(newFields)); 
    }
  };

  const removeStaticOption = (fieldIndex: number, optionIndex: number) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.options && field.options.length > 1) { 
      field.options.splice(optionIndex, 1);
      setFields(runAllCleanups(newFields));
    } else if (field.options && field.options.length === 1) {
        field.options.splice(optionIndex, 1);
        setFields(runAllCleanups(newFields));
    }
  };

  const addDynamicOptionMapping = (fieldIndex: number, sourceOptionValue: string) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.dynamicOptionsConfig && field.dynamicOptionsConfig.optionsMap) {
      if (!field.dynamicOptionsConfig.optionsMap[sourceOptionValue]) {
        field.dynamicOptionsConfig.optionsMap[sourceOptionValue] = [];
      }
      field.dynamicOptionsConfig.optionsMap[sourceOptionValue].push(`New Dependent Option ${field.dynamicOptionsConfig.optionsMap[sourceOptionValue].length + 1}`);
      setFields(runAllCleanups(newFields));
    }
  };

  const updateDynamicOptionMapping = (fieldIndex: number, sourceOptionValue: string, dependentOptionIndex: number, newValue: string) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.dynamicOptionsConfig?.optionsMap?.[sourceOptionValue]) {
      field.dynamicOptionsConfig.optionsMap[sourceOptionValue][dependentOptionIndex] = newValue;
      setFields(runAllCleanups(newFields));
    }
  };

  const removeDynamicOptionMapping = (fieldIndex: number, sourceOptionValue: string, dependentOptionIndex: number) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.dynamicOptionsConfig?.optionsMap?.[sourceOptionValue]) {
      field.dynamicOptionsConfig.optionsMap[sourceOptionValue].splice(dependentOptionIndex, 1);
      setFields(runAllCleanups(newFields));
    }
  };


  const handleSuggestFields = async () => {
    if (!formName.trim()) {
      setGeminiError("الرجاء إدخال عنوان للنموذج أولاً للحصول على اقتراحات.");
      return;
    }
    setGeminiLoading(true);
    setGeminiError(null);
    try {
      const suggested = await suggestFormFields(formName);
      const existingLabels = new Set(fields.map(f => f.label.toLowerCase()));
      const newUniqueFields = suggested.filter(sf => !existingLabels.has(sf.label.toLowerCase()));
      
      const fieldsWithDefaults = newUniqueFields.map(field => {
        if (field.type === FieldType.RANGE) {
          return { ...field, min: field.min ?? 0, max: field.max ?? 100, step: field.step ?? 1 };
        }
        if ((field.type === FieldType.SELECT || field.type === FieldType.RADIO_GROUP || field.type === FieldType.CHECKBOX_GROUP) && (!field.options || field.options.length === 0)) {
            return { ...field, options: ['Option 1', 'Option 2'] };
        }
        return field;
      });
      setFields(prevFields => runAllCleanups([...prevFields, ...fieldsWithDefaults]));
    } catch (e: any) {
      setGeminiError(e.message || "فشل في الحصول على اقتراحات من الذكاء الاصطناعي.");
    } finally {
      setGeminiLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!formName.trim()) { setError("اسم النموذج مطلوب."); return; }
    if (fields.some(f => !f.label.trim())) { setError("جميع تسميات الحقول مطلوبة."); return; }
    
    const finalCleanedFields = runAllCleanups(fields);

    for (const field of finalCleanedFields) {
        if (!field.dynamicOptionsConfig && (field.type === FieldType.SELECT || field.type === FieldType.RADIO_GROUP || field.type === FieldType.CHECKBOX_GROUP) && (!field.options || field.options.length === 0 || field.options.some(opt => !opt.trim()))) {
            setError(`للحقل "${field.label}", يجب ملء جميع الخيارات الثابتة وألا تكون فارغة.`);
            return;
        }
        if (field.dynamicOptionsConfig) {
            const sourceField = finalCleanedFields.find(f => f.id === field.dynamicOptionsConfig!.dependsOnFieldId);
            if(!sourceField || !sourceField.options || sourceField.options.length === 0 || sourceField.options.every(opt => opt.trim() === '')) { // Check if source field options are valid
                 setError(`الحقل المصدر للخيارات الديناميكية للحقل "${field.label}" غير صالح أو لا يحتوي على خيارات صالحة.`); return;
            }
            for(const sourceOpt of sourceField.options) {
                 if (sourceOpt.trim() === '') continue; 
                 if(!field.dynamicOptionsConfig.optionsMap[sourceOpt] || field.dynamicOptionsConfig.optionsMap[sourceOpt].length === 0 || field.dynamicOptionsConfig.optionsMap[sourceOpt].some(opt => !opt.trim())) {
                     setError(`للحقل "${field.label}", يجب ملء جميع الخيارات الديناميكية المقابلة للخيار الصالح "${sourceOpt}" من الحقل المصدر وألا تكون فارغة.`);
                     return;
                 }
            }
        }
    }
    
    setIsLoading(true);
    setError(null);

    let formCreatedAt = new Date().toISOString();
    if (isEditing && formId) {
        try {
            const existingForm = await getFormDefinitionById(formId);
            if (existingForm) {
                formCreatedAt = existingForm.createdAt;
            }
        } catch (fetchError) {
            console.error("Error fetching existing form for createdAt:", fetchError);
            setError("حدث خطأ أثناء جلب بيانات النموذج الحالية. حاول مرة أخرى.");
            setIsLoading(false);
            return;
        }
    }

    const newForm: FormDefinition = {
      id: formId || `form-${Date.now()}`,
      name: formName,
      description: formDescription,
      fields: finalCleanedFields,
      createdAt: formCreatedAt,
    };

    try {
      await saveFormDefinition(newForm);
      navigate('/');
    } catch (saveError: any) {
      console.error("Error saving form:", saveError);
      setError(saveError.message || "فشل حفظ النموذج. حاول مرة أخرى.");
    } finally {
      setIsLoading(false);
    }
  };
  
  const handleTriggerOptionChange = (fieldIndex: number, optionValue: string, checked: boolean) => {
    const newFields = [...fields];
    const field = newFields[fieldIndex];
    if (field.condition) {
      let currentTriggerValues = field.condition.triggerOptionValues || [];
      if (checked) {
        if (!currentTriggerValues.includes(optionValue)) {
          currentTriggerValues = [...currentTriggerValues, optionValue];
        }
      } else {
        currentTriggerValues = currentTriggerValues.filter(val => val !== optionValue);
      }
      // Use updateField to ensure all cleanups run
      updateField(fieldIndex, { condition: { ...field.condition, triggerOptionValues: currentTriggerValues } });
    }
  };

  const handleOpenImportFieldsModal = async () => {
    try {
        const allForms = await getFormDefinitions();
        setSourceForms(allForms.filter(f => f.id !== formId)); 
        setSelectedSourceFormId(null);
        setFieldsFromSourceForm([]);
        setSelectedFieldsToImport(new Set());
        setShowImportFieldsModal(true);
    } catch (e) {
        console.error("Failed to load forms for import:", e);
        setError("فشل في تحميل النماذج المتاحة للاستيراد.");
    }
  };

  const handleSourceFormChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const newSourceFormId = e.target.value;
    setSelectedSourceFormId(newSourceFormId);
    if (newSourceFormId) {
      const selectedForm = sourceForms.find(f => f.id === newSourceFormId);
      setFieldsFromSourceForm(selectedForm ? selectedForm.fields : []);
      setSelectedFieldsToImport(new Set());
    } else {
      setFieldsFromSourceForm([]);
    }
  };

  const toggleFieldToImport = (fieldIdToToggle: string) => {
    setSelectedFieldsToImport(prev => {
      const newSet = new Set(prev);
      if (newSet.has(fieldIdToToggle)) newSet.delete(fieldIdToToggle);
      else newSet.add(fieldIdToToggle);
      return newSet;
    });
  };
  
  const toggleSelectAllFieldsToImport = (checked: boolean) => {
      setSelectedFieldsToImport(checked ? new Set(fieldsFromSourceForm.map(f => f.id)) : new Set());
  };

  const handleConfirmImportFields = () => {
    if (!selectedSourceFormId || selectedFieldsToImport.size === 0) return;
    const idMapping = new Map<string, string>(); 
    const fieldsToActuallyImport = fieldsFromSourceForm.filter(f => selectedFieldsToImport.has(f.id));

    const clonedFieldsWithNewIds = fieldsToActuallyImport.map(originalField => {
      const newId = `imported-${originalField.id}-${Date.now()}-${Math.random().toString(16).slice(2)}`;
      idMapping.set(originalField.id, newId);
      return { ...JSON.parse(JSON.stringify(originalField)), id: newId };
    });

    const finalImportedFields = clonedFieldsWithNewIds.map(clonedField => {
      if (clonedField.condition && clonedField.condition.sourceFieldId) {
        const originalSourceOfConditionId = clonedField.condition.sourceFieldId;
        if (idMapping.has(originalSourceOfConditionId)) {
          clonedField.condition.sourceFieldId = idMapping.get(originalSourceOfConditionId)!;
        } else { 
          const currentFormHasSource = fields.some(f => f.id === originalSourceOfConditionId);
          if(!currentFormHasSource) delete clonedField.condition;
        }
      }
      if (clonedField.dynamicOptionsConfig && clonedField.dynamicOptionsConfig.dependsOnFieldId) {
        const originalDependsOnFieldId = clonedField.dynamicOptionsConfig.dependsOnFieldId;
        if (idMapping.has(originalDependsOnFieldId)) {
            clonedField.dynamicOptionsConfig.dependsOnFieldId = idMapping.get(originalDependsOnFieldId)!;
        } else {
           const currentFormHasSource = fields.some(f => f.id === originalDependsOnFieldId);
           if(!currentFormHasSource) delete clonedField.dynamicOptionsConfig;
        }
      }
      return clonedField;
    });

    setFields(prevFields => runAllCleanups([...prevFields, ...finalImportedFields]));
    setShowImportFieldsModal(false);
  };


  if (isLoading && !isEditing && !geminiLoading) return <div className="flex justify-center items-center h-64"><LoadingSpinner text="جاري تهيئة النموذج..." /></div>;
  if (isLoading && isEditing && !geminiLoading) return <div className="flex justify-center items-center h-64"><LoadingSpinner text="جاري تحميل النموذج..." /></div>;
  
  if (error && !isLoading) return <p className="text-red-400 text-center text-xl">{error}</p>;


  return (
    <div className="max-w-4xl mx-auto bg-slate-800 p-6 sm:p-8 rounded-lg shadow-2xl">
      <h1 className="text-3xl font-bold text-sky-400 mb-8">{isEditing ? 'تعديل النموذج' : 'إنشاء نموذج جديد'}</h1>
      <form onSubmit={handleSubmit} className="space-y-6">
        <div>
          <label htmlFor="formName" className="block text-sm font-medium text-gray-300 mb-1">اسم النموذج <span className="text-red-400">*</span></label>
          <input type="text" id="formName" value={formName} onChange={(e) => setFormName(e.target.value)} className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200" required />
        </div>
        <div>
          <label htmlFor="formDescription" className="block text-sm font-medium text-gray-300 mb-1">وصف النموذج (اختياري)</label>
          <textarea id="formDescription" value={formDescription} onChange={(e) => setFormDescription(e.target.value)} rows={3} className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200" />
        </div>

        <div className="border-t border-slate-700 pt-6">
          <div className="flex flex-wrap gap-2 justify-between items-center mb-4">
            <h2 className="text-xl font-semibold text-sky-400">حقول النموذج</h2>
            <div className="flex flex-wrap gap-2">
                <button type="button" onClick={handleOpenImportFieldsModal} className="bg-purple-500 hover:bg-purple-600 text-white font-medium py-2 px-4 rounded-md shadow-sm transition-colors inline-flex items-center" title="استيراد حقول من نموذج آخر">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5 mr-2"><path strokeLinecap="round" strokeLinejoin="round" d="M19.5 14.25v-2.625a3.375 3.375 0 00-3.375-3.375h-1.5A1.125 1.125 0 0113.5 7.125v-1.5a3.375 3.375 0 00-3.375-3.375H8.25m.75 12l3 3m0 0l3-3m-3 3v-6m-1.5-9H5.625c-.621 0-1.125.504-1.125 1.125v17.25c0 .621.504 1.125 1.125 1.125h12.75c.621 0 1.125-.504 1.125-1.125V11.25a9 9 0 00-9-9z" /></svg>
                    استيراد حقول
                </button>
                <button type="button" onClick={handleSuggestFields} disabled={geminiLoading || !process.env.API_KEY} className="bg-teal-500 hover:bg-teal-600 text-white font-medium py-2 px-4 rounded-md shadow-sm transition-colors disabled:opacity-50 inline-flex items-center" title={!process.env.API_KEY ? "مفتاح Gemini API غير مهيأ" : "اقترح حقولاً باستخدام الذكاء الاصطناعي"}>
                    {geminiLoading ? <LoadingSpinner size="sm" color="text-white"/> : ( <> <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5 mr-2"> <path strokeLinecap="round" strokeLinejoin="round" d="M9.813 15.904L9 18.75l-.813-2.846a4.5 4.5 0 00-3.09-3.09L2.25 12l2.846-.813a4.5 4.5 0 003.09-3.09L9 5.25l.813 2.846a4.5 4.5 0 003.09 3.09L15.75 12l-2.846.813a4.5 4.5 0 00-3.09 3.09zM18.259 8.715L18 9.75l-.259-1.035a3.375 3.375 0 00-2.455-2.456L14.25 6l1.036-.259a3.375 3.375 0 002.455-2.456L18 2.25l.259 1.035a3.375 3.375 0 002.456 2.456L21.75 6l-1.035.259a3.375 3.375 0 00-2.456 2.456zM16.894 20.567L16.5 21.75l-.394-1.183a2.25 2.25 0 00-1.423-1.423L13.5 18.75l1.183-.394a2.25 2.25 0 001.423-1.423l.394-1.183.394 1.183a2.25 2.25 0 001.423 1.423l1.183.394-1.183.394a2.25 2.25 0 00-1.423 1.423z" /> </svg> اقتراح حقول (AI) </> )}
                </button>
            </div>
          </div>
          {geminiError && <p className="text-red-400 text-sm mb-4">{geminiError}</p>}
          {!process.env.API_KEY && <p className="text-yellow-400 text-sm mb-4">اقتراح الحقول بواسطة الذكاء الاصطناعي معطل. لم يتم العثور على مفتاح API_KEY.</p>}

          {fields.map((field, index) => (
            <div key={field.id} className="p-4 mb-4 border border-slate-700 rounded-md bg-slate-700/30 space-y-4">
              <div className="flex justify-between items-start">
                  <h3 className="text-lg font-medium text-sky-300 mb-2">السؤال {index + 1}</h3>
                  <div className="flex space-x-1">
                      <button type="button" onClick={() => moveField(index, 'up')} disabled={index === 0} className="p-1.5 text-slate-300 hover:text-sky-300 disabled:text-slate-600 disabled:cursor-not-allowed rounded-md bg-slate-600 hover:bg-slate-500 disabled:bg-slate-700" title="نقل لأعلى"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M4.5 15.75l7.5-7.5 7.5 7.5" /></svg></button>
                      <button type="button" onClick={() => moveField(index, 'down')} disabled={index === fields.length - 1} className="p-1.5 text-slate-300 hover:text-sky-300 disabled:text-slate-600 disabled:cursor-not-allowed rounded-md bg-slate-600 hover:bg-slate-500 disabled:bg-slate-700" title="نقل لأسفل"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M19.5 8.25l-7.5 7.5-7.5-7.5" /></svg></button>
                      <button type="button" onClick={() => duplicateField(index)} className="p-1.5 text-slate-300 hover:text-purple-300 rounded-md bg-slate-600 hover:bg-slate-500" title="تكرار السؤال"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M15.75 17.25v3.375c0 .621-.504 1.125-1.125 1.125h-9.75a1.125 1.125 0 01-1.125-1.125V7.875c0-.621.504-1.125 1.125-1.125H6.75a9.06 9.06 0 011.5.124m7.5 10.376h3.375c.621 0 1.125-.504 1.125-1.125V11.25c0-4.46-3.243-8.161-7.5-8.876a9.06 9.06 0 00-1.5-.124H9.375c-.621 0-1.125.504-1.125 1.125v3.5m7.5 10.375H9.375a1.125 1.125 0 01-1.125-1.125v-9.25m12 6.625v-1.875a3.375 3.375 0 00-3.375-3.375h-1.5a1.125 1.125 0 01-1.125-1.125v-1.5a3.375 3.375 0 00-3.375-3.375H9.75" /></svg></button>
                  </div>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div><label htmlFor={`field-label-${index}`} className="block text-sm font-medium text-gray-300 mb-1">تسمية الحقل <span className="text-red-400">*</span></label><input type="text" id={`field-label-${index}`} value={field.label} onChange={(e) => updateField(index, { label: e.target.value })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200" required /></div>
                <div><label htmlFor={`field-type-${index}`} className="block text-sm font-medium text-gray-300 mb-1">نوع الحقل</label><select id={`field-type-${index}`} value={field.type} onChange={(e) => updateField(index, { type: e.target.value as FieldType })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200">{Object.values(FieldType).map(type => (<option key={type} value={type}>{fieldTypeDisplayNames[type]}</option>))}</select></div>
              </div>
              <div><label htmlFor={`field-placeholder-${index}`} className="block text-sm font-medium text-gray-300 mb-1">العنصر النائب (اختياري)</label><input type="text" id={`field-placeholder-${index}`} value={field.placeholder || ''} onChange={(e) => updateField(index, { placeholder: e.target.value })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200" /></div>
              {field.type === FieldType.TEXT && (<div><label htmlFor={`field-blockedWords-${index}`} className="block text-sm font-medium text-gray-300 mb-1">الكلمات المحظورة (مفصولة بفواصل)</label><textarea id={`field-blockedWords-${index}`} value={field.blockedWords?.join(', ') || ''} onChange={(e) => updateField(index, { blockedWordsInput: e.target.value })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200" rows={2} placeholder="مثال: كلمة1, كلمة2, كلمة3"/></div>)}
              {field.type === FieldType.FILE && ( <div> <label htmlFor={`field-accept-${index}`} className="block text-sm font-medium text-gray-300 mb-1">أنواع الملفات المقبولة (مثال: image/*, .pdf)</label> <input type="text" id={`field-accept-${index}`} value={field.accept || ''} onChange={(e) => updateField(index, { accept: e.target.value })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200" placeholder="image/*, application/pdf" /> </div> )}
              {field.type === FieldType.RANGE && ( <div className="grid grid-cols-1 md:grid-cols-3 gap-4"> <div> <label htmlFor={`field-min-${index}`} className="block text-sm font-medium text-gray-300 mb-1">الحد الأدنى</label> <input type="number" id={`field-min-${index}`} value={field.min ?? 0} onChange={(e) => updateField(index, { min: parseFloat(e.target.value) })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200"/> </div> <div> <label htmlFor={`field-max-${index}`} className="block text-sm font-medium text-gray-300 mb-1">الحد الأقصى</label> <input type="number" id={`field-max-${index}`} value={field.max ?? 100} onChange={(e) => updateField(index, { max: parseFloat(e.target.value) })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200"/> </div> <div> <label htmlFor={`field-step-${index}`} className="block text-sm font-medium text-gray-300 mb-1">الخطوة</label> <input type="number" id={`field-step-${index}`} value={field.step ?? 1} onChange={(e) => updateField(index, { step: parseFloat(e.target.value) })} className="w-full px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200"/> </div> </div> )}
              
              {(field.type === FieldType.SELECT || field.type === FieldType.RADIO_GROUP || field.type === FieldType.CHECKBOX_GROUP) && (
                <div className="mt-3 pt-3 border-t border-slate-600 space-y-3">
                    <div>
                        <input type="checkbox" id={`enable-dynamic-options-${index}`} checked={!!field.dynamicOptionsConfig}
                            onChange={(e) => {
                                const isEnabled = e.target.checked;
                                if (isEnabled) {
                                    updateField(index, { dynamicOptionsConfig: { dependsOnFieldId: '', optionsMap: {} }, options: field.options }); // Keep static options temporarily
                                } else {
                                    updateField(index, { dynamicOptionsConfig: undefined, options: field.options || ['Option 1'] });
                                }
                            }}
                            className="h-4 w-4 text-sky-500 border-slate-500 rounded focus:ring-sky-400 bg-slate-600 mr-2"
                        />
                        <label htmlFor={`enable-dynamic-options-${index}`} className="text-sm text-gray-300">استخدام خيارات ديناميكية (تعتمد على حقل آخر)</label>
                    </div>

                    {field.dynamicOptionsConfig ? (
                        <div className="pl-6 border-l-2 border-slate-500 ml-2 py-2 space-y-3">
                             <div>
                                <label htmlFor={`dynamic-options-source-${index}`} className="block text-xs font-medium text-gray-400 mb-1">تعتمد الخيارات على الحقل:</label>
                                <select id={`dynamic-options-source-${index}`}
                                    value={field.dynamicOptionsConfig.dependsOnFieldId}
                                    onChange={(e) => updateField(index, { dynamicOptionsConfig: { ...field.dynamicOptionsConfig!, dependsOnFieldId: e.target.value, optionsMap: {} }})}
                                    className="w-full px-3 py-1.5 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200 text-sm"
                                >
                                    <option value="">-- اختر الحقل المصدر --</option>
                                    {getValidSourceFields(fields, field.id).map(sf => (
                                        <option key={sf.id} value={sf.id}>{sf.label || `ID: ${sf.id}`}</option>
                                    ))}
                                </select>
                             </div>
                             {field.dynamicOptionsConfig.dependsOnFieldId && getValidSourceFields(fields, field.id).find(f => f.id === field.dynamicOptionsConfig!.dependsOnFieldId)?.options?.filter(opt => opt.trim() !== '').map(sourceOptValue => (
                                <div key={sourceOptValue} className="p-2 border border-slate-500 rounded-md">
                                    <p className="text-sm text-gray-300 mb-1">الخيارات عندما تكون قيمة "{getValidSourceFields(fields, field.id).find(f => f.id === field.dynamicOptionsConfig!.dependsOnFieldId)?.label}" هي: <strong className="text-sky-300">{sourceOptValue}</strong></p>
                                    {(field.dynamicOptionsConfig?.optionsMap?.[sourceOptValue] || []).map((depOpt, depOptIdx) => (
                                        <div key={depOptIdx} className="flex items-center space-x-2 mb-1">
                                            <input type="text" value={depOpt}
                                                onChange={(e) => updateDynamicOptionMapping(index, sourceOptValue, depOptIdx, e.target.value)}
                                                className="flex-grow px-2 py-1 bg-slate-500 border-slate-400 rounded-sm text-gray-200 text-xs" placeholder={`خيار تابع ${depOptIdx + 1}`}
                                            />
                                            <button type="button" onClick={() => removeDynamicOptionMapping(index, sourceOptValue, depOptIdx)} className="text-red-300 hover:text-red-200 p-1 rounded-sm bg-slate-600 hover:bg-slate-500 text-xs"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-3 h-3"><path strokeLinecap="round" strokeLinejoin="round" d="M19.5 12h-15" /></svg></button>
                                        </div>
                                    ))}
                                    <button type="button" onClick={() => addDynamicOptionMapping(index, sourceOptValue)} className="text-sky-300 hover:text-sky-200 text-xs font-medium py-0.5 px-1.5 border border-sky-400 rounded-sm hover:border-sky-300 inline-flex items-center mt-1">إضافة خيار تابع</button>
                                </div>
                             ))}
                             {field.dynamicOptionsConfig.dependsOnFieldId && (!getValidSourceFields(fields, field.id).find(f => f.id === field.dynamicOptionsConfig!.dependsOnFieldId)?.options || getValidSourceFields(fields, field.id).find(f => f.id === field.dynamicOptionsConfig!.dependsOnFieldId)!.options!.every(opt => opt.trim() === '')) &&
                                <p className="text-xs text-amber-400">الحقل المصدر المختار لا يحتوي على خيارات صالحة لتكوين الخيارات التابعة.</p>
                             }
                        </div>
                    ) : ( 
                        <div className="space-y-2">
                            <label className="block text-sm font-medium text-gray-300">الخيارات الثابتة:</label>
                            {field.options?.map((option, optIndex) => (
                                <div key={optIndex} className="flex items-center space-x-2">
                                <input type="text" value={option} onChange={(e) => updateStaticOption(index, optIndex, e.target.value)} className="flex-grow px-3 py-2 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200" placeholder={`الخيار ${optIndex + 1}`} />
                                <button type="button" onClick={() => removeStaticOption(index, optIndex)} className="text-red-400 hover:text-red-300 p-1.5 rounded-md bg-slate-700 hover:bg-slate-600"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M19.5 12h-15" /></svg></button>
                                </div>
                            ))}
                            <button type="button" onClick={() => addStaticOption(index)} className="text-sky-400 hover:text-sky-300 text-sm font-medium py-1 px-3 border border-sky-500 rounded-md hover:border-sky-400 inline-flex items-center"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-4 h-4 mr-1"><path strokeLinecap="round" strokeLinejoin="round" d="M12 4.5v15m7.5-7.5h-15" /></svg>إضافة خيار</button>
                        </div>
                    )}
                </div>
              )}

              <div className="mt-3 pt-3 border-t border-slate-600 space-y-2">
                <label className="block text-sm font-medium text-gray-300">العرض الشرطي (إظهار/إخفاء)</label>
                <div className="flex items-center">
                    <input type="checkbox" id={`field-conditional-enable-${index}`} checked={!!field.condition}
                        onChange={(e) => {
                            if (e.target.checked) {
                                updateField(index, { condition: { sourceFieldId: '', triggerOptionValues: [] } });
                            } else {
                                updateField(index, { condition: undefined });
                            }
                        }}
                        className="h-4 w-4 text-sky-500 border-slate-500 rounded focus:ring-sky-400 bg-slate-600" aria-describedby={`conditional-desc-${index}`}
                    />
                    <label htmlFor={`field-conditional-enable-${index}`} className="ml-2 text-sm text-gray-300">إظهار هذا الحقل بشكل شرطي</label>
                </div>
                {field.condition && (
                    <div className="space-y-3 pl-6 border-l-2 border-slate-500 ml-2 py-2">
                        <div>
                            <label htmlFor={`condition-sourceField-${index}`} className="block text-xs font-medium text-gray-400 mb-1">إذا كان الحقل:</label>
                            <select id={`condition-sourceField-${index}`} value={field.condition.sourceFieldId}
                                onChange={(e) => {
                                    const newSourceFieldId = e.target.value;
                                    updateField(index, { condition: { sourceFieldId: newSourceFieldId, triggerOptionValues: [] } });
                                }}
                                className="w-full px-3 py-1.5 bg-slate-600 border border-slate-500 rounded-md shadow-sm focus:outline-none focus:ring-1 focus:ring-sky-400 text-gray-200 text-sm"
                            >
                                <option value="">-- اختر الحقل المصدر --</option>
                                {getValidSourceFields(fields, field.id).map(sf => ( 
                                    <option key={sf.id} value={sf.id}>{sf.label || `ID: ${sf.id}`}</option>
                                ))}
                            </select>
                        </div>
                        {field.condition.sourceFieldId && getValidSourceFields(fields, field.id).find(f => f.id === field.condition!.sourceFieldId)?.options && (
                             <div>
                                <label className="block text-xs font-medium text-gray-400 mb-1">عندما تكون القيمة المختارة واحدة من:</label>
                                <div className="space-y-1 max-h-32 overflow-y-auto bg-slate-600/50 p-2 rounded-md">
                                {getValidSourceFields(fields, field.id).find(f => f.id === field.condition!.sourceFieldId)?.options?.filter(opt => opt.trim() !== '').map(optValue => (
                                    <label key={optValue} className="flex items-center text-sm text-gray-200">
                                        <input type="checkbox" value={optValue} checked={field.condition!.triggerOptionValues.includes(optValue)} onChange={(e) => handleTriggerOptionChange(index, optValue, e.target.checked)} className="h-3.5 w-3.5 text-sky-500 border-slate-400 rounded focus:ring-sky-400 bg-slate-500 mr-2"/>
                                        {optValue}
                                    </label>
                                ))}
                                </div>
                            </div>
                        )}
                        {field.condition.sourceFieldId && (!getValidSourceFields(fields, field.id).find(f => f.id === field.condition!.sourceFieldId)?.options || getValidSourceFields(fields, field.id).find(f => f.id === field.condition!.sourceFieldId)!.options!.every(opt => opt.trim() === '')) &&
                            <p className="text-xs text-amber-400">الحقل المصدر المختار لا يحتوي على خيارات صالحة لتحديد شروط الإظهار.</p>
                        }
                    </div>
                )}
              </div>

              <div className="flex items-center justify-between mt-3">
                 <div className="flex items-center"><input type="checkbox" id={`field-required-${index}`} checked={field.required || false} onChange={(e) => updateField(index, { required: e.target.checked })} className="h-4 w-4 text-sky-500 border-slate-500 rounded focus:ring-sky-400 bg-slate-600"/><label htmlFor={`field-required-${index}`} className="ml-2 text-sm text-gray-300">مطلوب</label></div>
                <button type="button" onClick={() => removeField(index)} className="text-red-400 hover:text-red-300 font-medium text-sm inline-flex items-center"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-4 h-4 mr-1"><path strokeLinecap="round" strokeLinejoin="round" d="M14.74 9l-.346 9m-4.788 0L9.26 9m9.968-3.21c.342.052.682.107 1.022.166m-1.022-.165L18.16 19.673a2.25 2.25 0 01-2.244 2.077H8.084a2.25 2.25 0 01-2.244-2.077L4.772 5.79m14.456 0a48.108 48.108 0 00-3.478-.397m-12.56 0c1.153 0 2.243.096 3.222.261m3.222.261L11 5.25v-.934a1.875 1.875 0 011.875-1.875h1.05M11 5.25v.308m0 0L11 8.25m0 0l-.346 9m.346-9l.346-9m0 0H12m6.23 6.39l-.346-9M12 8.25L11 5.25M12 8.25l.346 9m-.346-9L12 5.25M12 5.25H9.772" /></svg>إزالة الحقل</button>
              </div>
            </div>
          ))}
          <button type="button" onClick={addField} className="mt-2 text-sky-400 hover:text-sky-300 font-medium py-2 px-4 rounded-md border border-sky-500 hover:border-sky-400 transition-colors inline-flex items-center"><svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-5 h-5 mr-2"><path strokeLinecap="round" strokeLinejoin="round" d="M12 4.5v15m7.5-7.5h-15" /></svg>إضافة حقل</button>
        </div>
        
        {error && <p className="text-red-400 text-sm mt-4">{error}</p>}

        <div className="flex justify-end space-x-3 pt-6 border-t border-slate-700">
          <button type="button" onClick={() => navigate('/')} className="px-6 py-2 text-sm font-medium text-gray-300 bg-slate-600 hover:bg-slate-500 rounded-md focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-slate-800 focus:ring-slate-500 transition-colors">إلغاء</button>
          <button type="submit" disabled={isLoading} className="px-6 py-2 text-sm font-medium text-white bg-sky-500 hover:bg-sky-600 rounded-md focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-slate-800 focus:ring-sky-400 transition-colors disabled:opacity-70">
            {isLoading ? <LoadingSpinner size="sm"/> : (isEditing ? 'حفظ التغييرات' : 'إنشاء النموذج')}
          </button>
        </div>
      </form>

      <Modal isOpen={showImportFieldsModal} onClose={() => setShowImportFieldsModal(false)} title="استيراد حقول من نموذج آخر" primaryActionText="استيراد الحقول المحددة" onPrimaryAction={handleConfirmImportFields} secondaryActionText="إلغاء" onSecondaryAction={() => setShowImportFieldsModal(false)}>
        <div className="space-y-4">
          <div><label htmlFor="sourceFormSelect" className="block text-sm font-medium text-gray-300 mb-1">اختر نموذجًا لاستيراد الحقول منه:</label><select id="sourceFormSelect" value={selectedSourceFormId || ''} onChange={handleSourceFormChange} className="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-sky-400 focus:border-sky-400 text-gray-200"><option value="">-- اختر نموذجًا --</option>{sourceForms.map(form => (<option key={form.id} value={form.id}>{form.name}</option>))}</select></div>
          {selectedSourceFormId && fieldsFromSourceForm.length > 0 && (<div className="border-t border-slate-600 pt-4"><h4 className="text-md font-semibold text-sky-300 mb-2">الحقول في النموذج المصدر: "{sourceForms.find(f=>f.id === selectedSourceFormId)?.name}"</h4><div className="mb-2"><label className="flex items-center text-sm text-gray-300"><input type="checkbox" className="h-4 w-4 text-sky-500 border-slate-500 rounded focus:ring-sky-400 bg-slate-600 mr-2" checked={selectedFieldsToImport.size === fieldsFromSourceForm.length && fieldsFromSourceForm.length > 0} onChange={(e) => toggleSelectAllFieldsToImport(e.target.checked)} disabled={fieldsFromSourceForm.length === 0}/>تحديد الكل / إلغاء تحديد الكل</label></div><div className="max-h-60 overflow-y-auto space-y-1 bg-slate-700/50 p-3 rounded-md">{fieldsFromSourceForm.map(field => (<label key={field.id} className="flex items-center p-2 rounded hover:bg-slate-600 cursor-pointer"><input type="checkbox" className="h-4 w-4 text-sky-500 border-slate-500 rounded focus:ring-sky-400 bg-slate-600 mr-3" checked={selectedFieldsToImport.has(field.id)} onChange={() => toggleFieldToImport(field.id)}/><span className="text-sm text-gray-200">{field.label} ({fieldTypeDisplayNames[field.type]})</span></label>))}</div></div>)}
          {selectedSourceFormId && fieldsFromSourceForm.length === 0 && (<p className="text-sm text-slate-400">النموذج المصدر المختار لا يحتوي على حقول.</p>)}
        </div>
      </Modal>
    </div>
  );
};

export default CreateFormPage;
