
import { GoogleGenAI } from "@google/genai";
import { FormFieldDefinition, FieldType, SuggestedField } from '../types';
import { GEMINI_MODEL_TEXT } from '../constants';

const API_KEY = process.env.API_KEY;

if (!API_KEY) {
  console.error("API_KEY environment variable not set. Gemini API functionality will be disabled.");
}

const ai = API_KEY ? new GoogleGenAI({ apiKey: API_KEY }) : null;

const mapSuggestedTypeToFieldType = (suggestedType: string): FieldType => {
  const sType = suggestedType.toLowerCase().trim();
  switch (sType) {
    case 'number': return FieldType.NUMBER;
    case 'email': return FieldType.EMAIL;
    case 'date': return FieldType.DATE;
    case 'time': return FieldType.TIME; // Added time mapping
    case 'datetime-local': return FieldType.DATETIME_LOCAL;
    case 'select': return FieldType.SELECT;
    case 'textarea': return FieldType.TEXTAREA;
    case 'checkbox': return FieldType.CHECKBOX; // For single boolean checkbox
    case 'radio_group':
    case 'radio': 
      return FieldType.RADIO_GROUP;
    case 'checkbox_group':
    case 'checkboxes': // Plural might be suggested for group
      return FieldType.CHECKBOX_GROUP;
    case 'file': return FieldType.FILE;
    case 'tel':
    case 'phone':
      return FieldType.TEL;
    case 'url':
    case 'website':
      return FieldType.URL;
    case 'password': return FieldType.PASSWORD;
    case 'range': 
    case 'slider':
      return FieldType.RANGE;
    case 'text':
    default:
      return FieldType.TEXT;
  }
};

export const suggestFormFields = async (formTitle: string): Promise<FormFieldDefinition[]> => {
  if (!ai) {
    throw new Error("Gemini API client not initialized. Check API_KEY.");
  }

  const prompt = `
    Based on the form title "${formTitle}", suggest a list of 3 to 5 relevant form fields.
    For each field, provide a "label" (user-friendly name), a "type", and optionally "placeholder" text.
    If the type is 'select', 'radio_group', or 'checkbox_group', also provide "options" (an array of strings).
    If the type is 'file', you can optionally provide "accept" (string, e.g., "image/*", ".pdf").
    If the type is 'range', you can optionally provide "min" (number), "max" (number), and "step" (number).
    
    Valid types are: text, textarea, number, email, date, time, datetime-local, select, checkbox (for a single boolean choice like 'agree to terms'), radio_group (for multiple options, select one), checkbox_group (for multiple options, select many), file, tel, url, password, range.

    Return the response as a valid JSON array of objects. Each object should have keys: "label" (string), "type" (string), and optionally "placeholder" (string), "options" (array of strings), "accept" (string), "min" (number), "max" (number), "step" (number) as appropriate for the type.

    Example for "Event Registration" form:
    [
      { "label": "Full Name", "type": "text", "placeholder": "Enter your full name" },
      { "label": "Email Address", "type": "email", "placeholder": "you@example.com" },
      { "label": "Age", "type": "number", "placeholder": "Your age"},
      { "label": "Preferred Start Time", "type": "time", "placeholder": "HH:MM" },
      { "label": "Event Date Choice", "type": "radio_group", "options": ["Saturday Morning", "Saturday Afternoon", "Sunday Full Day"] },
      { "label": "Dietary Restrictions", "type": "checkbox_group", "options": ["Vegetarian", "Gluten-Free", "Nut Allergy"], "placeholder":"Select any that apply" },
      { "label": "Presentation Slides (Optional)", "type": "file", "accept": ".ppt, .pptx, .pdf" },
      { "label": "Experience Level (1-10)", "type": "range", "min": 1, "max": 10, "step": 1 }
    ]
  `;

  try {
    const response = await ai.models.generateContent({
      model: GEMINI_MODEL_TEXT,
      contents: prompt,
      config: {
        responseMimeType: "application/json",
      }
    });

    let jsonStr = response.text.trim();
    const fenceRegex = /^```(\w*)?\s*\n?(.*?)\n?\s*```$/s;
    const match = jsonStr.match(fenceRegex);
    if (match && match[2]) {
      jsonStr = match[2].trim();
    }
    
    const suggestedFieldsData: SuggestedField[] = JSON.parse(jsonStr);

    if (!Array.isArray(suggestedFieldsData)) {
        console.error("Gemini response is not an array:", suggestedFieldsData);
        return [];
    }

    return suggestedFieldsData.map((field, index) => {
      const fieldType = mapSuggestedTypeToFieldType(field.type);
      const mappedField: FormFieldDefinition = {
        id: `suggested-${Date.now()}-${index}`,
        label: field.label,
        type: fieldType,
        placeholder: field.placeholder,
        required: false, // Default to not required
      };

      if ((fieldType === FieldType.SELECT || fieldType === FieldType.RADIO_GROUP || fieldType === FieldType.CHECKBOX_GROUP)) {
        mappedField.options = field.options && field.options.length > 0 ? field.options : ['Option 1', 'Option 2'];
      }
      if (fieldType === FieldType.FILE && field.accept) {
        mappedField.accept = field.accept;
      }
      if (fieldType === FieldType.RANGE) {
        mappedField.min = typeof field.min === 'number' ? field.min : 0;
        mappedField.max = typeof field.max === 'number' ? field.max : 100;
        mappedField.step = typeof field.step === 'number' ? field.step : 1;
      }
      return mappedField;
    });

  } catch (error) {
    console.error("Error suggesting form fields with Gemini:", error);
    let errorMessage = "Failed to suggest fields.";
    if (error instanceof Error) {
        errorMessage += ` Details: ${error.message}`;
    }
    throw new Error(errorMessage);
  }
};