/**
 * ملف JavaScript لتحسين وظائف الموقع وتسريع إرسال الواتساب
 */

// التأكد من تحميل المستند بالكامل قبل تنفيذ الكود
document.addEventListener('DOMContentLoaded', function() {
    // إضافة خط Tajawal لدعم اللغة العربية
    const tajawalFont = document.createElement('link');
    tajawalFont.rel = 'stylesheet';
    tajawalFont.href = 'https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap';
    document.head.appendChild(tajawalFont);

    // تهيئة النموذج
    initForm();

    // إضافة مستمعي الأحداث
    addEventListeners();
    
    // التحقق من إذن النوافذ المنبثقة
    checkPopupPermission();
});

/**
 * تهيئة النموذج وإعداده
 */
function initForm() {
    // إضافة الفئات للعناصر لتطبيق التنسيقات الجديدة
    const form = document.getElementById('form');
    if (form) {
        form.classList.add('form-container');

        // تحويل أقسام النموذج إلى فئة form-section
        const formSections = document.querySelectorAll('.form-check');
        formSections.forEach((section, index) => {
            section.classList.add('form-section');
            // إزالة التنسيقات المضمنة
            section.removeAttribute('style');
            // إخفاء جميع الأقسام ما عدا القسم الأول (اختيار المكان)
            if (index > 0) {
                section.style.display = 'none';
            }
        });

        // تحسين عناصر الإدخال
        const inputs = document.querySelectorAll('input[type="text"], input[type="date"], input[type="number"], textarea');
        inputs.forEach(input => {
            input.classList.add('form-control');
        });

        // تحسين الأزرار
        const buttons = document.querySelectorAll('button[type="submit"]');
        buttons.forEach(button => {
            button.classList.add('btn-primary');
        });

        // تحسين علامات الحقول المطلوبة
        const requiredMarks = document.querySelectorAll('.vnumgf');
        requiredMarks.forEach(mark => {
            mark.classList.add('required-mark');
            mark.removeAttribute('style');
        });

        // إضافة معرف UUID للنموذج
        addUuidToForm();
    }
}

/**
 * إضافة مستمعي الأحداث للعناصر التفاعلية
 */
function addEventListeners() {
    // مستمع حدث للزر الرئيسي
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.addEventListener('click', function(e) {
            e.preventDefault();
            disableButton();
            sendFormToGoogleSheetsAndWhatsapp();
        });
    }

    // مستمعي أحداث للحقول المطلوبة للتحقق من الصحة في الوقت الفعلي
    const requiredInputs = document.querySelectorAll('input[required], select[required], textarea[required]');
    requiredInputs.forEach(input => {
        input.addEventListener('change', checkRequiredFieldsFilled);
        input.addEventListener('input', checkRequiredFieldsFilled);
    });

    // مستمعي أحداث للأزرار الراديو لتبديل رؤية الحقول
    const radioButtons = document.querySelectorAll('input[type="radio"][name="المكان الذي انت فيه الان ( الذي تكتب تقريره هذا )"]');
    radioButtons.forEach(radio => {
        radio.addEventListener('change', togglePhoneVisibility);
    });
}

/**
 * إضافة معرف UUID للنموذج لمنع الإرسال المتكرر
 */
function addUuidToForm() {
    const form = document.getElementById("form");
    if (form) {
        const uuidInput = document.createElement("input");
        uuidInput.type = "hidden";
        uuidInput.name = "uuid";
        uuidInput.value = generateUUID();
        form.appendChild(uuidInput);
    }
}

/**
 * إنشاء معرف UUID فريد
 * @returns {string} معرف UUID
 */
function generateUUID() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
        var r = (Math.random() * 16) | 0,
            v = c == 'x' ? r : (r & 0x3) | 0x8;
        return v.toString(16);
    });
}

/**
 * التحقق مما إذا كان العنصر مرئيًا
 * @param {HTMLElement} element العنصر المراد التحقق منه
 * @returns {boolean} ما إذا كان العنصر مرئيًا
 */
function isVisible(element) {
    if (!element) return false;
    return element.offsetParent !== null && 
           window.getComputedStyle(element).display !== 'none' && 
           window.getComputedStyle(element).visibility !== 'hidden';
}

/**
 * تعطيل زر الإرسال أثناء المعالجة
 */
function disableButton() {
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.disabled = true;
        submitBtn.classList.add('loading');
        submitBtn.innerText = 'جاري الإرسال...';
        
        // إضافة مؤقت لمنع الضغط المتكرر (10 ثوانٍ)
        window.buttonTimer = 10;
        window.buttonInterval = setInterval(function() {
            window.buttonTimer--;
            if (window.buttonTimer <= 0) {
                clearInterval(window.buttonInterval);
                enableButton();
            } else {
                submitBtn.innerText = 'انتظر ' + window.buttonTimer + ' ثوانٍ...';
            }
        }, 1000);
    }
}

/**
 * إعادة تفعيل زر الإرسال
 */
function enableButton() {
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        // إذا كان هناك مؤقت نشط، قم بإلغائه
        if (window.buttonInterval) {
            clearInterval(window.buttonInterval);
            window.buttonInterval = null;
        }
        
        submitBtn.disabled = false;
        submitBtn.classList.remove('loading');
        submitBtn.innerText = 'اضغط للإرسال';
    }
}

/**
 * التحقق من تعبئة جميع الحقول المطلوبة
 */
function checkRequiredFieldsFilled() {
    const requiredInputs = document.querySelectorAll('input[required], select[required], textarea[required]');
    let allFilled = true;

    requiredInputs.forEach(input => {
        if (isVisible(input) && !input.value) {
            allFilled = false;
        }
    });

    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.disabled = !allFilled;
    }
}

/**
 * إرسال النموذج إلى Google Sheets والواتساب
 */
function sendFormToGoogleSheetsAndWhatsapp() {
    // الحصول على موقع المستخدم
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(function (position) {
            const lat = position.coords.latitude;
            const lon = position.coords.longitude;
            const locationInput = document.getElementById("location");
            if (locationInput) {
                locationInput.value = lat + "," + lon;
            }
            
            // التحقق من صحة النموذج قبل الإرسال
            if (!validateForm(lat, lon)) {
                enableButton();
                return;
            }
            
            // إرسال البيانات إلى Google Sheets
            sendDataToGoogleSheets();
            
            // إرسال البيانات إلى واتساب
            sendDataToWhatsapp(lat, lon);
        }, function (error) {
            console.error("Error getting location:", error.message);
            showAlert("خطأ في الحصول على الموقع الجغرافي. يرجى التأكد من تفعيل خدمة الموقع.", "danger");
            enableButton();
        });
    } else {
        showAlert("للأسف، متصفحك لا يدعم الحصول على الموقع الجغرافي.", "danger");
        enableButton();
    }
}

/**
 * إرسال البيانات إلى Google Sheets
 */
function sendDataToGoogleSheets() {
    const form = document.getElementById('form');
    if (form) {
        const formData = new FormData(form);
        fetch("https://script.google.com/macros/s/AKfycbyYvLzmN___6m52avxflN7NiMS0XJ0YfubZFqdnqPSnM0R6qv5waKYa7jHSatOCky9imQ/exec", {
            method: "POST",
            body: formData
        })
        .then(function (response) {
            console.log("Form data sent successfully.", response);
            // استخدام النافذة المنبثقة بدلاً من التنبيه البسيط
            showModal(
                "تم الإرسال بنجاح", 
                "<p>تم إرسال بيانات النموذج بنجاح إلى قاعدة البيانات.</p><p>شكراً لك!</p>", 
                "حسناً"
            );
        })
        .catch(function (error) {
            console.error("Error sending form data:", error);
            // استخدام النافذة المنبثقة للأخطاء
            showModal(
                "خطأ في الإرسال", 
                "<p>حدث خطأ أثناء إرسال البيانات.</p><p>يرجى المحاولة مرة أخرى لاحقاً.</p>", 
                "حسناً"
            );
            enableButton();
        });
    }
}

/**
 * إرسال البيانات إلى واتساب
 * @param {number} lat خط العرض
 * @param {number} lon خط الطول
 */
function sendDataToWhatsapp(lat, lon) {
    // استخدام الرسالة المجهزة من validateForm
    if (window.preparedMessage) {
        let message = window.preparedMessage;
        
        // إضافة الموقع إلى الرسالة
        const googleMapsLink = 'https://www.google.com/maps/place/' + lat + ',' + lon;
        message += '*📍 الموقع:* ' + googleMapsLink + '\n';
        
        // إنشاء رابط واتساب وإرساله
        const whatsappNumber = '+9647866866324';
        const whatsappLink = 'https://api.whatsapp.com/send?phone=' + whatsappNumber + '&text=' + encodeURIComponent(message);
        
        // عرض نافذة منبثقة للتأكيد قبل الإرسال
        showConfirmModal(
            "إرسال البيانات عبر واتساب",
            "هل تريد إرسال البيانات عبر واتساب الآن؟",
            function() {
                // استخدام window.open بدلاً من عنصر الرابط لتسريع العملية
                window.open(whatsappLink, '_blank');
                
                // إعادة تفعيل الزر بعد فترة قصيرة
                setTimeout(function() {
                    enableButton();
                }, 1000);
            },
            function() {
                // إعادة تفعيل الزر إذا تم الإلغاء
                enableButton();
            }
        );
    } else {
        showModal("خطأ في إعداد الرسالة", "<p>حدث خطأ في إعداد الرسالة.</p><p>يرجى المحاولة مرة أخرى.</p>", "حسناً");
        enableButton();
    }
}

/**
 * عرض تنبيه للمستخدم
 * @param {string} message نص الرسالة
 * @param {string} type نوع التنبيه (success, danger, etc.)
 */
function showAlert(message, type) {
    // إنشاء عنصر التنبيه
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type}`;
    alertDiv.textContent = message;
    
    // إضافة التنبيه إلى الصفحة
    const form = document.getElementById('form');
    if (form && form.parentNode) {
        form.parentNode.insertBefore(alertDiv, form);
        
        // إزالة التنبيه بعد 5 ثوانٍ
        setTimeout(function() {
            alertDiv.remove();
        }, 5000);
    }
}

/**
 * إنشاء وعرض نافذة منبثقة
 * @param {string} title عنوان النافذة المنبثقة
 * @param {string} content محتوى النافذة المنبثقة
 * @param {string} buttonText نص الزر (اختياري)
 * @param {Function} onConfirm دالة يتم تنفيذها عند النقر على زر التأكيد (اختياري)
 * @param {boolean} showCloseButton إظهار زر الإغلاق (اختياري، افتراضي: true)
 */
function showModal(title, content, buttonText = 'موافق', onConfirm = null, showCloseButton = true) {
    // إنشاء عناصر النافذة المنبثقة
    const overlay = document.createElement('div');
    overlay.className = 'overlay';
    overlay.id = 'customModal';
    
    const popup = document.createElement('div');
    popup.className = 'popup';
    
    // إضافة العنوان
    const titleElement = document.createElement('h4');
    titleElement.className = 'modal-title';
    titleElement.textContent = title;
    titleElement.style.marginBottom = '15px';
    titleElement.style.fontWeight = 'bold';
    popup.appendChild(titleElement);
    
    // إضافة المحتوى
    const contentElement = document.createElement('div');
    contentElement.className = 'modal-content';
    contentElement.innerHTML = content;
    contentElement.style.marginBottom = '20px';
    popup.appendChild(contentElement);
    
    // إضافة الأزرار
    const buttonsDiv = document.createElement('div');
    buttonsDiv.className = 'modal-buttons';
    buttonsDiv.style.display = 'flex';
    buttonsDiv.style.justifyContent = 'center';
    buttonsDiv.style.gap = '10px';
    
    // زر التأكيد
    const confirmButton = document.createElement('button');
    confirmButton.className = 'btn btn-primary';
    confirmButton.textContent = buttonText;
    confirmButton.addEventListener('click', function() {
        closeModal();
        if (onConfirm && typeof onConfirm === 'function') {
            onConfirm();
        }
    });
    buttonsDiv.appendChild(confirmButton);
    
    // زر الإغلاق (اختياري)
    if (showCloseButton) {
        const closeButton = document.createElement('button');
        closeButton.className = 'btn btn-secondary';
        closeButton.textContent = 'إغلاق';
        closeButton.addEventListener('click', closeModal);
        buttonsDiv.appendChild(closeButton);
    }
    
    popup.appendChild(buttonsDiv);
    overlay.appendChild(popup);
    
    // إضافة النافذة المنبثقة إلى الصفحة
    document.body.appendChild(overlay);
    
    // إضافة مستمع حدث للنقر خارج النافذة المنبثقة
    overlay.addEventListener('click', function(event) {
        if (event.target === overlay) {
            closeModal();
        }
    });
    
    // إضافة مستمع حدث للضغط على زر Escape
    document.addEventListener('keydown', handleEscapeKey);
}

/**
 * إغلاق النافذة المنبثقة
 */
function closeModal() {
    const modal = document.getElementById('customModal');
    if (modal) {
        modal.remove();
        // إزالة مستمع حدث الضغط على زر Escape
        document.removeEventListener('keydown', handleEscapeKey);
    }
}

/**
 * معالجة الضغط على زر Escape لإغلاق النافذة المنبثقة
 * @param {KeyboardEvent} event حدث لوحة المفاتيح
 */
function handleEscapeKey(event) {
    if (event.key === 'Escape') {
        closeModal();
    }
}

/**
 * عرض نافذة منبثقة للتأكيد
 * @param {string} title عنوان النافذة المنبثقة
 * @param {string} message رسالة التأكيد
 * @param {Function} onConfirm دالة يتم تنفيذها عند النقر على زر التأكيد
 * @param {Function} onCancel دالة يتم تنفيذها عند النقر على زر الإلغاء (اختياري)
 */
function showConfirmModal(title, message, onConfirm, onCancel = null) {
    // إنشاء عناصر النافذة المنبثقة
    const overlay = document.createElement('div');
    overlay.className = 'overlay';
    overlay.id = 'confirmModal';
    
    const popup = document.createElement('div');
    popup.className = 'popup';
    
    // إضافة العنوان
    const titleElement = document.createElement('h4');
    titleElement.className = 'modal-title';
    titleElement.textContent = title;
    titleElement.style.marginBottom = '15px';
    titleElement.style.fontWeight = 'bold';
    popup.appendChild(titleElement);
    
    // إضافة الرسالة
    const messageElement = document.createElement('div');
    messageElement.className = 'modal-content';
    messageElement.textContent = message;
    messageElement.style.marginBottom = '20px';
    popup.appendChild(messageElement);
    
    // إضافة الأزرار
    const buttonsDiv = document.createElement('div');
    buttonsDiv.className = 'modal-buttons';
    buttonsDiv.style.display = 'flex';
    buttonsDiv.style.justifyContent = 'center';
    buttonsDiv.style.gap = '10px';
    
    // زر التأكيد
    const confirmButton = document.createElement('button');
    confirmButton.className = 'btn btn-primary';
    confirmButton.textContent = 'تأكيد';
    confirmButton.addEventListener('click', function() {
        closeConfirmModal();
        if (onConfirm && typeof onConfirm === 'function') {
            onConfirm();
        }
    });
    buttonsDiv.appendChild(confirmButton);
    
    // زر الإلغاء
    const cancelButton = document.createElement('button');
    cancelButton.className = 'btn btn-secondary';
    cancelButton.textContent = 'إلغاء';
    cancelButton.addEventListener('click', function() {
        closeConfirmModal();
        if (onCancel && typeof onCancel === 'function') {
            onCancel();
        }
    });
    buttonsDiv.appendChild(cancelButton);
    
    popup.appendChild(buttonsDiv);
    overlay.appendChild(popup);
    
    // إضافة النافذة المنبثقة إلى الصفحة
    document.body.appendChild(overlay);
    
    // إضافة مستمع حدث للضغط على زر Escape
    document.addEventListener('keydown', handleConfirmEscapeKey);
}

/**
 * إغلاق نافذة التأكيد المنبثقة
 */
function closeConfirmModal() {
    const modal = document.getElementById('confirmModal');
    if (modal) {
        modal.remove();
        // إزالة مستمع حدث الضغط على زر Escape
        document.removeEventListener('keydown', handleConfirmEscapeKey);
    }
}

/**
 * معالجة الضغط على زر Escape لإغلاق نافذة التأكيد المنبثقة
 * @param {KeyboardEvent} event حدث لوحة المفاتيح
 */
function handleConfirmEscapeKey(event) {
    if (event.key === 'Escape') {
        closeConfirmModal();
    }
}

/**
 * التحقق من إذن النوافذ المنبثقة وعرض رسالة مناسبة للمستخدم
 */
function checkPopupPermission() {
    // التحقق مما إذا كان المستخدم قد اختار "دائمًا" من قبل
    if (localStorage.getItem('popupPermissionAlways') === 'true') {
        return; // لا تعرض الرسالة إذا اختار المستخدم "دائمًا"
    }
    
    // إنشاء نافذة منبثقة لطلب الإذن
    const overlay = document.createElement('div');
    overlay.className = 'overlay';
    overlay.id = 'popupPermissionModal';
    
    const popup = document.createElement('div');
    popup.className = 'popup';
    
    // إضافة العنوان
    const titleElement = document.createElement('h4');
    titleElement.className = 'modal-title';
    titleElement.textContent = 'إذن النوافذ المنبثقة';
    titleElement.style.marginBottom = '15px';
    titleElement.style.fontWeight = 'bold';
    popup.appendChild(titleElement);
    
    // إضافة المحتوى
    const contentElement = document.createElement('div');
    contentElement.className = 'modal-content';
    contentElement.innerHTML = '<p>يرجى السماح بالنوافذ المنبثقة لهذا الموقع للحصول على تجربة أفضل.</p><p>إذا كنت تواجه مشاكل في فتح النوافذ المنبثقة، يرجى التحقق من إعدادات المتصفح الخاص بك.</p>';
    contentElement.style.marginBottom = '20px';
    popup.appendChild(contentElement);
    
    // إضافة خيار "عدم الإظهار مرة أخرى"
    const checkboxDiv = document.createElement('div');
    checkboxDiv.style.marginBottom = '15px';
    checkboxDiv.style.textAlign = 'right';
    
    const checkbox = document.createElement('input');
    checkbox.type = 'checkbox';
    checkbox.id = 'dontShowAgain';
    checkbox.style.marginLeft = '8px';
    
    const checkboxLabel = document.createElement('label');
    checkboxLabel.htmlFor = 'dontShowAgain';
    checkboxLabel.textContent = 'عدم الإظهار مرة أخرى';
    
    checkboxDiv.appendChild(checkbox);
    checkboxDiv.appendChild(checkboxLabel);
    popup.appendChild(checkboxDiv);
    
    // إضافة الأزرار
    const buttonsDiv = document.createElement('div');
    buttonsDiv.className = 'modal-buttons';
    buttonsDiv.style.display = 'flex';
    buttonsDiv.style.justifyContent = 'center';
    buttonsDiv.style.gap = '10px';
    
    // زر الموافقة
    const confirmButton = document.createElement('button');
    confirmButton.className = 'btn btn-primary';
    confirmButton.textContent = 'موافق';
    confirmButton.addEventListener('click', function() {
        // حفظ خيار المستخدم إذا اختار "عدم الإظهار مرة أخرى"
        if (checkbox.checked) {
            localStorage.setItem('popupPermissionAlways', 'true');
        }
        closePopupPermissionModal();
    });
    buttonsDiv.appendChild(confirmButton);
    
    popup.appendChild(buttonsDiv);
    overlay.appendChild(popup);
    
    // إضافة النافذة المنبثقة إلى الصفحة
    document.body.appendChild(overlay);
    
    // إضافة مستمع حدث للنقر خارج النافذة المنبثقة
    overlay.addEventListener('click', function(event) {
        if (event.target === overlay) {
            closePopupPermissionModal();
        }
    });
    
    // إضافة مستمع حدث للضغط على زر Escape
    document.addEventListener('keydown', handlePopupPermissionEscapeKey);
}

/**
 * إغلاق نافذة إذن النوافذ المنبثقة
 */
function closePopupPermissionModal() {
    const modal = document.getElementById('popupPermissionModal');
    if (modal) {
        modal.remove();
        // إزالة مستمع حدث الضغط على زر Escape
        document.removeEventListener('keydown', handlePopupPermissionEscapeKey);
    }
}

/**
 * معالجة الضغط على زر Escape لإغلاق نافذة إذن النوافذ المنبثقة
 * @param {KeyboardEvent} event حدث لوحة المفاتيح
 */
function handlePopupPermissionEscapeKey(event) {
    if (event.key === 'Escape') {
        closePopupPermissionModal();
    }
}