<?php
require 'config.php';

// عند الطلب عبر AJAX لتحديث اسم الجهاز
if (isset($_GET['action']) && $_GET['action'] === 'rename') {
    header('Content-Type: application/json; charset=UTF-8');

    // تأكد من استلام القيم المطلوبة
    if (!isset($_POST['device_id']) || !isset($_POST['new_name']) || $_POST['device_id'] === '' || $_POST['new_name'] === '') {
        echo json_encode([
            'status'  => 'error',
            'message' => 'بيانات غير مكتملة لتحديث الاسم.'
        ]);
        exit;
    }

    $device_id = $_POST['device_id'];
    $new_name  = $_POST['new_name'];

    // حدث اسم الجهاز في جميع السجلات التي تحمل نفس الـdevice_id
    $stmt = $conn->prepare("UPDATE locations SET device_name = ? WHERE device_id = ?");
    $stmt->execute([$new_name, $device_id]);

    echo json_encode([
        'status'  => 'success',
        'message' => 'تم تحديث الاسم بنجاح!'
    ]);
    exit;
}

// جلب قائمة الأجهزة الفريدة من قاعدة البيانات
$stmt = $conn->query("SELECT DISTINCT device_id, device_name FROM locations ORDER BY device_name ASC");
$devices = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="ar">
<head>
    <meta charset="UTF-8"/>
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>لوحة تتبع المواقع GPS</title>

    <!-- مكتبات خريطة Leaflet -->
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.3/dist/leaflet.css"/>
    <script src="https://unpkg.com/leaflet@1.9.3/dist/leaflet.js"></script>

    <!-- مكتبات Mapbox للبحث العكسي عن الأماكن (Geocoding) -->
    <script src="https://api.mapbox.com/mapbox-gl-js/v2.6.1/mapbox-gl.js"></script>
    <link href="https://api.mapbox.com/mapbox-gl-js/v2.6.1/mapbox-gl.css" rel="stylesheet"/>

    <!-- مكتبات الخط الزمني Timeline -->
    <link rel="stylesheet" href="https://cdn.knightlab.com/libs/timeline3/latest/css/timeline.css">
    <script src="https://cdn.knightlab.com/libs/timeline3/latest/js/timeline.js"></script>

    <style>
        body {
            font-family: Arial, sans-serif;
            text-align: center;
            margin: 0;
            padding: 0;
            background-color: #f8f9fa;
        }
        .header {
            background: #007BFF;
            padding: 15px;
            color: white;
            font-size: 20px;
            font-weight: bold;
            text-align: center;
        }
        .controls {
            background: white;
            padding: 15px;
            display: flex;
            flex-wrap: wrap;
            justify-content: center;
            gap: 10px;
            align-items: center;
            border-bottom: 2px solid #ddd;
            box-shadow: 0px 2px 5px rgba(0, 0, 0, 0.1);
            position: sticky;
            top: 0;
            z-index: 1000;
        }
        select, input, button {
            padding: 10px;
            font-size: 16px;
            border-radius: 5px;
            border: 1px solid #ddd;
        }
        button {
            background-color: #007BFF;
            color: white;
            cursor: pointer;
            transition: 0.3s;
            border: none;
            font-weight: bold;
        }
        button:hover {
            background-color: #0056b3;
        }
        #map {
            height: 600px;
            width: 90%;
            margin: 20px auto;
            border-radius: 10px;
            box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.2);
        }
        .logs {
            width: 90%;
            margin: 20px auto;
            padding: 15px;
            background: white;
            border-radius: 10px;
            box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.2);
            text-align: left;
        }
        .logs h2 {
            margin-top: 0;
        }
        .logs table {
            width: 100%;
            border-collapse: collapse;
        }
        .logs table, .logs th, .logs td {
            border: 1px solid #ddd;
        }
        .logs th, .logs td {
            padding: 10px;
            text-align: center;
        }
        #timeline {
            width: 90%;
            margin: 20px auto;
            height: 300px;
            border-radius: 10px;
            box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.2);
        }
        .rename-section {
            display: flex;
            flex-direction: row;
            align-items: center;
            gap: 8px;
        }
    </style>
</head>
<body>

<div class="header">🚀 لوحة تتبع المواقع GPS</div>

<div class="controls">
    <!-- اختيار الجهاز والبحث -->
    <label>🔽 اختر الجهاز:</label>
    <select id="deviceSelect">
        <option value="">جميع الأجهزة</option>
        <?php foreach ($devices as $device): ?>
            <option value="<?= $device['device_id'] ?>">
                <?= htmlspecialchars($device['device_name']) ?>
            </option>
        <?php endforeach; ?>
    </select>

    <label>📅 اختر التاريخ:</label>
    <input type="date" id="dateSelect">

    <!-- استبدال إدخال الوقت اليدوي بقوائم منسدلة -->
    <label>⏰ من (12h):</label>
    <select id="hourFrom">
        <?php 
          // ساعات 1 إلى 12
          for ($h = 1; $h <= 12; $h++) {
              $val = str_pad($h, 2, '0', STR_PAD_LEFT);
              echo "<option value='$val'>$val</option>";
          }
        ?>
    </select>
    <select id="minFrom">
        <?php
          // دقائق 00 إلى 59
          for ($m = 0; $m < 60; $m += 5) { 
              // يمكنك تغيير القفزة إلى دقيقة واحدة إن أردت
              $val = str_pad($m, 2, '0', STR_PAD_LEFT);
              echo "<option value='$val'>$val</option>";
          }
        ?>
    </select>
    <select id="ampmFrom">
        <option value="AM">AM</option>
        <option value="PM">PM</option>
    </select>

    <label>⏰ إلى (12h):</label>
    <select id="hourTo">
        <?php 
          // ساعات 1 إلى 12
          for ($h = 1; $h <= 12; $h++) {
              $val = str_pad($h, 2, '0', STR_PAD_LEFT);
              echo "<option value='$val'>$val</option>";
          }
        ?>
    </select>
    <select id="minTo">
        <?php
          // دقائق 00 إلى 59
          for ($m = 0; $m < 60; $m += 5) {
              $val = str_pad($m, 2, '0', STR_PAD_LEFT);
              echo "<option value='$val'>$val</option>";
          }
        ?>
    </select>
    <select id="ampmTo">
        <option value="AM">AM</option>
        <option value="PM">PM</option>
    </select>

    <button onclick="fetchLocations()">🔍 بحث</button>
    <button onclick="resetFilters()">🔄 إعادة تعيين</button>
    <button onclick="fetchLatestLocation()">📡 آخر موقع مباشر</button>
</div>

<div class="controls">
    <!-- قسم تعديل اسم الجهاز -->
    <div class="rename-section">
        <label>✏️ تعديل اسم الجهاز:</label>
        <input type="text" id="newDeviceName" placeholder="اكتب الاسم الجديد...">
        <button onclick="updateDeviceName()">💾 حفظ</button>
    </div>
</div>

<div id="map"></div>

<div class="logs">
    <h2>سجلات الدخول والخروج (توقفات 5 دقائق فما فوق)</h2>
    <table id="logsTable">
        <thead>
        <tr>
            <th>اسم المكان</th>
            <th>وقت الدخول</th>
            <th>وقت الخروج</th>
            <th>الحالة</th>
        </tr>
        </thead>
        <tbody></tbody>
    </table>
</div>

<div id="timeline"></div>

<script>
    // ------------------------------------------------------------------
    // مفاتيح وخيارات
    // ------------------------------------------------------------------
    const accessToken = "pk.eyJ1IjoiaWJyYWhpbWNvbSIsImEiOiJja2YxZnB6OWowc2xxMnFxZGxya2ppdmZxIn0.keECYQuu1Bs0JeWDP8dSFQ";

    // ------------------------------------------------------------------
    // تهيئة خريطتي (تضاريس/قمر صناعي) من Mapbox
    // ------------------------------------------------------------------
    const outdoorsLayer = L.tileLayer(
        `https://api.mapbox.com/styles/v1/mapbox/outdoors-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
        {
            maxZoom: 20,
            tileSize: 512,
            zoomOffset: -1,
            attribution: '&copy; <a href="https://www.openstreetmap.org/">OSM</a> & <a href="https://www.mapbox.com/">Mapbox</a>'
        }
    );

    const satelliteLayer = L.tileLayer(
        `https://api.mapbox.com/styles/v1/mapbox/satellite-streets-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
        {
            maxZoom: 20,
            tileSize: 512,
            zoomOffset: -1,
            attribution: '&copy; <a href="https://www.openstreetmap.org/">OSM</a> & <a href="https://www.mapbox.com/">Mapbox</a>'
        }
    );

    // إنشاء الخريطة
    var map = L.map('map', {
        center: [30, 0],
        zoom: 2,
        layers: [outdoorsLayer] // الطبقة الافتراضية
    });

    // تحكم بالطبقات
    var baseMaps = {
        "خريطة التضاريس": outdoorsLayer,
        "خريطة القمر الصناعي": satelliteLayer
    };
    L.control.layers(baseMaps).addTo(map);

    // ------------------------------------------------------------------
    // أيقونات
    // ------------------------------------------------------------------
    var parkIcon = L.icon({
        iconUrl: 'https://almanara.space/gps_tracker/uploads/image.png',
        iconSize: [32, 32],
        iconAnchor: [16, 32],
        popupAnchor: [0, -32]
    });
    var defaultIcon = L.icon({
        iconUrl: 'https://almanara.space/gps_tracker/uploads/car.png',
        iconSize: [32, 32],
        iconAnchor: [16, 32],
        popupAnchor: [0, -32]
    });
    var startIcon = L.icon({
        iconUrl: 'https://almanara.space/gps_tracker/uploads/start.png',
        iconSize: [60, 60],
        iconAnchor: [16, 32]
    });
    var endIcon = L.icon({
        iconUrl: 'https://almanara.space/gps_tracker/uploads/end.png',
        iconSize: [60, 60],
        iconAnchor: [16, 32]
    });
    var latestIcon = L.icon({
        iconUrl: 'https://cdn-icons-png.flaticon.com/512/684/684908.png',
        iconSize: [32, 32],
        iconAnchor: [16, 32]
    });

    // ------------------------------------------------------------------
    // متغيرات عامة
    // ------------------------------------------------------------------
    var markers   = [];
    var circles   = [];
    var polylines = [];
    var logs      = [];
    var timelineData = [];
    var markerCounter = 1;

    var latestMarker           = null;
    var latestLocationInterval = null;
    var lastTimestamp          = null;

    // ------------------------------------------------------------------
    // دوال عامة
    // ------------------------------------------------------------------
    function clearMap() {
        markers.forEach(m => map.removeLayer(m));
        markers = [];

        circles.forEach(c => map.removeLayer(c));
        circles = [];

        polylines.forEach(p => map.removeLayer(p));
        polylines = [];

        if (latestMarker) {
            map.removeLayer(latestMarker);
            latestMarker = null;
        }
        if (latestLocationInterval) {
            clearTimeout(latestLocationInterval);
            latestLocationInterval = null;
        }

        logs = [];
        timelineData = [];
        markerCounter = 1;

        updateLogsTable();
        updateTimeline();
    }

    function resetFilters() {
        document.getElementById("deviceSelect").value = "";
        document.getElementById("dateSelect").value   = "";
        
        // إعادة القوائم المنسدلة للقيم الافتراضية (01:00 AM ..)
        document.getElementById("hourFrom").value  = "01";
        document.getElementById("minFrom").value   = "00";
        document.getElementById("ampmFrom").value  = "AM";
        document.getElementById("hourTo").value    = "01";
        document.getElementById("minTo").value     = "00";
        document.getElementById("ampmTo").value    = "PM";

        clearMap();
    }

    // ------------------------------------------------------------------
    // جلب المواقع حسب الفلاتر
    // ------------------------------------------------------------------
    function fetchLocations() {
        clearMap();

        var deviceId = document.getElementById("deviceSelect").value;
        var date     = document.getElementById("dateSelect").value;

        // نجمع الوقت من القوائم المنسدلة ونحوله إلى صيغة 12h ثم 24h
        var hourFrom   = document.getElementById("hourFrom").value;
        var minFrom    = document.getElementById("minFrom").value;
        var ampmFrom   = document.getElementById("ampmFrom").value;
        var timeFrom12 = hourFrom + ":" + minFrom + " " + ampmFrom;
        var timeFrom   = convertTo24Hour(timeFrom12);

        var hourTo   = document.getElementById("hourTo").value;
        var minTo    = document.getElementById("minTo").value;
        var ampmTo   = document.getElementById("ampmTo").value;
        var timeTo12 = hourTo + ":" + minTo + " " + ampmTo;
        var timeTo   = convertTo24Hour(timeTo12);

        var url = "get_locations.php?";
        if (deviceId) url += "device_id=" + encodeURIComponent(deviceId) + "&";
        if (date)     url += "date="      + encodeURIComponent(date)     + "&";
        if (timeFrom) url += "time_from=" + encodeURIComponent(timeFrom) + "&";
        if (timeTo)   url += "time_to="   + encodeURIComponent(timeTo)   + "&";
        url = url.slice(0, -1);

        fetch(url)
            .then(res => res.json())
            .then(data => {
                console.log("📍 المواقع المسترجعة:", data);
                if (Array.isArray(data) && data.length > 0) {
                    // ترتيب البيانات تصاعدياً بالزمن
                    let dataSortedAsc = data.slice().sort((a,b) => new Date(a.timestamp) - new Date(b.timestamp));
                    let startLoc = dataSortedAsc[0];
                    let endLoc   = dataSortedAsc[dataSortedAsc.length - 1];

                    // ماركر بداية
                    let startLat = parseFloat(startLoc.latitude);
                    let startLng = parseFloat(startLoc.longitude);
                    if (!isNaN(startLat) && !isNaN(startLng)) {
                        getPlaceName(startLat, startLng).then(startPlaceName => {
                            let markerStart = L.marker([startLat, startLng], { icon: startIcon })
                                .addTo(map)
                                .bindPopup(
                                    `<b>بداية المسار</b><br>
                                     الوقت: ${formatTo12Hour(startLoc.timestamp)}<br>
                                     ${startPlaceName}`
                                );
                            markers.push(markerStart);
                        });
                    }

                    // ماركر نهاية
                    let endLat = parseFloat(endLoc.latitude);
                    let endLng = parseFloat(endLoc.longitude);
                    if (!isNaN(endLat) && !isNaN(endLng)) {
                        getPlaceName(endLat, endLng).then(endPlaceName => {
                            let markerEnd = L.marker([endLat, endLng], { icon: endIcon })
                                .addTo(map)
                                .bindPopup(
                                    `<b>نهاية المسار</b><br>
                                     الوقت: ${formatTo12Hour(endLoc.timestamp)}<br>
                                     ${endPlaceName}`
                                );
                            markers.push(markerEnd);
                        });
                    }

                    // تجميع النقاط في دوائر
                    var groupedLocations = groupLocations(data, 30);
                    groupedLocations.sort((a, b) => new Date(b.entryTime) - new Date(a.entryTime));

                    var routePoints = [];

                    groupedLocations.forEach(group => {
                        var lat = group.center[0];
                        var lng = group.center[1];

                        if (!isNaN(lat) && !isNaN(lng)) {
                            // نضيف كل النقاط إلى المسار (Polyline) بغض النظر عن الحالة
                            routePoints.push([lat, lng]);

                            // في حالة "بارك توقف" فقط، أضف دائرة وأيقونة
                            // ------------------------------------------------
                            if (group.status === "بارك توقف") {
                                var circle = L.circle([lat, lng], {
                                    radius: 30,
                                    color: 'red',
                                    fillColor: '#f03',
                                    fillOpacity: 0.5
                                }).addTo(map);
                                circles.push(circle);

                                // أيقونة (موقف أو عادي)
                                var icon = parkIcon; // لأن الحالة بارك توقف
                                var marker = L.marker([lat, lng], { icon: icon })
                                    .addTo(map)
                                    .bindPopup(
                                        `<b>النقطة ${markerCounter}</b><br>
                                         وقت الدخول: ${formatTo12Hour(group.entryTime)}<br>
                                         وقت الخروج: ${formatTo12Hour(group.exitTime)}<br>
                                         الحالة: ${group.status}`
                                    );
                                markers.push(marker);
                                markerCounter++;

                                // إضافة إلى السجلات
                                getPlaceName(lat, lng).then(placeName => {
                                    logs.push({
                                        placeName: placeName,
                                        entryTime: group.entryTime,
                                        exitTime: group.exitTime,
                                        center: [lat, lng],
                                        status: group.status
                                    });
                                    timelineData.push({
                                        start_date: group.entryTime,
                                        end_date: group.exitTime,
                                        text: {
                                            headline: placeName,
                                            text: `
                                                وقت الدخول: ${formatTo12Hour(group.entryTime)}<br>
                                                وقت الخروج: ${formatTo12Hour(group.exitTime)}<br>
                                                الحالة: ${group.status}
                                            `
                                        }
                                    });
                                    updateLogsTable();
                                    updateTimeline();
                                });
                            }
                            // ------------------------------------------------
                            // في حالة "تحرك" لن نعرض ماركر أو دائرة
                            // ------------------------------------------------
                        }
                    });

                    // رسم المسار Polyline
                    if (routePoints.length > 1) {
                        var polyline = L.polyline(routePoints, { color: 'blue', weight: 5 }).addTo(map);
                        polylines.push(polyline);
                    }

                    // ضبط الخريطة حسب الحدود
                    if (routePoints.length > 0) {
                        var bounds = new L.LatLngBounds(routePoints);
                        map.fitBounds(bounds);
                    }

                } else {
                    alert("❌ لا توجد بيانات ضمن الشروط المحددة.");
                }
            })
            .catch(err => console.error("⚠️ خطأ في جلب البيانات:", err));
    }

    // ------------------------------------------------------------------
    // دوال تحويل الوقت والتجميع
    // ------------------------------------------------------------------
    function convertTo24Hour(timeStr) {
        if (!timeStr) return null;
        var parts = timeStr.split(':');
        if (parts.length < 2) return null;

        var hours = parseInt(parts[0], 10);
        var minutesPart = parts[1].trim();
        var minutes = parseInt(minutesPart, 10);
        var ampm = minutesPart.slice(-2).toUpperCase();  // "PM" أو "AM"
        if (ampm !== 'AM' && ampm !== 'PM') {
            // إن لم يجد AM/PM ربما أدخل المستخدم صيغة 24 مباشرة
            return timeStr + ":00";
        }
        if (ampm === 'PM' && hours < 12) hours += 12;
        if (ampm === 'AM' && hours === 12) hours = 0;

        var hh = (hours < 10) ? '0' + hours : hours;
        var mm = (minutes < 10) ? '0' + minutes : minutes;
        return hh + ":" + mm + ":00";
    }

    function formatTo12Hour(dateStr) {
        if (!dateStr) return '';
        var d = new Date(dateStr);
        if (isNaN(d.getTime())) return dateStr;
        return d.toLocaleString('en-US', {
            hour12: true,
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: 'numeric',
            minute: '2-digit',
            second: '2-digit'
        });
    }

    function groupLocations(locations, radius) {
        var groups = [];
        var usedIndices = new Set();

        for (var i = 0; i < locations.length; i++) {
            if (!usedIndices.has(i)) {
                var loc = locations[i];
                var group = {
                    points: [loc],
                    center: [parseFloat(loc.latitude), parseFloat(loc.longitude)],
                    entryTime: loc.timestamp,
                    exitTime: loc.timestamp
                };

                for (var j = i + 1; j < locations.length; j++) {
                    if (!usedIndices.has(j)) {
                        var loc2 = locations[j];
                        var distance = getDistance(
                            group.center[0], group.center[1],
                            parseFloat(loc2.latitude), parseFloat(loc2.longitude)
                        );
                        if (distance <= radius) {
                            group.points.push(loc2);
                            if (new Date(loc2.timestamp) > new Date(group.exitTime)) {
                                group.exitTime = loc2.timestamp;
                            }
                            usedIndices.add(j);
                        }
                    }
                }

                if (group.points.length > 1) {
                    var diffMin = (new Date(group.exitTime) - new Date(group.entryTime)) / 60000;
                    group.status = (diffMin >= 5) ? "بارك توقف" : "تحرك";
                } else {
                    group.status = "بداية";
                }
                groups.push(group);
                usedIndices.add(i);
            }
        }
        return groups;
    }

    function getDistance(lat1, lon1, lat2, lon2) {
        var R = 6371e3; // نصف قطر الأرض بالمتر
        var φ1 = lat1 * Math.PI / 180;
        var φ2 = lat2 * Math.PI / 180;
        var Δφ = (lat2 - lat1) * Math.PI / 180;
        var Δλ = (lon2 - lon1) * Math.PI / 180;

        var a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
                Math.cos(φ1) * Math.cos(φ2) *
                Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
        var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c;
    }

    function getPlaceName(lat, lng) {
        var url = `https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${accessToken}`;
        return fetch(url)
            .then(res => res.json())
            .then(data => {
                if (data.features && data.features.length > 0) {
                    return data.features[0].place_name;
                }
                return "مكان غير معروف";
            })
            .catch(_ => "مكان غير معروف");
    }

    // ------------------------------------------------------------------
    // سجلات الدخول/الخروج في الجدول + الخط الزمني
    // ------------------------------------------------------------------
    function updateLogsTable() {
        var tableBody = document.getElementById("logsTable").getElementsByTagName("tbody")[0];
        tableBody.innerHTML = "";

        var filteredLogs = logs.filter(log => {
            var diff = (new Date(log.exitTime) - new Date(log.entryTime)) / 60000;
            return diff >= 5; // فقط توقفات 5 دقائق فما فوق
        });
        filteredLogs.sort((a, b) => new Date(b.entryTime) - new Date(a.entryTime));

        filteredLogs.forEach(log => {
            var row = tableBody.insertRow();
            row.insertCell(0).innerHTML = `<a href="#" onclick="focusOnLocation([${log.center}])">${log.placeName}</a>`;
            row.insertCell(1).textContent = formatTo12Hour(log.entryTime);
            row.insertCell(2).textContent = formatTo12Hour(log.exitTime);
            row.insertCell(3).textContent = log.status;
        });
    }

    function updateTimeline() {
        var filteredTimeline = timelineData.filter(evt => {
            var diff = (new Date(evt.end_date) - new Date(evt.start_date)) / 60000;
            return diff >= 5;
        });
        var timeline = new TL.Timeline('timeline', { events: filteredTimeline });
    }

    function focusOnLocation(center) {
        map.setView(center, 15);
    }

    // ------------------------------------------------------------------
    // جلب آخر موقع مباشر (زر منفصل)
    // ------------------------------------------------------------------
    function fetchLatestLocation() {
        var deviceId = document.getElementById("deviceSelect").value;
        if (!deviceId) {
            alert("⚠️ يرجى اختيار جهاز لجلب أحدث موقع.");
            return;
        }

        if (latestLocationInterval) {
            clearTimeout(latestLocationInterval);
        }

        function getLatest() {
            var url = `get_latest_location.php?device_id=${deviceId}&t=${Date.now()}`;
            fetch(url)
                .then(res => res.json())
                .then(data => {
                    console.log("📡 آخر بيانات:", data);
                    if (data.status === "success") {
                        var latest = data.data;
                        var lat = parseFloat(latest.latitude);
                        var lng = parseFloat(latest.longitude);
                        var timestamp = latest.timestamp;

                        if (!isNaN(lat) && !isNaN(lng)) {
                            // إذا تغير التوقيت
                            if (timestamp !== lastTimestamp) {
                                lastTimestamp = timestamp;
                                if (latestMarker) {
                                    map.removeLayer(latestMarker);
                                }
                                latestMarker = L.marker([lat, lng], {
                                    icon: latestIcon
                                }).addTo(map).bindPopup(`
                                    <b>📡 آخر موقع</b><br>
                                    الجهاز: ${latest.device_name}<br>
                                    الوقت: ${formatTo12Hour(latest.timestamp)}
                                `).openPopup();

                                var circle = L.circle([lat, lng], {
                                    radius: 30,
                                    color: 'blue',
                                    fillColor: '#00f',
                                    fillOpacity: 0.5
                                }).addTo(map);
                                circles.push(circle);

                                map.setView([lat, lng], 15);
                            }
                        }
                    } else {
                        console.warn("❌ لا يوجد موقع متاح لهذا الجهاز.");
                    }
                    latestLocationInterval = setTimeout(getLatest, 10000);
                })
                .catch(err => console.error("⚠️ خطأ في جلب آخر موقع:", err));
        }

        getLatest();
    }

    // ------------------------------------------------------------------
    // دالة تحديث اسم الجهاز عبر AJAX
    // ------------------------------------------------------------------
    function updateDeviceName() {
        var deviceId = document.getElementById("deviceSelect").value;
        var newName  = document.getElementById("newDeviceName").value.trim();

        if (!deviceId) {
            alert("الرجاء اختيار جهاز من القائمة أولاً!");
            return;
        }
        if (!newName) {
            alert("الرجاء إدخال الاسم الجديد!");
            return;
        }

        var formData = new FormData();
        formData.append("device_id", deviceId);
        formData.append("new_name", newName);

        fetch("dashboard.php?action=rename", {
            method: "POST",
            body: formData
        })
        .then(res => res.json())
        .then(data => {
            if (data.status === "success") {
                alert(data.message);
                // إعادة تحميل الصفحة لتحديث أسماء الأجهزة
                location.reload();
            } else {
                alert(data.message || "فشل تحديث الاسم!");
            }
        })
        .catch(err => {
            console.error(err);
            alert("خطأ في الاتصال بالسيرفر!");
        });
    }

    // اختبار اختيار الجهاز
    document.getElementById("deviceSelect").addEventListener("change", function() {
        console.log("تم اختيار الجهاز UUID:", this.value);
    });
</script>
</body>
</html>
