<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *"); // للسماح بالطلبات من أي مصدر
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

require 'db_connect.php';

// التأكد من أن الاتصال بقاعدة البيانات يستخدم UTF-8
$conn->set_charset("utf8mb4");

// التعامل مع طلب OPTIONS (preflight request)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// استقبال البيانات من الطلب (نتوقع أن تكون البيانات مرسلة كـ JSON)
$data = json_decode(file_get_contents("php://input"), true);

if (!$data) {
    error_log("لم يتم استلام أي بيانات في delete_locations.php");
    echo json_encode(["status" => "error", "message" => "❌ لم يتم استلام أي بيانات"]);
    $conn->close();
    exit;
}

error_log("البيانات المستقبلة في delete_locations.php: " . print_r($data, true));

// التحقق من وجود الحقول المطلوبة
if (!isset($data['start_date']) || !isset($data['end_date'])) {
    error_log("بيانات ناقصة: start_date أو end_date غير موجودة");
    echo json_encode(["status" => "error", "message" => "❌ بيانات ناقصة: يرجى تحديد تاريخ البدء وتاريخ الانتهاء"]);
    $conn->close();
    exit;
}

$start_date_str = $data['start_date'];
$end_date_str = $data['end_date'];

// التحقق من صحة تنسيق التاريخ (YYYY-MM-DD)
if (!preg_match("/^\d{4}-\d{2}-\d{2}$/", $start_date_str) || !preg_match("/^\d{4}-\d{2}-\d{2}$/", $end_date_str)) {
    error_log("تنسيق تاريخ غير صالح: " . $start_date_str . " أو " . $end_date_str);
    echo json_encode(["status" => "error", "message" => "❌ تنسيق تاريخ غير صالح. يرجى استخدام YYYY-MM-DD"]);
    $conn->close();
    exit;
}

// تحويل التواريخ إلى كائنات DateTime لإضافة الوقت
try {
    $start_datetime = new DateTime($start_date_str . ' 00:00:00');
    $end_datetime = new DateTime($end_date_str . ' 23:59:59');
} catch (Exception $e) {
    error_log("خطأ في تحويل التاريخ: " . $e->getMessage());
    echo json_encode(["status" => "error", "message" => "❌ خطأ في معالجة التواريخ"]);
    $conn->close();
    exit;
}

// تنسيق التواريخ لتناسب عمود timestamp الذي يخزن الوقت بتنسيق 12 ساعة مع AM/PM
// مثال: '2024-05-01 12:00:00 AM'
// بما أن عمود timestamp في جدول locations يخزن التاريخ والوقت مع AM/PM،
// يجب أن نكون حذرين عند المقارنة.
// الطريقة الأفضل هي تحويل عمود timestamp إلى تنسيق DATETIME قياسي في الاستعلام
// أو التأكد من أن التواريخ المدخلة تغطي اليوم بأكمله.

// استخدام التواريخ كما هي مع إضافة الوقت (بداية اليوم ونهايته)
$start_timestamp_for_query = $start_datetime->format('Y-m-d H:i:s'); // سيكون 00:00:00
$end_timestamp_for_query = $end_datetime->format('Y-m-d H:i:s');   // سيكون 23:59:59

// تحضير الاستعلام
// نفترض أن عمود `timestamp` هو من نوع DATETIME أو TIMESTAMP أو VARCHAR يخزن التاريخ والوقت
// إذا كان VARCHAR ويخزن التوقيت بتنسيق AM/PM، قد نحتاج إلى STR_TO_DATE
// $sql = "DELETE FROM locations WHERE STR_TO_DATE(timestamp, '%Y-%m-%d %h:%i:%s %p') >= ? AND STR_TO_DATE(timestamp, '%Y-%m-%d %h:%i:%s %p') <= ?";

// إذا كان عمود timestamp من نوع DATETIME أو TIMESTAMP قياسي، يمكننا المقارنة مباشرة
// ولكن بما أن add_location.php يحفظه بتنسيق 'Y-m-d h:i:s A', فالأفضل تحويله في الاستعلام
$sql = "DELETE FROM locations WHERE STR_TO_DATE(timestamp, '%Y-%m-%d %h:%i:%s %p') BETWEEN ? AND ?";

$stmt = $conn->prepare($sql);

if ($stmt === false) {
    error_log("خطأ في تحضير الاستعلام: " . $conn->error);
    echo json_encode(["status" => "error", "message" => "⚠️ خطأ في تحضير الاستعلام: " . $conn->error]);
    $conn->close();
    exit;
}

$stmt->bind_param("ss", $start_timestamp_for_query, $end_timestamp_for_query);

if ($stmt->execute()) {
    $affected_rows = $stmt->affected_rows;
    if ($affected_rows > 0) {
        echo json_encode(["status" => "success", "message" => "✅ تم حذف " . $affected_rows . " سجل بنجاح"]);
    } else {
        echo json_encode(["status" => "success", "message" => "ℹ️ لم يتم العثور على سجلات للحذف في الفترة المحددة"]);
    }
} else {
    error_log("خطأ في تنفيذ الاستعلام: " . $stmt->error);
    echo json_encode(["status" => "error", "message" => "⚠️ خطأ في تنفيذ الحذف: " . $stmt->error]);
}

$stmt->close();
$conn->close();
?>
