const sheetName = 'Sheet1'; // تأكد من أن هذا هو الاسم الصحيح لعلامة تبويب الورقة الخاصة بك
const settingsSheetName = 'Settings'; // ورقة الإعدادات
const scriptProp = PropertiesService.getScriptProperties();

// قم بتشغيل هذه الدالة مرة واحدة من محرر السكربت لإعداد خاصية معرف جدول البيانات
const initialSetup = () => {
  const activeSpreadsheet = SpreadsheetApp.getActiveSpreadsheet();
  scriptProp.setProperty('key', activeSpreadsheet.getId());
  
  // إنشاء ورقة الإعدادات إذا لم تكن موجودة
  let settingsSheet = activeSpreadsheet.getSheetByName(settingsSheetName);
  if (!settingsSheet) {
    settingsSheet = activeSpreadsheet.insertSheet(settingsSheetName);
    // إضافة رؤوس الأعمدة
    settingsSheet.getRange(1, 1, 1, 5).setValues([['UserID', 'ScriptUrl', 'RepresentativeName', 'Governorates', 'District']]);
  }
  
  Logger.log('تم تعيين معرف جدول البيانات: ' + activeSpreadsheet.getId());
}

// التعامل مع طلبات GET - استرجاع الإعدادات والبحث
const doGet = (e) => {
  try {
    // معالجة طلبات OPTIONS للـ CORS
    if (e.parameter.method === 'OPTIONS') {
      const output = ContentService.createTextOutput('');
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    const action = e.parameter.action;
    const userId = e.parameter.userId;
    
    if (action === 'getSettings' && userId) {
      const settings = getSettings(userId);
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': settings }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    // البحث عن المستشفيات/الأطباء/المختبرات
    if (action === 'searchHospitals') {
      const searchTerm = e.parameter.searchTerm;
      const results = searchHospitals(searchTerm);
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': results }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    // الحصول على بيانات مستشفى/طبيب/مختبر محدد
    if (action === 'getHospitalData') {
      const hospitalName = e.parameter.hospitalName;
      const data = getHospitalData(hospitalName);
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': data }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    // الحصول على بيانات جهة الاتصال
    if (action === 'getContactPersonData') {
      const contactPersonName = e.parameter.contactPersonName;
      const data = getContactPersonData(contactPersonName);
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': data }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    // البحث عن جهات الاتصال
    if (action === 'searchContactPersons') {
      const searchTerm = e.parameter.searchTerm;
      const results = searchContactPersons(searchTerm);
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': results }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    // أداة التشخيص الشاملة
    if (action === 'diagnosticCheck') {
      const diagnosticResult = runDiagnosticCheck();
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': diagnosticResult }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }

    // البحث المتقدم باستخدام اسم المستشفى وجهة الاتصال
    if (action === 'searchByHospitalAndContact') {
      const hospitalName = e.parameter.hospitalName;
      const contactPersonName = e.parameter.contactPersonName;
      const data = getDataByHospitalAndContact(hospitalName, contactPersonName);
      const output = ContentService.createTextOutput(JSON.stringify({ 'result': 'success', 'data': data }));
      output.setMimeType(ContentService.MimeType.JSON);
      return output;
    }
    
    return ContentService
      .createTextOutput(JSON.stringify({ 'result': 'error', 'message': 'Invalid parameters' }))
      .setMimeType(ContentService.MimeType.JSON);
      
  } catch (error) {
    Logger.log('خطأ في doGet: ' + error.toString());
    return ContentService
      .createTextOutput(JSON.stringify({ 'result': 'error', 'message': error.toString() }))
      .setMimeType(ContentService.MimeType.JSON);
  }
}

// دالة لحفظ الإعدادات
const saveSettings = (userId, scriptUrl, representativeName, governorates, district) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  let settingsSheet = doc.getSheetByName(settingsSheetName);
  
  if (!settingsSheet) {
    settingsSheet = doc.insertSheet(settingsSheetName);
    settingsSheet.getRange(1, 1, 1, 5).setValues([['UserID', 'ScriptUrl', 'RepresentativeName', 'Governorates', 'District']]);
  }
  
  // البحث عن المستخدم الموجود
  const data = settingsSheet.getDataRange().getValues();
  let userRowIndex = -1;
  
  for (let i = 1; i < data.length; i++) {
    if (data[i][0] === userId) {
      userRowIndex = i + 1;
      break;
    }
  }
  
  const newRow = [userId, scriptUrl, representativeName, governorates, district];
  
  if (userRowIndex > 0) {
    // تحديث الصف الموجود
    settingsSheet.getRange(userRowIndex, 1, 1, 5).setValues([newRow]);
  } else {
    // إضافة صف جديد
    settingsSheet.appendRow(newRow);
  }
}

// دالة لاسترجاع الإعدادات
const getSettings = (userId) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  const settingsSheet = doc.getSheetByName(settingsSheetName);
  
  if (!settingsSheet) {
    return null;
  }
  
  const data = settingsSheet.getDataRange().getValues();
  
  for (let i = 1; i < data.length; i++) {
    if (data[i][0] === userId) {
      return {
        scriptUrl: data[i][1],
        representativeName: data[i][2],
        governorates: data[i][3],
        district: data[i][4]
      };
    }
  }
  
  console.log('❌ لم يتم العثور على مطابقة للمستخدم:', userId);
  
  return null;
}

// دالة التشخيص الشاملة لفحص بنية الجدول ومشاكل البيانات
const runDiagnosticCheck = () => {
  try {
    const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
    const sheet = doc.getSheetByName(sheetName);
    
    if (!sheet) {
      return {
        status: 'error',
        message: 'لم يتم العثور على الورقة المحددة: ' + sheetName,
        details: {}
      };
    }
    
    const data = sheet.getDataRange().getValues();
    const headers = data[0];
    
    // فحص الرؤوس والأعمدة
    const columnAnalysis = {
      totalColumns: headers.length,
      totalRows: data.length - 1, // استثناء صف الرؤوس
      headers: headers,
      contactPersonColumns: []
    };
    
    // البحث عن جميع الأعمدة المشابهة لـ "اسم من تحدثت معه"
    const possibleContactPersonNames = [
      'اسم من تحدثت معه',
      'اسم من تحدثت معه ',
      ' اسم من تحدثت معه',
      'اسم الشخص المسؤول',
      'جهة الاتصال',
      'اسم المسؤول',
      'الشخص المسؤول'
    ];
    
    headers.forEach((header, index) => {
      const cleanHeader = header.toString().trim();
      if (possibleContactPersonNames.some(name => cleanHeader.includes(name.trim()) || name.trim().includes(cleanHeader))) {
        columnAnalysis.contactPersonColumns.push({
          index: index,
          name: header,
          cleanName: cleanHeader,
          exactMatch: possibleContactPersonNames.includes(cleanHeader)
        });
      }
    });
    
    // فحص عينة من البيانات
    const sampleData = [];
    const maxSamples = Math.min(5, data.length - 1);
    
    for (let i = 1; i <= maxSamples; i++) {
      const rowData = {
        rowNumber: i + 1,
        hospitalName: data[i][headers.indexOf('اسم المستشفى او الطبيب او المختبر او المركز ... الخ')] || '',
        contactPersonData: {}
      };
      
      // جمع بيانات جميع أعمدة جهات الاتصال المحتملة
      columnAnalysis.contactPersonColumns.forEach(col => {
        rowData.contactPersonData[col.name] = {
          value: data[i][col.index] || '',
          isEmpty: !data[i][col.index] || data[i][col.index].toString().trim() === ''
        };
      });
      
      sampleData.push(rowData);
    }
    
    // إحصائيات البيانات
    const statistics = {
      emptyContactPersonRows: 0,
      filledContactPersonRows: 0,
      totalDataRows: data.length - 1
    };
    
    // فحص جميع الصفوف لحساب الإحصائيات
    if (columnAnalysis.contactPersonColumns.length > 0) {
      const primaryContactColumn = columnAnalysis.contactPersonColumns[0];
      
      for (let i = 1; i < data.length; i++) {
        const contactValue = data[i][primaryContactColumn.index];
        if (!contactValue || contactValue.toString().trim() === '') {
          statistics.emptyContactPersonRows++;
        } else {
          statistics.filledContactPersonRows++;
        }
      }
    }
    
    return {
      status: 'success',
      message: 'تم إجراء التشخيص بنجاح',
      details: {
        sheetInfo: {
          name: sheetName,
          totalColumns: columnAnalysis.totalColumns,
          totalRows: columnAnalysis.totalRows
        },
        columnAnalysis: columnAnalysis,
        sampleData: sampleData,
        statistics: statistics,
        recommendations: generateRecommendations(columnAnalysis, statistics)
      }
    };
    
  } catch (error) {
    return {
      status: 'error',
      message: 'حدث خطأ أثناء التشخيص: ' + error.toString(),
      details: { error: error.toString() }
    };
  }
}

// دالة لتوليد التوصيات بناءً على نتائج التشخيص
const generateRecommendations = (columnAnalysis, statistics) => {
  const recommendations = [];
  
  if (columnAnalysis.contactPersonColumns.length === 0) {
    recommendations.push({
      type: 'critical',
      message: 'لم يتم العثور على عمود "اسم من تحدثت معه" في الجدول',
      solution: 'تأكد من وجود عمود بهذا الاسم الدقيق في الجدول'
    });
  } else if (columnAnalysis.contactPersonColumns.length > 1) {
    recommendations.push({
      type: 'warning',
      message: 'تم العثور على أكثر من عمود مشابه لـ "اسم من تحدثت معه"',
      solution: 'تأكد من استخدام اسم واحد فقط للعمود'
    });
  }
  
  if (statistics.emptyContactPersonRows > statistics.filledContactPersonRows) {
    recommendations.push({
      type: 'warning',
      message: `معظم الصفوف (${statistics.emptyContactPersonRows} من ${statistics.totalDataRows}) لا تحتوي على اسم جهة الاتصال`,
      solution: 'تأكد من ملء بيانات جهات الاتصال في الجدول'
    });
  }
  
  if (columnAnalysis.contactPersonColumns.length > 0 && !columnAnalysis.contactPersonColumns[0].exactMatch) {
    recommendations.push({
      type: 'info',
      message: 'اسم العمود لا يطابق تماماً "اسم من تحدثت معه"',
      solution: 'تأكد من استخدام الاسم الدقيق للعمود'
    });
  }
  
  return recommendations;
}

// دالة الحصول على بيانات جهة الاتصال
const getContactPersonData = (contactPersonName) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  const sheet = doc.getSheetByName(sheetName);
  
  if (!sheet) {
    return null;
  }
  
  const data = sheet.getDataRange().getValues();
  const headers = data[0];
  
  // العثور على فهارس الأعمدة المطلوبة
  const columnIndexes = {
    contactPerson: headers.indexOf('اسم من تحدثت معه'),
    hospitalName: headers.indexOf('اسم المستشفى او الطبيب او المختبر او المركز ... الخ'),
    phone: headers.indexOf('رقم الهاتف'),
    specialization: headers.indexOf('الاختصاص'),
    doctorType: headers.indexOf('تصنيف الطبيب'),
    customerDeal: headers.indexOf('تعامل الزبون'),
    newCustomer: headers.indexOf('هل الزبون جديد ؟ ام هناك تعامل سابق ؟'),
    importance: headers.indexOf('اهمية الزبون'),
    requestImportance: headers.indexOf('اهمية الطلب'),
    representativeName: headers.indexOf('اسم المندوب'),
    governorate: headers.indexOf('المحافظة'),
    district: headers.indexOf('قضاء'),
    city: headers.indexOf('الناحية او المدينة'),
    street: headers.indexOf('الشارع'),
    landmark: headers.indexOf('اقرب نقطة دالة'),
    detailedSpecialization: findColumnIndex(['الاختصاص الدقيق', 'اختصاص دقيق', 'التخصص الدقيق', 'التخصص']),
    jobTitle: headers.indexOf('العنوان الوظيفي او المنصب او القسم الذي يعمل فيه')
  };
  
  // البحث عن آخر سجل لجهة الاتصال
  for (let i = data.length - 1; i >= 1; i--) {
    if (data[i][columnIndexes.contactPerson] === contactPersonName) {
      const result = {
        contactPerson: data[i][columnIndexes.contactPerson] || '',
        hospitalName: data[i][columnIndexes.hospitalName] || '',
        phone: data[i][columnIndexes.phone] || '',
        specialization: data[i][columnIndexes.specialization] || '',
        doctorType: data[i][columnIndexes.doctorType] || '',
        customerDeal: data[i][columnIndexes.customerDeal] || '',
        newCustomer: data[i][columnIndexes.newCustomer] || '',
        importance: data[i][columnIndexes.importance] || '',
        requestImportance: data[i][columnIndexes.requestImportance] || '',
        representativeName: data[i][columnIndexes.representativeName] || '',
        governorate: data[i][columnIndexes.governorate] || '',
        district: data[i][columnIndexes.district] || '',
        city: data[i][columnIndexes.city] || '',
        street: data[i][columnIndexes.street] || '',
        landmark: data[i][columnIndexes.landmark] || '',
        detailedSpecialization: data[i][columnIndexes.detailedSpecialization] || '',
        jobTitle: data[i][columnIndexes.jobTitle] || ''
      };
      return result;
    }
  }
  
  return null;
}

// التعامل مع طلبات POST - يعالج إرسال النموذج والإعدادات
const doPost = (e) => {
  Logger.log('🔄 بدء معالجة طلب POST جديد في: ' + new Date().toISOString());
  
  // التحقق من وجود البيانات المرسلة
  if (!e) {
    Logger.log('❌ لا توجد بيانات في الطلب');
    return ContentService
      .createTextOutput(JSON.stringify({ 
        'result': 'error', 
        'message': 'لا توجد بيانات في الطلب',
        'timestamp': new Date().toISOString()
      }))
      .setMimeType(ContentService.MimeType.JSON);
  }

  if (!e.parameter) {
    Logger.log('❌ لا توجد معلمات في الطلب');
    return ContentService
      .createTextOutput(JSON.stringify({ 
        'result': 'error', 
        'message': 'لا توجد معلمات في الطلب',
        'timestamp': new Date().toISOString()
      }))
      .setMimeType(ContentService.MimeType.JSON);
  }

  Logger.log('📊 عدد المعلمات المستلمة: ' + Object.keys(e.parameter).length);
  Logger.log('📋 أسماء المعلمات: ' + Object.keys(e.parameter).join(', '));

  // فحص التقارير المكررة قبل المعالجة
  const duplicateCheck = checkForDuplicateReport(e.parameter);
  if (duplicateCheck.isDuplicate) {
    Logger.log('⚠️ تم اكتشاف تقرير مكرر: ' + duplicateCheck.reason);
    return ContentService
      .createTextOutput(JSON.stringify({ 
        'result': 'duplicate', 
        'message': 'تم اكتشاف تقرير مكرر - لم يتم الحفظ',
        'reason': duplicateCheck.reason,
        'timestamp': new Date().toISOString()
      }))
      .setMimeType(ContentService.MimeType.JSON);
  }

  const lock = LockService.getScriptLock();
  
  try {
    // محاولة الحصول على القفل مع مهلة زمنية أطول
    const lockAcquired = lock.tryLock(10000);
    if (!lockAcquired) {
      Logger.log('❌ فشل في الحصول على قفل العملية خلال 10 ثوانٍ');
      return ContentService
        .createTextOutput(JSON.stringify({ 
          'result': 'error', 
          'message': 'النظام مشغول، يرجى المحاولة مرة أخرى',
          'timestamp': new Date().toISOString()
        }))
        .setMimeType(ContentService.MimeType.JSON);
    }
    
    Logger.log('✅ تم الحصول على قفل العملية بنجاح');

    Logger.log('🔍 بدء معالجة طلب POST');
    Logger.log('📝 معلمات الطلب المستلمة: ' + JSON.stringify(e.parameter, null, 2));

    // التحقق من نوع الطلب
    if (e.parameter.action === 'saveSettings') {
      Logger.log('⚙️ معالجة طلب حفظ الإعدادات');
      // حفظ الإعدادات
      const userId = e.parameter.userId;
      const scriptUrl = e.parameter.scriptUrl;
      const representativeName = e.parameter.representativeName;
      const governorates = e.parameter.governorates;
      const district = e.parameter.district;
      
      saveSettings(userId, scriptUrl, representativeName, governorates, district);
      
      return ContentService
        .createTextOutput(JSON.stringify({ 'result': 'success', 'message': 'تم حفظ الإعدادات بنجاح' }))
        .setMimeType(ContentService.MimeType.JSON);
    }

    // معالجة طلب حفظ روابط الصور
    if (e.parameter.action === 'saveImageUrls') {
      Logger.log('📸 معالجة طلب حفظ روابط الصور');
      
      const imageUrls = e.parameter.imageUrls;
      const rowId = e.parameter.rowId || null; // معرف الصف إذا كان متوفراً
      
      if (!imageUrls) {
        Logger.log('❌ لا توجد روابط صور للحفظ');
        return ContentService
          .createTextOutput(JSON.stringify({ 
            'result': 'error', 
            'message': 'لا توجد روابط صور للحفظ',
            'timestamp': new Date().toISOString()
          }))
          .setMimeType(ContentService.MimeType.JSON);
      }
      
      try {
        const result = saveImageUrlsToSheet(imageUrls, rowId);
        Logger.log('✅ تم حفظ روابط الصور بنجاح');
        
        return ContentService
          .createTextOutput(JSON.stringify({ 
            'result': 'success', 
            'message': 'تم حفظ روابط الصور بنجاح',
            'data': result,
            'timestamp': new Date().toISOString()
          }))
          .setMimeType(ContentService.MimeType.JSON);
          
      } catch (error) {
        Logger.log('❌ خطأ في حفظ روابط الصور: ' + error.toString());
        return ContentService
          .createTextOutput(JSON.stringify({ 
            'result': 'error', 
            'message': 'خطأ في حفظ روابط الصور: ' + error.toString(),
            'timestamp': new Date().toISOString()
          }))
          .setMimeType(ContentService.MimeType.JSON);
      }
    }

    // التحقق من وجود معرف الجدول
    const spreadsheetId = scriptProp.getProperty('key');
    if (!spreadsheetId) {
      Logger.log('❌ معرف الجدول غير موجود في خصائص النص البرمجي');
      throw new Error('معرف جدول البيانات غير موجود. يرجى تشغيل دالة initialSetup أولاً.');
    }
    
    Logger.log('📊 معرف الجدول: ' + spreadsheetId);
    Logger.log('محاولة فتح جدول البيانات بالمعرف: ' + spreadsheetId);
    
    const doc = SpreadsheetApp.openById(spreadsheetId);
    if (!doc) {
      Logger.log('❌ فشل في فتح الجدول');
      throw new Error('فشل في فتح جدول البيانات. تحقق من صحة المعرف والصلاحيات.');
    }
    Logger.log('✅ تم فتح الجدول بنجاح');

    const sheet = doc.getSheetByName(sheetName);
    if (!sheet) {
      Logger.log('❌ الورقة غير موجودة: ' + sheetName);
      throw new Error(`لم يتم العثور على الورقة "${sheetName}". الأوراق المتاحة: ${doc.getSheets().map(s => s.getName()).join(', ')}`);
    }
    Logger.log('✅ تم الوصول إلى الورقة بنجاح: ' + sheetName);

    Logger.log('تم فتح الورقة بنجاح: ' + sheetName);

    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];
    Logger.log('رؤوس الأعمدة المستخرجة: ' + JSON.stringify(headers));

    // التحقق من وجود رؤوس صحيحة
    if (!headers || headers.length === 0) {
      throw new Error('لم يتم العثور على رؤوس أعمدة في الورقة. تأكد من وجود صف الرؤوس.');
    }

    const newRow = headers.map((header, index) => {
      const trimmedHeader = header.trim();
      Logger.log(`معالجة العمود ${index + 1}: "${trimmedHeader}"`);
      
      if (trimmedHeader === 'Date') {
        const currentDate = new Date();
        Logger.log('إضافة التاريخ الحالي: ' + currentDate);
        return currentDate;
      }
      
      // معالجة خاصة لحقل الاختصاص
      if (trimmedHeader === 'الاختصاص') {
        // البحث عن البيانات المدمجة أولاً
        if (e.parameter['combinedSpecializations']) {
          Logger.log('استخدام الاختصاصات المدمجة: ' + e.parameter['combinedSpecializations']);
          return e.parameter['combinedSpecializations'];
        }
        // إذا لم توجد، جمع القيم من الحقول المتعددة
        const specializationArray = e.parameter['الاختصاص[]'];
        if (specializationArray) {
          const result = Array.isArray(specializationArray) ? specializationArray.join(', ') : specializationArray;
          Logger.log('استخدام مصفوفة الاختصاصات: ' + result);
          return result;
        }
        Logger.log('لم يتم العثور على بيانات الاختصاص');
        return '';
      }
      
      // معالجة خاصة لحقل "جوب شيت" لرفع الملفات
      // تم التعديل ليتوافق مع إرسال رابط نصي من النموذج
      if (trimmedHeader === 'جوب شيت') {
        // إذا كان الحقل يحتوي على رابط (URL)، قم بحفظه مباشرة
        const jobsheetValue = e.parameter[trimmedHeader] || '';
        if (jobsheetValue.startsWith('http')) {
          Logger.log('📝 تم العثور على رابط جوب شيت: ' + jobsheetValue);
          return jobsheetValue;
        }
        // إذا كان الحقل يحتوي على ملف (للتوافق مع الإصدارات القديمة)، قم برفعه
        else if (e.parameter[trimmedHeader] && e.parameter[trimmedHeader].getBlob) {
          Logger.log('📂 تم العثور على ملف جوب شيت، جاري الرفع...');
          const fileBlob = e.parameter[trimmedHeader];
          // ... (يمكن إضافة منطق الرفع إلى Drive هنا إذا لزم الأمر) ...
          return 'File Uploaded (Legacy)'; // قيمة مؤقتة
        }
        return ''; // لا يوجد ملف مرفق
      }
      
      // استخدم e.parameter مباشرةً لتبسيط الكود
      const value = e.parameter[trimmedHeader] || '';
      Logger.log(`قيمة العمود "${trimmedHeader}": "${value}"`);
      return value;
    });

    Logger.log('الصف الجديد المُعد للإضافة: ' + JSON.stringify(newRow));

    // التحقق من أن الصف يحتوي على بيانات
    const hasData = newRow.some(cell => cell && cell.toString().trim() !== '');
    if (!hasData) {
      // التحقق مما إذا كان الطلب يحتوي على ملف فقط
      if (e.parameter['جوب شيت'] && e.parameter['جوب شيت'].getBlob) {
        Logger.log('⚠️ الصف فارغ ولكنه يحتوي على ملف جوب شيت. سيتم معالجته.');
        const fileUploadResult = handleFileUpload(e);
        // إرجاع نتيجة رفع الملف
        return ContentService.createTextOutput(JSON.stringify(fileUploadResult)).setMimeType(ContentService.MimeType.JSON);
      }
      throw new Error('الصف المرسل فارغ. تأكد من إرسال بيانات صحيحة.');
    }

    // استخدام appendRow() لإضافة البيانات مع تسجيل مفصل
    Logger.log('📝 محاولة إضافة الصف التالي إلى الجدول:');
    Logger.log('📋 محتوى الصف: ' + JSON.stringify(newRow));
    Logger.log('📊 عدد الأعمدة: ' + newRow.length);
    
    try {
      const startTime = new Date().getTime();
      sheet.appendRow(newRow);
      const endTime = new Date().getTime();
      
      Logger.log('✅ تم إضافة الصف بنجاح إلى الجدول في ' + (endTime - startTime) + ' مللي ثانية');
      Logger.log('📈 العدد الحالي للصفوف: ' + sheet.getLastRow());
      
    } catch (appendError) {
      Logger.log('❌ خطأ في إضافة الصف: ' + appendError.toString());
      Logger.log('❌ نوع الخطأ: ' + appendError.name);
      Logger.log('❌ تفاصيل الخطأ: ' + appendError.message);
      Logger.log('❌ الصف المحاول إضافته: ' + JSON.stringify(newRow));
      throw new Error('فشل في إضافة البيانات إلى الجدول: ' + appendError.message);
    }
    
    // تسجيل البيانات المرسلة بنجاح للمراجعة
    Logger.log('تم إرسال البيانات بنجاح');
    Logger.log('بيانات الاختصاص المرسلة: ' + (e.parameter['combinedSpecializations'] || 'غير موجودة'));

    // التحقق من وجود ملف مرفق ومعالجته
    let fileInfo = null;
    if (e.parameter['جوب شيت'] && e.parameter['جوب شيت'].getBlob) {
      const fileUploadResult = handleFileUpload(e);
      if (fileUploadResult.success) {
        fileInfo = fileUploadResult.fileInfo;
      }
    }

    // إرجاع استجابة JSON سريعة بدلاً من HTML
    return ContentService
      .createTextOutput(JSON.stringify({ 
        'result': 'success', 
        'message': 'تم إرسال التقرير بنجاح.',
        'rowData': newRow,
        'fileInfo': fileInfo, // إضافة معلومات الملف إلى الاستجابة
        'timestamp': new Date().toISOString()
      }))
      .setMimeType(ContentService.MimeType.JSON);

  } catch (error) {
    Logger.log('❌ خطأ في doPost: ' + error.toString());
    Logger.log('❌ تتبع المكدس: ' + error.stack);
    Logger.log('❌ معلمات الطلب: ' + JSON.stringify(e ? e.parameter : 'لا توجد معلمات'));
    
    // تسجيل خاص لبيانات الاختصاص للمساعدة في التشخيص
    if (e && e.parameter) {
      Logger.log('❌ بيانات الاختصاص المدمجة: ' + e.parameter['combinedSpecializations']);
      Logger.log('❌ بيانات الاختصاص المتعددة: ' + JSON.stringify(e.parameter['الاختصاص[]']));
      
      // تسجيل جميع المعلمات المرسلة للتشخيص
      Object.keys(e.parameter).forEach(key => {
        Logger.log(`❌ معلمة "${key}": "${e.parameter[key]}"`);
      });
    }
    
    // إرسال تفاصيل أكثر في حالة الخطأ
    const errorResponse = {
      'result': 'error',
      'message': error.toString(),
      'timestamp': new Date().toISOString(),
      'errorType': error.name || 'UnknownError',
      'details': {
        'stack': error.stack,
        'parameters': e ? Object.keys(e.parameter || {}) : [],
        'hasParameters': !!(e && e.parameter),
        'parameterCount': e && e.parameter ? Object.keys(e.parameter).length : 0
      }
    };
    
    return ContentService
      .createTextOutput(JSON.stringify(errorResponse))
      .setMimeType(ContentService.MimeType.JSON);

  } finally {
    // التأكد من تحرير القفل في جميع الحالات
    try {
      lock.releaseLock();
      Logger.log('تم تحرير قفل العملية بنجاح');
    } catch (lockError) {
      Logger.log('خطأ في تحرير القفل: ' + lockError.toString());
    }
  }
}

// --- دالة إرسال واتساب (محسنة للأداء) ---
function sendWhatsAppMessage(phoneNumber, message) {
  const accountSid = PropertiesService.getScriptProperties().getProperty('TWILIO_ACCOUNT_SID');
  const authToken = PropertiesService.getScriptProperties().getProperty('TWILIO_AUTH_TOKEN');
  const twilioPhoneNumber = PropertiesService.getScriptProperties().getProperty('TWILIO_PHONE_NUMBER');

  const endpoint = `https://api.twilio.com/2010-04-01/Accounts/${accountSid}/Messages.json`;

  const payload = {
    'To': `whatsapp:${phoneNumber}`,
    'From': `whatsapp:${twilioPhoneNumber}`,
    'Body': message
  };

  const options = {
    'method': 'post',
    'payload': payload,
    'headers': {
      'Authorization': 'Basic ' + Utilities.base64Encode(`${accountSid}:${authToken}`)
    },
    'muteHttpExceptions': true 
  };
  
  // استخدام fetchAll لإرسال الطلب بشكل غير متزامن
  UrlFetchApp.fetchAll([{
    url: endpoint,
    options: options
  }]);
  
  return true; // إرجاع قيمة سريعة دون انتظار الاستجابة
}

// دالة البحث عن المستشفيات/الأطباء/المختبرات
const searchHospitals = (searchTerm) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  const sheet = doc.getSheetByName(sheetName);
  
  if (!sheet) {
    return [];
  }
  
  const data = sheet.getDataRange().getValues();
  const headers = data[0];
  
  // تحسين البحث عن عمود اسم المستشفى
  const findHospitalColumnIndex = (headers) => {
    const possibleNames = [
      'اسم المستشفى او الطبيب او المختبر او المركز ... الخ',
      'اسم المستشفى',
      'اسم الطبيب',
      'اسم المختبر',
      'اسم المركز',
      'المستشفى',
      'الطبيب',
      'المختبر'
    ];
    
    // البحث الدقيق أولاً
    for (let i = 0; i < headers.length; i++) {
      const cleanHeader = headers[i].toString().trim();
      if (possibleNames.includes(cleanHeader)) {
        console.log(`تم العثور على عمود المستشفى بمطابقة دقيقة: "${headers[i]}" في الفهرس ${i}`);
        return i;
      }
    }
    
    // البحث الجزئي
    for (let i = 0; i < headers.length; i++) {
      const cleanHeader = headers[i].toString().trim();
      for (const name of possibleNames) {
        if (cleanHeader.includes(name.trim()) || name.trim().includes(cleanHeader)) {
          console.log(`تم العثور على عمود المستشفى بمطابقة جزئية: "${headers[i]}" في الفهرس ${i}`);
          return i;
        }
      }
    }
    
    console.log('لم يتم العثور على عمود المستشفى');
    return -1;
  };
  
  const hospitalNameIndex = findHospitalColumnIndex(headers);
  
  if (hospitalNameIndex === -1) {
    return [];
  }
  
  const results = [];
  const searchTermLower = searchTerm.toLowerCase();
  
  for (let i = 1; i < data.length; i++) {
    const hospitalName = data[i][hospitalNameIndex];
    if (hospitalName && hospitalName.toString().toLowerCase().includes(searchTermLower)) {
      if (!results.includes(hospitalName)) {
        results.push(hospitalName);
      }
    }
  }
  
  return results.slice(0, 10); // إرجاع أول 10 نتائج
}

// دالة البحث عن جهات الاتصال
const searchContactPersons = (searchTerm) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  const sheet = doc.getSheetByName(sheetName);
  
  if (!sheet) {
    return [];
  }
  
  const data = sheet.getDataRange().getValues();
  const headers = data[0];
  
  // تحسين البحث عن عمود جهة الاتصال
  const findContactPersonColumnIndex = (headers) => {
    const possibleNames = [
      'اسم من تحدثت معه',
      'اسم من تحدثت معه ',
      ' اسم من تحدثت معه',
      'اسم الشخص المسؤول',
      'جهة الاتصال',
      'اسم المسؤول',
      'الشخص المسؤول',
      'اسم جهة الاتصال',
      'المسؤول'
    ];
    
    // البحث الدقيق أولاً
    for (let i = 0; i < headers.length; i++) {
      const cleanHeader = headers[i].toString().trim();
      if (possibleNames.includes(cleanHeader)) {
        console.log(`تم العثور على عمود جهة الاتصال بمطابقة دقيقة: "${headers[i]}" في الفهرس ${i}`);
        return i;
      }
    }
    
    // البحث الجزئي
    for (let i = 0; i < headers.length; i++) {
      const cleanHeader = headers[i].toString().trim();
      for (const name of possibleNames) {
        if (cleanHeader.includes(name.trim()) || name.trim().includes(cleanHeader)) {
          console.log(`تم العثور على عمود جهة الاتصال بمطابقة جزئية: "${headers[i]}" في الفهرس ${i}`);
          return i;
        }
      }
    }
    
    console.log('لم يتم العثور على عمود جهة الاتصال');
    return -1;
  };
  
  const contactPersonIndex = findContactPersonColumnIndex(headers);
  
  if (contactPersonIndex === -1) {
    return [];
  }
  
  const results = [];
  const searchTermLower = searchTerm.toLowerCase();
  
  for (let i = 1; i < data.length; i++) {
    const contactPersonName = data[i][contactPersonIndex];
    if (contactPersonName && contactPersonName.toString().toLowerCase().includes(searchTermLower)) {
      if (!results.includes(contactPersonName)) {
        results.push(contactPersonName);
      }
    }
  }
  
  return results.slice(0, 10); // إرجاع أول 10 نتائج
}



// دالة الحصول على بيانات مستشفى/طبيب/مختبر محدد
const getHospitalData = (hospitalName) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  const sheet = doc.getSheetByName(sheetName);
  
  if (!sheet) {
    return null;
  }
  
  const data = sheet.getDataRange().getValues();
  const headers = data[0];
  
  // العثور على فهارس الأعمدة المطلوبة
  // البحث عن أسماء بديلة للأعمدة
  const findColumnIndex = (possibleNames) => {
    for (const name of possibleNames) {
      const index = headers.indexOf(name);
      if (index !== -1) return index;
    }
    return -1;
  };
  
  const columnIndexes = {
    hospitalName: headers.indexOf('اسم المستشفى او الطبيب او المختبر او المركز ... الخ'),
    contactPerson: findColumnIndex(['اسم من تحدثت معه', 'اسم من تحدثت معه ', ' اسم من تحدثت معه', 'اسم الشخص المسؤول', 'جهة الاتصال']),
    phone: headers.indexOf('رقم الهاتف'),
    specialization: headers.indexOf('الاختصاص'),
    doctorType: headers.indexOf('تصنيف الطبيب'),
    customerDeal: headers.indexOf('تعامل الزبون'),
    customerStatus: findColumnIndex(['هل الزبون جديد ؟ ام هناك تعامل سابق ؟', 'حالة الزبون', 'نوع الزبون']),
    customerImportance: findColumnIndex(['اهمية الزبون', 'أهمية الزبون']),
    importance: findColumnIndex(['اهمية الطلب', 'أهمية الطلب']),
    representativeName: findColumnIndex(['اسم المندوب', 'المندوب', 'اسم المندوب المسؤول']),
    governorate: headers.indexOf('المحافظة'),
    district: headers.indexOf('قضاء'),
    city: headers.indexOf('الناحية او المدينة'),
    street: headers.indexOf('الشارع'),
    landmark: headers.indexOf('اقرب نقطة دالة'),
    detailedSpecialization: findColumnIndex(['الاختصاص الدقيق', 'اختصاص دقيق', 'التخصص الدقيق', 'التخصص']),
    jobTitle: headers.indexOf('العنوان الوظيفي او المنصب او القسم الذي يعمل فيه')
  };
  
  console.log(`البحث عن المستشفى: ${hospitalName}`);
  console.log(`فهرس عمود المستشفى: ${columnIndexes.hospitalName}`);
  console.log(`فهرس عمود جهة الاتصال: ${columnIndexes.contactPerson}`);
  console.log(`عدد الصفوف في الجدول: ${data.length - 1}`);
  console.log(`رؤوس الأعمدة:`, headers);
  
  // تسجيل تفصيلي للبحث
  let searchLog = {
    searchTerm: hospitalName,
    hospitalColumnIndex: columnIndexes.hospitalName,
    contactPersonColumnIndex: columnIndexes.contactPerson,
    totalRows: data.length - 1,
    matchesFound: [],
    searchMethod: 'bottom-up'
  };

  // البحث عن آخر سجل للمستشفى/الطبيب/المختبر
  for (let i = data.length - 1; i >= 1; i--) {
    const currentHospitalName = (data[i][columnIndexes.hospitalName] || '').toString().trim();
    const searchHospitalName = hospitalName.toString().trim();
    
    // تسجيل كل محاولة بحث
    searchLog.matchesFound.push({
      rowIndex: i,
      currentValue: currentHospitalName,
      isMatch: currentHospitalName === searchHospitalName
    });
    
    if (currentHospitalName === searchHospitalName) {
      console.log(`✅ تم العثور على مطابقة في الصف ${i}`);
      console.log('سجل البحث الكامل:', JSON.stringify(searchLog, null, 2));
      
      // التأكد من وجود العمود قبل الوصول إليه وتنظيف البيانات
      const contactPersonValue = columnIndexes.contactPerson !== -1 ? 
        (data[i][columnIndexes.contactPerson] || '').toString().trim() : '';
      
      // تسجيل مؤقت للتشخيص
      console.log('🔍 تشخيص البيانات:');
      console.log('فهرس عمود اسم من تحدثت معه:', columnIndexes.contactPerson);
      console.log('القيمة الخام:', data[i][columnIndexes.contactPerson]);
      console.log('القيمة بعد التنظيف:', contactPersonValue);
      console.log('جميع بيانات الصف:', data[i]);
      
      const result = {
        hospitalName: currentHospitalName,
        contactPerson: contactPersonValue,
        phone: columnIndexes.phone !== -1 ? (data[i][columnIndexes.phone] || '').toString().trim() : '',
        specialization: columnIndexes.specialization !== -1 ? (data[i][columnIndexes.specialization] || '').toString().trim() : '',
        doctorType: columnIndexes.doctorType !== -1 ? (data[i][columnIndexes.doctorType] || '').toString().trim() : '',
        customerDeal: columnIndexes.customerDeal !== -1 ? (data[i][columnIndexes.customerDeal] || '').toString().trim() : '',
        customerStatus: columnIndexes.customerStatus !== -1 ? (data[i][columnIndexes.customerStatus] || '').toString().trim() : '',
        customerImportance: columnIndexes.customerImportance !== -1 ? (data[i][columnIndexes.customerImportance] || '').toString().trim() : '',
        importance: columnIndexes.importance !== -1 ? (data[i][columnIndexes.importance] || '').toString().trim() : '',
        representativeName: columnIndexes.representativeName !== -1 ? (data[i][columnIndexes.representativeName] || '').toString().trim() : '',
        governorate: columnIndexes.governorate !== -1 ? (data[i][columnIndexes.governorate] || '').toString().trim() : '',
        district: columnIndexes.district !== -1 ? (data[i][columnIndexes.district] || '').toString().trim() : '',
        city: columnIndexes.city !== -1 ? (data[i][columnIndexes.city] || '').toString().trim() : '',
        street: columnIndexes.street !== -1 ? (data[i][columnIndexes.street] || '').toString().trim() : '',
        landmark: columnIndexes.landmark !== -1 ? (data[i][columnIndexes.landmark] || '').toString().trim() : '',
        detailedSpecialization: columnIndexes.detailedSpecialization !== -1 ? (data[i][columnIndexes.detailedSpecialization] || '').toString().trim() : '',
        jobTitle: columnIndexes.jobTitle !== -1 ? (data[i][columnIndexes.jobTitle] || '').toString().trim() : ''
      };
      
      console.log(`تم العثور على بيانات المستشفى: ${hospitalName}`);
      console.log('البيانات المسترجعة:', result);
      console.log('سجل البحث الكامل:', JSON.stringify(searchLog, null, 2));
      
      return result;
    }
  }
  
  console.log(`❌ لم يتم العثور على المستشفى: ${hospitalName}`);
  console.log('سجل البحث الكامل:', JSON.stringify(searchLog, null, 2));
  return null;
}

// دالة للحصول على البيانات بناءً على اسم المستشفى واسم جهة الاتصال
const getDataByHospitalAndContact = (hospitalName, contactPersonName) => {
  const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
  const sheet = doc.getSheetByName(sheetName);
  
  if (!sheet) {
    return null;
  }
  
  const data = sheet.getDataRange().getValues();
  const headers = data[0];
  
  // العثور على فهارس الأعمدة المطلوبة
  const findColumnIndex = (possibleNames) => {
    for (const name of possibleNames) {
      const index = headers.indexOf(name);
      if (index !== -1) return index;
    }
    // البحث الجزئي كخيار احتياطي
    for (let i = 0; i < headers.length; i++) {
      const cleanHeader = headers[i].toString().trim();
      for (const name of possibleNames) {
        if (cleanHeader.includes(name.trim())) return i;
      }
    }
    return -1;
  };
  
  const columnIndexes = {
    hospitalName: findColumnIndex([
      'اسم المستشفى او الطبيب او المختبر او المركز ... الخ', 
      'اسم المستشفى', 
      'اسم الطبيب'
    ]),
    contactPerson: findColumnIndex([
      'اسم من تحدثت معه', 
      'اسم من تحدثت معه ', // مع مسافة زائدة
      ' اسم من تحدثت معه', // مع مسافة زائدة
      'اسم الشخص المسؤول', 
      'جهة الاتصال',
      'اسم المسؤول',
      'الشخص المسؤول'
    ]),
    phone: headers.indexOf('رقم الهاتف'),
    specialization: headers.indexOf('الاختصاص'),
    doctorType: headers.indexOf('تصنيف الطبيب'),
    customerDeal: headers.indexOf('تعامل الزبون'),
    customerStatus: findColumnIndex(['هل الزبون جديد ؟ ام هناك تعامل سابق ؟', 'حالة الزبون']),
    customerImportance: findColumnIndex(['اهمية الزبون', 'أهمية الزبون']),
    importance: findColumnIndex(['اهمية الطلب', 'أهمية الطلب']),
    representativeName: findColumnIndex(['اسم المندوب', 'المندوب']),
    governorate: headers.indexOf('المحافظة'),
    district: headers.indexOf('قضاء'),
    city: headers.indexOf('الناحية او المدينة'),
    street: headers.indexOf('الشارع'),
    landmark: headers.indexOf('اقرب نقطة دالة'),
    detailedSpecialization: headers.indexOf('الاختصاص الدقيق'),
    jobTitle: headers.indexOf('العنوان الوظيفي او المنصب او القسم الذي يعمل فيه')
  };
  
  // تشخيص عمود الاختصاص الدقيق
  Logger.log('🔍 تشخيص عمود الاختصاص الدقيق:');
  Logger.log('فهرس عمود الاختصاص الدقيق: ' + columnIndexes.detailedSpecialization);
  Logger.log('رؤوس الأعمدة المتاحة: ' + JSON.stringify(headers));
  Logger.log('البحث عن الأعمدة التي تحتوي على كلمة "اختصاص" أو "تخصص":');
  headers.forEach((header, index) => {
    if (header && (header.includes('اختصاص') || header.includes('تخصص'))) {
      Logger.log(`العمود ${index}: "${header}"`);
    }
  });

  if (columnIndexes.hospitalName === -1 || columnIndexes.contactPerson === -1) {
    Logger.log('لم يتم العثور على أعمدة البحث الرئيسية (المستشفى أو جهة الاتصال)');
    return null;
  }

  // البحث من الأسفل للأعلى للعثور على آخر سجل مطابق
  for (let i = data.length - 1; i >= 1; i--) {
    const currentHospitalName = (data[i][columnIndexes.hospitalName] || '').toString();
    const currentContactPerson = (data[i][columnIndexes.contactPerson] || '').toString();
    
    // تطبيع النصوص للمقارنة المرنة (يتجاهل الفروقات بين أ,ا,إ و ي,ى و ه,ة)
    const normalizedSheetHospital = normalizeArabic(currentHospitalName);
    const normalizedSheetContact = normalizeArabic(currentContactPerson);
    const normalizedSearchHospital = normalizeArabic(hospitalName);
    const normalizedSearchContact = normalizeArabic(contactPersonName);
    
    if (normalizedSheetHospital === normalizedSearchHospital && normalizedSheetContact === normalizedSearchContact) {
      Logger.log(`تم العثور على تطابق في الصف ${i + 1} بعد التطبيع.`);
      const result = {};
      // بناء كائن النتيجة ديناميكيًا
      for (const key in columnIndexes) {
        const colIndex = columnIndexes[key];
        if (colIndex !== -1 && data[i][colIndex] !== undefined) {
          result[key] = (data[i][colIndex] || '').toString().trim();
        } else {
          result[key] = ''; // إرجاع قيمة فارغة إذا لم يتم العثور على العمود
        }
      }
      
      // تشخيص خاص بالاختصاص الدقيق
      Logger.log('🔍 تشخيص الاختصاص الدقيق في النتيجة:');
      Logger.log('قيمة الاختصاص الدقيق الخام: "' + (data[i][columnIndexes.detailedSpecialization] || 'غير موجود') + '"');
      Logger.log('قيمة الاختصاص الدقيق في النتيجة: "' + result.detailedSpecialization + '"');
      Logger.log('فهرس عمود الاختصاص الدقيق: ' + columnIndexes.detailedSpecialization);
      
      return result;
    }
  }
  
  Logger.log(`لم يتم العثور على تطابق لـ: المستشفى="${hospitalName}", جهة الاتصال="${contactPersonName}"`);
  return null;
}

/**
 * دالة لتطبيع النص العربي للمقارنة.
 * توحد أشكال الحروف المختلفة (أ, إ, آ -> ا), (ى -> ي), (ة -> ه).
 * @param {string} text النص المراد تطبيعه.
 * @returns {string} النص بعد التطبيع.
 */
function normalizeArabic(text) {
  if (!text || typeof text !== 'string') {
    return '';
  }
  return text
    .replace(/[أإآ]/g, 'ا') // توحيد الألف
    .replace(/ى/g, 'ي')     // توحيد الياء
    .replace(/ة/g, 'ه')     // توحيد التاء المربوطة
    .trim();
}

/**
 * دالة للتحقق من التقارير المكررة
 * @param {Object} parameters - معلمات الطلب المرسل
 * @returns {Object} نتيجة فحص التكرار
 */
function checkForDuplicateReport(parameters) {
  try {
    const sheet = SpreadsheetApp.openById(scriptProp.getProperty('key')).getSheetByName('responses');
    if (!sheet) {
      Logger.log('❌ لم يتم العثور على ورقة العمل responses');
      return { isDuplicate: false, reason: 'لا يمكن الوصول لورقة العمل' };
    }

    const currentTime = new Date();
    const timeThreshold = 5 * 60 * 1000; // 5 دقائق بالميلي ثانية
    
    // الحصول على البيانات الحديثة (آخر 50 صف)
    const lastRow = sheet.getLastRow();
    if (lastRow <= 1) {
      return { isDuplicate: false, reason: 'لا توجد بيانات سابقة' };
    }
    
    const startRow = Math.max(2, lastRow - 49); // آخر 50 صف أو من الصف 2
    const range = sheet.getRange(startRow, 1, lastRow - startRow + 1, sheet.getLastColumn());
    const data = range.getValues();
    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];
    
    // إنشاء مؤشرات الأعمدة
    const columnIndices = {};
    headers.forEach((header, index) => {
      columnIndices[header] = index;
    });
    
    // استخراج البيانات الحالية للمقارنة
    const currentData = {
      timestamp: parameters['Timestamp'] || '',
      name: parameters['الاسم'] || '',
      location: parameters['المكان'] || '',
      governorate: parameters['المحافظة'] || '',
      district: parameters['القضاء'] || '',
      coordinates: parameters['location'] || '',
      uuid: parameters['uuid'] || ''
    };
    
    Logger.log('🔍 فحص التكرار للبيانات: ' + JSON.stringify(currentData));
    
    // فحص كل صف من البيانات الحديثة
    for (let i = 0; i < data.length; i++) {
      const row = data[i];
      const rowTime = new Date(row[columnIndices['Timestamp']] || row[0]);
      
      // تخطي الصفوف القديمة (أكثر من 5 دقائق)
      if (currentTime - rowTime > timeThreshold) {
        continue;
      }
      
      // فحص UUID المطابق
      if (currentData.uuid && row[columnIndices['uuid']] === currentData.uuid) {
        return { 
          isDuplicate: true, 
          reason: `UUID مطابق: ${currentData.uuid}` 
        };
      }
      
      // فحص البيانات المطابقة (الاسم + المكان + الإحداثيات)
      const rowData = {
        name: row[columnIndices['الاسم']] || '',
        location: row[columnIndices['المكان']] || '',
        coordinates: row[columnIndices['location']] || ''
      };
      
      if (currentData.name && currentData.location && currentData.coordinates) {
        if (rowData.name === currentData.name && 
            rowData.location === currentData.location && 
            rowData.coordinates === currentData.coordinates) {
          const timeDiff = Math.round((currentTime - rowTime) / 1000);
          return { 
            isDuplicate: true, 
            reason: `بيانات مطابقة خلال ${timeDiff} ثانية: ${currentData.name} - ${currentData.location}` 
          };
        }
      }
    }
    
    Logger.log('✅ لم يتم اكتشاف تكرار');
    return { isDuplicate: false, reason: 'لا يوجد تكرار' };
    
  } catch (error) {
    Logger.log('❌ خطأ في فحص التكرار: ' + error.toString());
    // في حالة الخطأ، نسمح بالمرور لتجنب منع التقارير الصحيحة
    return { isDuplicate: false, reason: 'خطأ في فحص التكرار: ' + error.message };
  }
}

// ===== دوال Google Drive API =====

/**
 * رفع ملف إلى Google Drive
 * @param {Blob} fileBlob - الملف المراد رفعه
 * @param {string} fileName - اسم الملف
 * @param {string} folderId - معرف المجلد في Google Drive
 * @return {Object} - معلومات الملف المرفوع
 */
function uploadFileToGoogleDrive(fileBlob, fileName, folderId = DRIVE_FOLDER_ID) {
  try {
    Logger.log('🔄 بدء رفع الملف: ' + fileName);
    
    // التحقق من نوع الملف
    const fileExtension = fileName.split('.').pop().toLowerCase();
    if (!ALLOWED_FILE_TYPES.includes(fileExtension)) {
      throw new Error('نوع الملف غير مسموح: ' + fileExtension);
    }
    
    // التحقق من حجم الملف
    if (fileBlob.getBytes().length > MAX_FILE_SIZE) {
      throw new Error('حجم الملف كبير جداً. الحد الأقصى: ' + (MAX_FILE_SIZE / 1024 / 1024) + ' ميجابايت');
    }
    
    // إنشاء اسم ملف فريد
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
    const uniqueFileName = `${timestamp}_${fileName}`;
    
    // رفع الملف إلى Google Drive
    const folder = DriveApp.getFolderById(folderId);
    const file = folder.createFile(fileBlob.setName(uniqueFileName));
    
    // تعيين صلاحيات المشاركة
    file.setSharing(DriveApp.Access.ANYONE_WITH_LINK, DriveApp.Permission.VIEW);
    
    const fileInfo = {
      id: file.getId(),
      name: file.getName(),
      url: file.getUrl(),
      downloadUrl: `https://drive.google.com/uc?id=${file.getId()}`,
      size: file.getSize(),
      mimeType: file.getBlob().getContentType(),
      createdTime: file.getDateCreated().toISOString(),
      webViewLink: `https://drive.google.com/file/d/${file.getId()}/view`
    };
    
    Logger.log('✅ تم رفع الملف بنجاح: ' + JSON.stringify(fileInfo));
    return {
      success: true,
      file: fileInfo,
      message: 'تم رفع الملف بنجاح'
    };
    
  } catch (error) {
    Logger.log('❌ خطأ في رفع الملف: ' + error.toString());
    return {
      success: false,
      error: error.toString(),
      message: 'فشل في رفع الملف: ' + error.toString()
    };
  }
}

/**
 * حفظ رابط الملف في Google Sheets
 * @param {Object} fileInfo - معلومات الملف
 * @param {Object} formData - بيانات النموذج
 * @return {Object} - نتيجة العملية
 */
function saveFileInfoToSheet(fileInfo, formData) {
  try {
    const doc = SpreadsheetApp.openById(scriptProp.getProperty('key'));
    const sheet = doc.getSheetByName(sheetName);
    
    if (!sheet) {
      throw new Error('لم يتم العثور على الورقة: ' + sheetName);
    }
    
    // إضافة معلومات الملف إلى بيانات النموذج
    const updatedFormData = {
      ...formData,
      'رابط الملف': fileInfo.webViewLink,
      'اسم الملف': fileInfo.name,
      'حجم الملف': Math.round(fileInfo.size / 1024) + ' كيلوبايت',
      'نوع الملف': fileInfo.mimeType,
      'تاريخ الرفع': fileInfo.createdTime
    };
    
    // حفظ البيانات في الجدول
    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];
    const newRow = headers.map(header => updatedFormData[header] || '');
    
    sheet.appendRow(newRow);
    
    Logger.log('✅ تم حفظ معلومات الملف في الجدول');
    return {
      success: true,
      message: 'تم حفظ معلومات الملف بنجاح'
    };
    
  } catch (error) {
    Logger.log('❌ خطأ في حفظ معلومات الملف: ' + error.toString());
    return {
      success: false,
      error: error.toString(),
      message: 'فشل في حفظ معلومات الملف: ' + error.toString()
    };
  }
}

/**
 * دالة حفظ روابط الصور في Google Sheets
 * @param {string} imageUrls - روابط الصور مفصولة بفواصل
 * @param {string} rowId - معرف الصف (اختياري)
 * @return {Object} - نتيجة العملية
 */
const saveImageUrlsToSheet = (imageUrls, rowId = null) => {
  try {
    Logger.log('🔄 بدء حفظ روابط الصور في الجدول');
    Logger.log('📸 روابط الصور: ' + imageUrls);
    Logger.log('🆔 معرف الصف: ' + rowId);
    
    const spreadsheetId = scriptProp.getProperty('key');
    if (!spreadsheetId) {
      throw new Error('معرف جدول البيانات غير موجود');
    }
    
    const doc = SpreadsheetApp.openById(spreadsheetId);
    const sheet = doc.getSheetByName(sheetName);
    
    if (!sheet) {
      throw new Error('لم يتم العثور على الورقة: ' + sheetName);
    }
    
    // البحث عن عمود الصور
    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];
    let imageColumnIndex = -1;
    
    // البحث عن عمود الصور بأسماء مختلفة
    const possibleImageColumnNames = [
      'الصور',
      'صور',
      'روابط الصور',
      'Image URLs',
      'Images',
      'الملفات المرفقة',
      'المرفقات'
    ];
    
    for (let i = 0; i < headers.length; i++) {
      const header = headers[i].toString().trim();
      if (possibleImageColumnNames.includes(header)) {
        imageColumnIndex = i + 1; // العمود يبدأ من 1
        break;
      }
    }
    
    // إذا لم يوجد عمود للصور، إنشاء عمود جديد
    if (imageColumnIndex === -1) {
      Logger.log('⚠️ لم يتم العثور على عمود للصور، سيتم إنشاء عمود جديد');
      imageColumnIndex = headers.length + 1;
      sheet.getRange(1, imageColumnIndex).setValue('روابط الصور');
    }
    
    Logger.log('📍 عمود الصور في الموضع: ' + imageColumnIndex);
    
    let targetRow;
    
    if (rowId) {
      // البحث عن الصف بناءً على معرف الصف
      const data = sheet.getDataRange().getValues();
      targetRow = -1;
      
      for (let i = 1; i < data.length; i++) {
        // البحث في العمود الأول أو أي عمود يحتوي على معرف
        if (data[i][0] && data[i][0].toString() === rowId.toString()) {
          targetRow = i + 1;
          break;
        }
      }
      
      if (targetRow === -1) {
        Logger.log('⚠️ لم يتم العثور على الصف بالمعرف: ' + rowId + '، سيتم إضافة صف جديد');
        targetRow = sheet.getLastRow() + 1;
      }
    } else {
      // إضافة في صف جديد
      targetRow = sheet.getLastRow() + 1;
    }
    
    Logger.log('📍 الصف المستهدف: ' + targetRow);
    
    // حفظ روابط الصور في العمود المحدد
    sheet.getRange(targetRow, imageColumnIndex).setValue(imageUrls);
    
    Logger.log('✅ تم حفظ روابط الصور بنجاح في الصف ' + targetRow + ' العمود ' + imageColumnIndex);
    
    return {
      success: true,
      row: targetRow,
      column: imageColumnIndex,
      imageUrls: imageUrls,
      message: 'تم حفظ روابط الصور بنجاح'
    };
    
  } catch (error) {
    Logger.log('❌ خطأ في حفظ روابط الصور: ' + error.toString());
    throw error;
  }
}

/**
 * معالجة رفع الملف من النموذج
 * @param {Object} e - بيانات الطلب
 * @return {Object} - نتيجة العملية
 */
function handleFileUpload(e) {
  try {
    Logger.log('🔄 بدء معالجة رفع الملف');
    
    const parameters = e.parameter;
    const fileBlob = e.parameters['جوب شيت'] ? e.parameters['جوب شيت'][0] : null;
    
    if (!fileBlob) {
      return {
        success: false,
        message: 'لم يتم العثور على ملف للرفع'
      };
    }
    
    // رفع الملف إلى Google Drive
    const uploadResult = uploadFileToGoogleDrive(fileBlob, 'jobsheet_file');
    
    if (!uploadResult.success) {
      return uploadResult;
    }
    
    // حفظ معلومات الملف في الجدول
    const saveResult = saveFileInfoToSheet(uploadResult.file, parameters);
    
    if (!saveResult.success) {
      return saveResult;
    }
    
    return {
      success: true,
      fileInfo: uploadResult.file,
      message: 'تم رفع الملف وحفظ البيانات بنجاح'
    };
    
  } catch (error) {
    Logger.log('❌ خطأ في معالجة رفع الملف: ' + error.toString());
    return {
      success: false,
      error: error.toString(),
      message: 'فشل في معالجة رفع الملف: ' + error.toString()
    };
  }
}