<?php
require 'config.php';

// التحقق من تسجيل الدخول
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_role = $_SESSION['role'];

// إنشاء شريط تنقل علوي بشكل احترافي
$navBar = '
  <nav style="width:100%; background:#1e1e1e; padding:10px 20px; box-shadow: 0 2px 5px rgba(0,0,0,0.5); display:flex; justify-content:space-between; align-items:center;">
    <div style="font-size:18px; color:#fff;">Almanara</div>
    <div>
      <a href="logout.php" style="color:#fff; text-decoration:none; margin-right:20px;">خروج</a>';
if ($user_role === 'admin') {
    $navBar .= '<a href="admin.php" style="color:#fff; text-decoration:none; margin-right:20px;">لوحة الأدمن</a>';
}
$navBar .= '
    </div>
  </nav>';

// إذا كانت معلمات المشاركة موجودة (وضع العرض المشترك)
if (isset($_GET['share']) && isset($_GET['device_id']) && isset($_GET['expires'])) {
    $device_id = $_GET['device_id'];
    $expires = $_GET['expires'];
    ?>
    <!DOCTYPE html>
    <html lang="ar">
    <head>
      <meta charset="UTF-8"/>
      <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
      <title>عرض الموقع المشترك</title>
      <!-- مكتبات خريطة Leaflet -->
      <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.3/dist/leaflet.css"/>
      <script src="https://unpkg.com/leaflet@1.9.3/dist/leaflet.js"></script>
      <style>
        body {
          font-family: Arial, sans-serif;
          margin: 0;
          padding: 0;
          background-color: #121212;
          color: #e0e0e0;
          text-align: center;
        }
        .header {
          background: #007BFF;
          padding: 15px;
          font-size: 20px;
          font-weight: bold;
          color: #fff;
        }
        #sharedMap {
          height: 600px;
          width: 100%;
          margin-top: 20px;
        }
      </style>
    </head>
    <body>
      <div class="header">
        عرض الموقع المشترك للجهاز: <?php echo htmlspecialchars($device_id); ?>
      </div>
      <div id="sharedMap"></div>
      <script>
        // انتهاء صلاحية المشاركة (بالملّي ثانية)
        var shareExpires = <?php echo json_encode($expires); ?>;
        var deviceId = <?php echo json_encode($device_id); ?>;
        // إنشاء الخريطة
        var map = L.map('sharedMap', { center: [30, 0], zoom: 2 });
        L.tileLayer(
          "https://api.mapbox.com/styles/v1/mapbox/outdoors-v11/tiles/{z}/{x}/{y}?access_token=pk.eyJ1IjoiaWJyYWhpbWNvbSIsImEiOiJja2YxZnB6OWowc2xxMnFxZGxya2ppdmZxIn0.keECYQuu1Bs0JeWDP8dSFQ", 
          { maxZoom: 20, tileSize: 512, zoomOffset: -1, attribution: '&copy; <a href="https://www.openstreetmap.org/">OSM</a> &amp; <a href="https://www.mapbox.com/">Mapbox</a>' }
        ).addTo(map);
        var sharedMarker = null;
        function fetchSharedLocation() {
            if (Date.now() > shareExpires) {
                alert("انتهت صلاحية مشاركة الموقع.");
                return;
            }
            fetch("get_latest_location.php?device_id=" + deviceId + "&t=" + Date.now())
                .then(res => res.json())
                .then(data => {
                    if (data.status === "success") {
                        var lat = parseFloat(data.data.latitude);
                        var lng = parseFloat(data.data.longitude);
                        var battery = data.data.battery; // نسبة الشحن
                        if (!isNaN(lat) && !isNaN(lng)) {
                            if (!sharedMarker) {
                                sharedMarker = L.marker([lat, lng]).addTo(map);
                            } else {
                                sharedMarker.setLatLng([lat, lng]);
                            }
                            sharedMarker.bindPopup(`
                                <b>موقع الجهاز</b><br>
                                البطارية: ${battery ? battery + '%' : 'غير معروفة'}
                            `).openPopup();
                            map.setView([lat, lng], 15);
                        }
                    } else {
                        console.warn("❌ لا يوجد موقع متاح لهذا الجهاز.");
                    }
                })
                .catch(err => console.error("⚠️ خطأ في جلب الموقع:", err));
        }
        fetchSharedLocation();
        setInterval(fetchSharedLocation, 10000); // تحديث كل 10 ثوانٍ
      </script>
    </body>
    </html>
    <?php
    exit;
}

// جلب الأجهزة بناءً على صلاحيات المستخدم
if ($user_role === 'admin') {
    $stmt = $conn->query("SELECT DISTINCT device_id, device_name FROM locations ORDER BY device_name ASC");
} else {
    // جلب قائمة الأجهزة المسموح بها للمستخدم الحالي من جدول users
    $stmt_user = $conn->prepare("SELECT allowed_devices FROM users WHERE id = :id");
    $stmt_user->execute(['id' => $_SESSION['user_id']]);
    $row = $stmt_user->fetch(PDO::FETCH_ASSOC);
    $allowed_devices = ($row && $row['allowed_devices']) ? array_map('trim', explode(',', $row['allowed_devices'])) : [];
    
    if (count($allowed_devices) > 0) {
        $placeholders = implode(',', array_fill(0, count($allowed_devices), '?'));
        $stmt = $conn->prepare("SELECT DISTINCT device_id, device_name FROM locations WHERE device_id IN ($placeholders) ORDER BY device_name ASC");
        $stmt->execute($allowed_devices);
    } else {
        // إذا لم يكن للمستخدم أجهزة مسموحة فعدم جلب أي بيانات
        $stmt = $conn->prepare("SELECT DISTINCT device_id, device_name FROM locations WHERE 1=0");
        $stmt->execute();
    }
}
$devices = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="ar">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Almanara</title>
  
  <!-- مكتبات خريطة Leaflet -->
  <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.3/dist/leaflet.css"/>
  <script src="https://unpkg.com/leaflet@1.9.3/dist/leaflet.js"></script>
  
  <!-- مكتبات Mapbox للبحث العكسي (Geocoding) -->
  <script src="https://api.mapbox.com/mapbox-gl-js/v2.6.1/mapbox-gl.js"></script>
  <link href="https://api.mapbox.com/mapbox-gl-js/v2.6.1/mapbox-gl.css" rel="stylesheet"/>
  
  <!-- مكتبة Timeline -->
  <link rel="stylesheet" href="https://cdn.knightlab.com/libs/timeline3/latest/css/timeline.css">
  <script src="https://cdn.knightlab.com/libs/timeline3/latest/js/timeline.js"></script>
  
  <!-- مكتبة Chart.js لتحليل البيانات -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

  <!-- مكتبة SheetJS لتوليد ملفات XLSX -->
  <script src="https://cdn.sheetjs.com/xlsx-latest/package/dist/xlsx.full.min.js"></script>
  
  <style>
    /* تنسيق الوضع الداكن والاستجابة */
    * {
      box-sizing: border-box;
    }
    body {
      font-family: Arial, sans-serif;
      margin: 0;
      padding: 0;
      background-color: #121212;
      color: #e0e0e0;
      display: flex;
      flex-direction: column;
      min-height: 100vh;
    }
    /* شريط التنقل العلوي */
    .top-nav {
      width: 100%;
      background: #1e1e1e;
      padding: 10px 20px;
      box-shadow: 0 2px 5px rgba(0,0,0,0.5);
      display: flex;
      justify-content: space-between;
      align-items: center;
      flex-wrap: wrap;
    }
    .top-nav .logo {
      font-size: 18px;
      color: #fff;
      margin-bottom: 5px;
    }
    .top-nav a {
      color: #fff;
      text-decoration: none;
      margin-left: 20px;
      font-size: 16px;
    }
    /* المحتوى الرئيسي */
    .container {
      display: flex;
      flex: 1;
      overflow: hidden;
    }
    .sidebar {
      width: 300px;
      background: #1e1e1e;
      padding: 15px;
      overflow-y: auto;
    }
    .sidebar h2 {
      margin-top: 0;
      text-align: center;
      color: #fff;
    }
    .sidebar input[type="text"] {
      width: 100%;
      padding: 10px;
      margin-bottom: 10px;
      border-radius: 5px;
      border: 1px solid #333;
      background: #333;
      color: #e0e0e0;
    }
    .sidebar ul {
      list-style: none;
      padding: 0;
    }
    .sidebar ul li {
      padding: 10px;
      border-bottom: 1px solid #333;
      cursor: pointer;
      transition: background 0.3s;
    }
    .sidebar ul li:hover {
      background: #333;
    }
    .main-content {
      flex: 1;
      padding: 15px;
      background: #1e1e1e;
      overflow-y: auto;
    }
    .controls {
      background: #2d2d2d;
      padding: 15px;
      display: flex;
      flex-wrap: wrap;
      justify-content: center;
      gap: 10px;
      align-items: center;
      border-bottom: 2px solid #444;
      box-shadow: 0 2px 5px rgba(0,0,0,0.5);
      position: sticky;
      top: 60px;
      z-index: 1000;
    }
    select, input, button {
      padding: 10px;
      font-size: 16px;
      border-radius: 5px;
      border: 1px solid #444;
      background: #333;
      color: #e0e0e0;
    }
    button {
      background: #007BFF;
      border: none;
      font-weight: bold;
      cursor: pointer;
      transition: background 0.3s;
      color: #fff;
    }
    button:hover {
      background: #0056b3;
    }
    #map {
      height: 600px;
      width: 90%;
      margin: 20px auto;
      border-radius: 10px;
      box-shadow: 0 0 10px rgba(0,0,0,0.5);
    }
    .logs {
      width: 90%;
      margin: 20px auto;
      padding: 15px;
      background: #2d2d2d;
      border-radius: 10px;
      box-shadow: 0 0 10px rgba(0,0,0,0.5);
    }
    .logs h2 {
      margin-top: 0;
      color: #fff;
    }
    .logs table {
      width: 100%;
      border-collapse: collapse;
    }
    .logs th, .logs td {
      padding: 10px;
      text-align: center;
      border: 1px solid #444;
      color: #e0e0e0;
    }
    #timeline {
      display: none;
    }
    #analytics {
      width: 95%;
      max-width: 1200px;
      margin: 30px auto;
      background: #2d2d2d;
      padding: 20px;
      border-radius: 10px;
      box-shadow: 0 0 15px rgba(0,0,0,0.6);
      color: #e0e0e0;
      text-align: center;
    }
    #analytics h2 {
      margin-top: 0;
      font-size: 28px;
    }
    #analytics p {
      font-size: 18px;
      margin: 10px 0;
    }
    #analytics canvas {
      margin: 20px auto;
      max-width: 100%;
      height: 400px !important;
    }
    /* استجابة الشاشات الصغيرة */
    @media (max-width: 768px) {
      .container {
        flex-direction: column;
      }
      .sidebar {
        width: 100%;
        order: 2;
      }
      .main-content {
        width: 100%;
        order: 1;
      }
      .controls {
        position: static;
      }
    }
  </style>
</head>
<body>
  <?= $navBar ?>
  <div class="container">
    <div class="sidebar">
      <h2>الأجهزة</h2>
      <input type="text" id="searchDevice" placeholder="🔍 ابحث عن جهاز..." oninput="filterDevices()">
      <ul id="deviceList">
        <?php foreach ($devices as $device): ?>
          <li data-device-id="<?= $device['device_id'] ?>"><?= htmlspecialchars($device['device_name']) ?></li>
        <?php endforeach; ?>
      </ul>
    </div>

    <div class="main-content">
      <div class="controls">
        <label>🔽 اختر الجهاز:</label>
        <select id="deviceSelect">
          <option value="">جميع الأجهزة</option>
          <?php foreach ($devices as $device): ?>
            <option value="<?= $device['device_id'] ?>">
              <?= htmlspecialchars($device['device_name']) ?>
            </option>
          <?php endforeach; ?>
        </select>

        <label>📅 اختر التاريخ:</label>
        <input type="date" id="dateSelect">

        <label>⏰ من (12h):</label>
        <input type="text" id="timeFrom" placeholder="مثال: 01:00 PM">

        <label>⏰ إلى (12h):</label>
        <input type="text" id="timeTo" placeholder="مثال: 05:30 PM">

        <button onclick="fetchLocations()">🔍 بحث</button>
        <button onclick="resetFilters()">🔄 إعادة تعيين</button>
        <button onclick="fetchLatestLocation()">📡 آخر موقع مباشر</button>
        <!-- زر مشاركة الموقع مع تضمين اسم الجهاز -->
        <button onclick="shareLocation()">🔗 مشاركة الموقع</button>
        
        <label>⚡ سرعة الحركة (ملّي ثانية):</label>
        <input type="number" id="animationSpeed" value="1000" min="100" style="width: 120px;"/>

        <button onclick="playAnimation()">▶️ تشغيل الحركة</button>
        <button onclick="stopAnimation()">⏹ إيقاف الحركة</button>
      </div>

      <div id="map"></div>

      <div class="logs">
        <h2>سجلات الدخول والخروج (توقفات 5 دقائق فما فوق)</h2>
        <!-- أزرار التنزيل -->
        <button onclick="downloadLogsXLSX()">📥 Excel (XLSX)</button>

        <table id="logsTable">
          <thead>
            <tr>
              <th>اسم المكان</th>
              <th>وقت الدخول</th>
              <th>وقت الخروج</th>
              <th>الحالة</th>
              <th>البطارية</th>
            </tr>
          </thead>
          <tbody></tbody>
        </table>
      </div>

      <!-- إخفاء الـ Timeline -->
      <div id="timeline"></div>

      <!-- لوحة تحليل البيانات -->
      <div id="analytics"></div>
    </div>
  </div>

  <script>
    // مفاتيح وخيارات
    const accessToken = "pk.eyJ1IjoiaWJyYWhpbWNvbSIsImEiOiJja2YxZnB6OWowc2xxMnFxZGxya2ppdmZxIn0.keECYQuu1Bs0JeWDP8dSFQ";

    // تهيئة الطبقات من Mapbox
    const outdoorsLayer = L.tileLayer(
      `https://api.mapbox.com/styles/v1/mapbox/outdoors-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
      {
        maxZoom: 20,
        tileSize: 512,
        zoomOffset: -1,
        attribution: '&copy; <a href="https://www.openstreetmap.org/">OSM</a> & <a href="https://www.mapbox.com/">Mapbox</a>'
      }
    );
    const satelliteLayer = L.tileLayer(
      `https://api.mapbox.com/styles/v1/mapbox/satellite-streets-v11/tiles/{z}/{x}/{y}?access_token=${accessToken}`,
      {
        maxZoom: 20,
        tileSize: 512,
        zoomOffset: -1,
        attribution: '&copy; <a href="https://www.openstreetmap.org/">OSM</a> & <a href="https://www.mapbox.com/">Mapbox</a>'
      }
    );
    // إنشاء الخريطة
    var map = L.map('map', {
      center: [30, 0],
      zoom: 2,
      layers: [outdoorsLayer]
    });
    // تحكم بالطبقات
    var baseMaps = {
      "خريطة التضاريس": outdoorsLayer,
      "خريطة القمر الصناعي": satelliteLayer
    };
    L.control.layers(baseMaps).addTo(map);

    // تعريف أيقونات
    var parkIcon = L.icon({
      iconUrl: 'https://almanara.space/gps_tracker/uploads/image.png',
      iconSize: [32, 32],
      iconAnchor: [16, 32],
      popupAnchor: [0, -32]
    });
    var hiddenIcon = L.icon({
      iconUrl: 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=',
      iconSize: [1, 1],
      iconAnchor: [0, 0],
      popupAnchor: [0, 0]
    });
    var startIcon = L.icon({
      iconUrl: 'https://almanara.space/gps_tracker/uploads/start.png',
      iconSize: [60, 60],
      iconAnchor: [16, 32]
    });
    var endIcon = L.icon({
      iconUrl: 'https://almanara.space/gps_tracker/uploads/end.png',
      iconSize: [60, 60],
      iconAnchor: [16, 32]
    });
    var latestIcon = L.icon({
      iconUrl: 'https://cdn-icons-png.flaticon.com/512/684/684908.png',
      iconSize: [32, 32],
      iconAnchor: [16, 32]
    });

    // متغيرات عامة
    var markers = [];
    var circles = [];
    var polylines = [];
    var logs = [];
    var timelineData = [];
    var markerCounter = 1;
    var latestMarker = null;
    var latestLocationInterval = null;
    var lastTimestamp = null;

    // متغيرات الحركة
    var animationMarker = null;
    var animationTimeout = null; // بدلًا من setInterval سنستخدم setTimeout
    var currentAnimationIndex = 0;
    var routeCoordinates = [];
    var analyticsGroups = [];

    // دوال عامة
    function clearMap() {
      markers.forEach(m => map.removeLayer(m));
      markers = [];
      circles.forEach(c => map.removeLayer(c));
      circles = [];
      polylines.forEach(p => map.removeLayer(p));
      polylines = [];
      if (latestMarker) {
        map.removeLayer(latestMarker);
        latestMarker = null;
      }
      if (animationMarker) {
        map.removeLayer(animationMarker);
        animationMarker = null;
      }
      if (latestLocationInterval) {
        clearTimeout(latestLocationInterval);
        latestLocationInterval = null;
      }
      if (animationTimeout) {
        clearTimeout(animationTimeout);
        animationTimeout = null;
      }
      logs = [];
      timelineData = [];
      markerCounter = 1;
      routeCoordinates = [];
      analyticsGroups = [];
      updateLogsTable();
      updateTimeline();
      updateAnalytics();
    }

    function resetFilters() {
      document.getElementById("deviceSelect").value = "";
      document.getElementById("dateSelect").value = "";
      document.getElementById("timeFrom").value = "";
      document.getElementById("timeTo").value = "";
      clearMap();
    }

    // جلب المواقع حسب الفلاتر
    function fetchLocations() {
      clearMap();
      var deviceId = document.getElementById("deviceSelect").value;
      var date = document.getElementById("dateSelect").value;
      var timeFrom = convertTo24Hour(document.getElementById("timeFrom").value.trim());
      var timeTo = convertTo24Hour(document.getElementById("timeTo").value.trim());
      var url = "get_locations.php?";
      if (deviceId) url += "device_id=" + encodeURIComponent(deviceId) + "&";
      if (date) url += "date=" + encodeURIComponent(date) + "&";
      if (timeFrom) url += "time_from=" + encodeURIComponent(timeFrom) + "&";
      if (timeTo) url += "time_to=" + encodeURIComponent(timeTo) + "&";
      url = url.slice(0, -1);
      fetch(url)
        .then(res => res.json())
        .then(data => {
          console.log("📍 المواقع المسترجعة:", data);
          if (Array.isArray(data) && data.length > 0) {
            let dataSortedAsc = data.slice().sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
            let startLoc = dataSortedAsc[0];
            let endLoc = dataSortedAsc[dataSortedAsc.length - 1];
            // ماركر البداية
            let startLat = parseFloat(startLoc.latitude);
            let startLng = parseFloat(startLoc.longitude);
            if (!isNaN(startLat) && !isNaN(startLng)) {
              getPlaceName(startLat, startLng).then(startPlaceName => {
                let markerStart = L.marker([startLat, startLng], { icon: startIcon })
                  .addTo(map)
                  .bindPopup(
                    `<b>بداية المسار</b><br>
                     الوقت: ${formatTo12Hour(startLoc.timestamp)}<br>
                     بطارية: ${startLoc.battery ? startLoc.battery + '%' : 'غير معروفة'}<br>
                     ${startPlaceName}`
                  );
                markers.push(markerStart);
              });
            }
            // ماركر النهاية
            let endLat = parseFloat(endLoc.latitude);
            let endLng = parseFloat(endLoc.longitude);
            if (!isNaN(endLat) && !isNaN(endLng)) {
              getPlaceName(endLat, endLng).then(endPlaceName => {
                let markerEnd = L.marker([endLat, endLng], { icon: endIcon })
                  .addTo(map)
                  .bindPopup(
                    `<b>نهاية المسار</b><br>
                     الوقت: ${formatTo12Hour(endLoc.timestamp)}<br>
                     بطارية: ${endLoc.battery ? endLoc.battery + '%' : 'غير معروفة'}<br>
                     ${endPlaceName}`
                  );
                markers.push(markerEnd);
              });
            }
            // تجميع النقاط في مجموعات
            var groupedLocations = groupLocations(data, 30);
            groupedLocations.sort((a, b) => new Date(a.entryTime) - new Date(b.entryTime));
            analyticsGroups = groupedLocations;
            var routePoints = [];
            groupedLocations.forEach(group => {
              var lat = group.center[0];
              var lng = group.center[1];
              if (!isNaN(lat) && !isNaN(lng)) {
                // حساب البطارية (مدى بين أدنى وأعلى قيمة في المجموعة)
                var batteryInfo = "";
                if (group.minBattery !== null && group.maxBattery !== null) {
                  if (group.minBattery === group.maxBattery) {
                    batteryInfo = `بطارية: ${group.minBattery}%<br>`;
                  } else {
                    batteryInfo = `بطارية: ${group.minBattery}% → ${group.maxBattery}%<br>`;
                  }
                }

                var icon = (group.status === "بارك توقف") ? parkIcon : hiddenIcon;
                var marker = L.marker([lat, lng], { icon: icon })
                  .addTo(map)
                  .bindPopup(
                    `<b>النقطة ${markerCounter}</b><br>
                     وقت الدخول: ${formatTo12Hour(group.entryTime)}<br>
                     وقت الخروج: ${formatTo12Hour(group.exitTime)}<br>
                     ${batteryInfo}
                     الحالة: ${group.status}`
                  );
                markers.push(marker);
                markerCounter++;
                getPlaceName(lat, lng).then(placeName => {
                  logs.push({
                    placeName: placeName,
                    entryTime: group.entryTime,
                    exitTime: group.exitTime,
                    center: [lat, lng],
                    status: group.status,
                    batteryRange: (group.minBattery !== null && group.maxBattery !== null)
                      ? (group.minBattery === group.maxBattery
                          ? group.minBattery + '%'
                          : group.minBattery + '% - ' + group.maxBattery + '%')
                      : ''
                  });
                  timelineData.push({
                    start_date: group.entryTime,
                    end_date: group.exitTime,
                    text: {
                      headline: placeName,
                      text: `وقت الدخول: ${formatTo12Hour(group.entryTime)}<br>
                             وقت الخروج: ${formatTo12Hour(group.exitTime)}<br>
                             الحالة: ${group.status}`
                    }
                  });
                  updateLogsTable();
                  updateTimeline();
                });
                routePoints.push([lat, lng]);
                routeCoordinates.push({ 
                  lat: lat, 
                  lng: lng, 
                  time: group.entryTime, 
                  minBattery: group.minBattery, 
                  maxBattery: group.maxBattery 
                });
              }
            });
            if (routePoints.length > 1) {
              var polyline = L.polyline(routePoints, { color: 'blue', weight: 5 }).addTo(map);
              polylines.push(polyline);
            }
            if (routePoints.length > 0) {
              var bounds = new L.LatLngBounds(routePoints);
              map.fitBounds(bounds);
            }
            updateAnalytics();
          } else {
            alert("❌ لا توجد بيانات ضمن الشروط المحددة.");
          }
        })
        .catch(err => console.error("⚠️ خطأ في جلب البيانات:", err));
    }

    // دوال تحويل الوقت والتجميع
    function convertTo24Hour(timeStr) {
      if (!timeStr) return null;
      var parts = timeStr.split(':');
      if (parts.length < 2) return null;
      var hours = parseInt(parts[0], 10);
      var minutesPart = parts[1].trim();
      var minutes = parseInt(minutesPart, 10);
      var ampm = minutesPart.slice(-2).toUpperCase();
      if (ampm !== 'AM' && ampm !== 'PM') {
        return timeStr + ":00";
      }
      if (ampm === 'PM' && hours < 12) hours += 12;
      if (ampm === 'AM' && hours === 12) hours = 0;
      var hh = (hours < 10) ? '0' + hours : hours;
      var mm = (minutes < 10) ? '0' + minutes : minutes;
      return hh + ":" + mm + ":00";
    }

    function formatTo12Hour(dateStr) {
      if (!dateStr) return '';
      var d = new Date(dateStr);
      if (isNaN(d.getTime())) return dateStr;
      return d.toLocaleString('en-US', {
        hour12: true,
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: 'numeric',
        minute: '2-digit',
        second: '2-digit'
      });
    }

    // تعديل دالة groupLocations لإضافة خاصية "batteryLevels"
    function groupLocations(locations, radius) {
      if (!locations.length) return [];
      locations.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
      var groups = [];
      var first = locations[0];
      var currentGroup = {
        points: [first],
        center: [parseFloat(first.latitude), parseFloat(first.longitude)],
        entryTime: first.timestamp,
        exitTime: first.timestamp,
        batteryLevels: [ parseInt(first.battery) || null ]
      };
      var previousTimestamp = new Date(first.timestamp);
      for (var i = 1; i < locations.length; i++) {
        var loc = locations[i];
        var currentTimestamp = new Date(loc.timestamp);
        var timeDiff = (currentTimestamp - previousTimestamp) / 60000; // بالدقائق
        var lastPoint = currentGroup.points[currentGroup.points.length - 1];
        var lastLat = parseFloat(lastPoint.latitude);
        var lastLng = parseFloat(lastPoint.longitude);
        var currLat = parseFloat(loc.latitude);
        var currLng = parseFloat(loc.longitude);
        var distance = getDistance(lastLat, lastLng, currLat, currLng);

        // نضيف البطارية
        var locBattery = parseInt(loc.battery) || null;

        if (timeDiff <= 5 && distance <= radius) {
          currentGroup.points.push(loc);
          currentGroup.exitTime = loc.timestamp;
          currentGroup.batteryLevels.push(locBattery);
        } else {
          groups.push(currentGroup);
          currentGroup = {
            points: [loc],
            center: [currLat, currLng],
            entryTime: loc.timestamp,
            exitTime: loc.timestamp,
            batteryLevels: [locBattery]
          };
        }
        previousTimestamp = currentTimestamp;
      }
      groups.push(currentGroup);

      // تحديد الحالة (بارك أم تحرك) + حساب min/max للبطارية
      groups.forEach(g => {
        var diffMin = (new Date(g.exitTime) - new Date(g.entryTime)) / 60000;
        g.status = (diffMin >= 5) ? "بارك توقف" : "تحرك";
        var validBatteries = g.batteryLevels.filter(b => b !== null && !isNaN(b));
        g.minBattery = (validBatteries.length > 0) ? Math.min(...validBatteries) : null;
        g.maxBattery = (validBatteries.length > 0) ? Math.max(...validBatteries) : null;
      });
      return groups;
    }

    function getDistance(lat1, lon1, lat2, lon2) {
      var R = 6371e3;
      var φ1 = lat1 * Math.PI / 180;
      var φ2 = lat2 * Math.PI / 180;
      var Δφ = (lat2 - lat1) * Math.PI / 180;
      var Δλ = (lon2 - lon1) * Math.PI / 180;
      var a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
              Math.cos(φ1) * Math.cos(φ2) *
              Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
      var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
      return R * c;
    }

    function getPlaceName(lat, lng) {
      var url = `https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${accessToken}`;
      return fetch(url)
        .then(res => res.json())
        .then(data => {
          if (data.features && data.features.length > 0) {
            return data.features[0].place_name;
          }
          return "مكان غير معروف";
        })
        .catch(_ => "مكان غير معروف");
    }

    // تحديث سجلات الدخول والخروج
    function updateLogsTable() {
      var tableBody = document.getElementById("logsTable").getElementsByTagName("tbody")[0];
      tableBody.innerHTML = "";
      var filteredLogs = logs.filter(log => {
        var diff = (new Date(log.exitTime) - new Date(log.entryTime)) / 60000;
        return diff >= 5;
      });
      // تم عكس الترتيب إلى الأقدم فالأحدث
      filteredLogs.sort((a, b) => new Date(a.entryTime) - new Date(b.entryTime));
      filteredLogs.forEach(log => {
        var row = tableBody.insertRow();
        row.insertCell(0).innerHTML = `<a href="#" onclick="focusOnLocation([${log.center}])">${log.placeName}</a>`;
        row.insertCell(1).textContent = formatTo12Hour(log.entryTime);
        row.insertCell(2).textContent = formatTo12Hour(log.exitTime);
        row.insertCell(3).textContent = log.status;
        row.insertCell(4).textContent = log.batteryRange;
      });
    }

    function updateTimeline() {
      var filteredTimeline = timelineData.filter(evt => {
        var diff = (new Date(evt.end_date) - new Date(evt.start_date)) / 60000;
        return diff >= 5;
      });
      new TL.Timeline('timeline', { events: filteredTimeline });
    }

    function focusOnLocation(center) {
      map.setView(center, 15);
    }

    // جلب آخر موقع مباشر
    function fetchLatestLocation() {
      var deviceId = document.getElementById("deviceSelect").value;
      if (!deviceId) {
        alert("⚠️ يرجى اختيار جهاز لجلب أحدث موقع.");
        return;
      }
      if (latestLocationInterval) {
        clearTimeout(latestLocationInterval);
      }
      function getLatest() {
        var url = `get_latest_location.php?device_id=${deviceId}&t=${Date.now()}`;
        fetch(url)
          .then(res => res.json())
          .then(data => {
            console.log("📡 آخر بيانات:", data);
            if (data.status === "success") {
              var latest = data.data;
              var lat = parseFloat(latest.latitude);
              var lng = parseFloat(latest.longitude);
              var timestamp = latest.timestamp;
              var battery = latest.battery; // نسبة الشحن
              if (!isNaN(lat) && !isNaN(lng)) {
                if (timestamp !== lastTimestamp) {
                  lastTimestamp = timestamp;
                  if (latestMarker) {
                    map.removeLayer(latestMarker);
                  }
                  latestMarker = L.marker([lat, lng], { icon: latestIcon })
                    .addTo(map)
                    .bindPopup(`
                      <b>📡 آخر موقع</b><br>
                      الجهاز: ${latest.device_name}<br>
                      الوقت: ${formatTo12Hour(latest.timestamp)}<br>
                      البطارية: ${battery ? battery + '%' : 'غير معروفة'}
                    `).openPopup();
                  var circle = L.circle([lat, lng], {
                    radius: 30,
                    color: 'blue',
                    fillColor: '#00f',
                    fillOpacity: 0.5
                  }).addTo(map);
                  circles.push(circle);
                  map.setView([lat, lng], 15);
                }
              }
            } else {
              console.warn("❌ لا يوجد موقع متاح لهذا الجهاز.");
            }
            latestLocationInterval = setTimeout(getLatest, 10000);
          })
          .catch(err => console.error("⚠️ خطأ في جلب آخر موقع:", err));
      }
      getLatest();
    }

    // إنشاء أيقونة الحركة مع عرض الوقت
    function createAnimationIcon(time, battery) {
      var batteryStr = battery === null ? '' : `<br>بطارية: ${battery}%`;
      return L.divIcon({
        html: `<div style="position: relative; background: rgba(0,0,0,0.7); padding: 4px 8px; border-radius: 6px; border: 2px solid #fff; display: inline-flex; align-items: center;">
                 <img src="https://cdn-icons-png.flaticon.com/512/684/684908.png" style="width:40px;height:40px;">
                 <span style="margin-left: 8px; color: #fff; font-size: 14px;">
                   ${formatTo12Hour(time)}${batteryStr}
                 </span>
               </div>`,
        iconSize: [60, 50],
        className: ''
      });
    }

    /**
     *  دالة حركة المسار خطوة بخطوة باستخدام setTimeout
     */
    function animateStep() {
      if (currentAnimationIndex >= routeCoordinates.length) {
        return;
      }
      var currentPoint = routeCoordinates[currentAnimationIndex];
      var minB = currentPoint.minBattery;
      var maxB = currentPoint.maxBattery;
      var batteryText = null;
      if (minB !== null && maxB !== null) {
        batteryText = (minB === maxB) ? minB : (minB + '% → ' + maxB + '%');
      }

      if (!animationMarker) {
        animationMarker = L.marker(
          [currentPoint.lat, currentPoint.lng],
          { icon: createAnimationIcon(currentPoint.time, batteryText) }
        ).addTo(map);
      } else {
        animationMarker.setLatLng([currentPoint.lat, currentPoint.lng]);
        animationMarker.setIcon(createAnimationIcon(currentPoint.time, batteryText));
      }
      map.panTo([currentPoint.lat, currentPoint.lng]);
      currentAnimationIndex++;

      if (currentAnimationIndex < routeCoordinates.length) {
        var speed = parseInt(document.getElementById("animationSpeed").value);
        if (isNaN(speed) || speed < 100) speed = 1000;
        animationTimeout = setTimeout(animateStep, speed);
      }
    }

    // تشغيل الحركة
    function playAnimation() {
      if (!routeCoordinates.length) {
        alert("❌ لا يوجد مسار لتشغيل الحركة.");
        return;
      }
      if (animationTimeout) {
        clearTimeout(animationTimeout);
        animationTimeout = null;
      }
      if (animationMarker) {
        map.removeLayer(animationMarker);
        animationMarker = null;
      }
      currentAnimationIndex = 0;
      animateStep();
    }

    // إيقاف الحركة
    function stopAnimation() {
      if (animationTimeout) {
        clearTimeout(animationTimeout);
        animationTimeout = null;
      }
    }

    // تحديث لوحة تحليل البيانات باستخدام Chart.js
    function updateAnalytics() {
      var totalDistance = 0;
      for (var i = 1; i < routeCoordinates.length; i++) {
        totalDistance += getDistance(
          routeCoordinates[i-1].lat, routeCoordinates[i-1].lng,
          routeCoordinates[i].lat,   routeCoordinates[i].lng
        );
      }
      totalDistance = totalDistance / 1000; // تحويل إلى كم

      var movingTime = 0, stoppedTime = 0;
      analyticsGroups.forEach(g => {
        var t1 = new Date(g.entryTime).getTime();
        var t2 = new Date(g.exitTime).getTime();
        var duration = (t2 - t1) / 1000;
        if (g.status === "تحرك") {
          movingTime += duration;
        } else {
          stoppedTime += duration;
        }
      });
      var movingMinutes = movingTime / 60;
      var stoppedMinutes = stoppedTime / 60;
      var averageSpeed = (movingTime > 0)
        ? totalDistance / (movingTime / 3600)
        : 0;

      var analyticsDiv = document.getElementById("analytics");
      analyticsDiv.innerHTML = `<h2>تحليل البيانات</h2>
         <p>المسافة المقطوعة: ${totalDistance.toFixed(2)} كم</p>
         <p>وقت التحرك: ${movingMinutes.toFixed(2)} دقيقة</p>
         <p>وقت التوقف: ${stoppedMinutes.toFixed(2)} دقيقة</p>
         <p>متوسط السرعة: ${averageSpeed.toFixed(2)} كم/س</p>
         <canvas id="timeChart"></canvas>
         <canvas id="speedChart"></canvas>`;

      var ctx1 = document.getElementById("timeChart").getContext("2d");
      new Chart(ctx1, {
        type: 'doughnut',
        data: {
          labels: ['تحرك (دقيقة)', 'توقف (دقيقة)'],
          datasets: [{
            data: [movingMinutes, stoppedMinutes]
          }]
        },
        options: {
          responsive: true,
          plugins: { legend: { position: 'bottom' } }
        }
      });

      var speeds = [];
      var speedLabels = [];
      for (var i = 1; i < routeCoordinates.length; i++) {
        var d = getDistance(
          routeCoordinates[i-1].lat, routeCoordinates[i-1].lng,
          routeCoordinates[i].lat,   routeCoordinates[i].lng
        );
        var tDiff = (new Date(routeCoordinates[i].time).getTime() -
                     new Date(routeCoordinates[i-1].time).getTime()) / 3600000;
        var sp = (tDiff > 0) ? (d/1000) / tDiff : 0;
        speeds.push(sp.toFixed(2));
        speedLabels.push("نقطة " + i);
      }
      var ctx2 = document.getElementById("speedChart").getContext("2d");
      new Chart(ctx2, {
        type: 'line',
        data: {
          labels: speedLabels,
          datasets: [{
            label: 'السرعة (كم/س)',
            data: speeds,
            fill: false,
            tension: 0.1
          }]
        },
        options: {
          responsive: true,
          plugins: { legend: { position: 'bottom' } }
        }
      });
    }

    // مشاركة الموقع المباشر
    function shareLocation() {
      var deviceSelect = document.getElementById("deviceSelect");
      var deviceId = deviceSelect.value;
      if (!deviceId) {
        alert("❌ يرجى اختيار الجهاز الذي تريد مشاركة موقعه.");
        return;
      }
      if (!latestMarker) {
        alert("❌ لا يوجد موقع حديث للمشاركة. يرجى جلب آخر موقع أولاً.");
        return;
      }
      var deviceName = deviceSelect.options[deviceSelect.selectedIndex].text;
      var expires = Date.now() + 3600000; // ساعة واحدة
      var shareUrl = window.location.origin + window.location.pathname
        + "?share=1&device_id=" + encodeURIComponent(deviceId)
        + "&device_name=" + encodeURIComponent(deviceName)
        + "&expires=" + expires;
      window.open(shareUrl, "_blank");
    }

    // تصفية الأجهزة في القائمة الجانبية
    function filterDevices() {
      var input = document.getElementById("searchDevice").value.toUpperCase();
      var deviceList = document.getElementById("deviceList");
      var devices = deviceList.getElementsByTagName("li");
      for (var i = 0; i < devices.length; i++) {
        var deviceName = devices[i].textContent || devices[i].innerText;
        devices[i].style.display = (deviceName.toUpperCase().indexOf(input) > -1) ? "" : "none";
      }
    }

    // النقر على جهاز في القائمة الجانبية
    document.getElementById("deviceList").addEventListener("click", function(event) {
      if (event.target.tagName === "LI") {
        var deviceId = event.target.getAttribute("data-device-id");
        document.getElementById("deviceSelect").value = deviceId;
        fetchLatestLocation();
      }
    });

    // إشعارات المستخدم (مثال)
    function requestNotificationPermission() {
      if (Notification.permission !== "granted") {
        Notification.requestPermission().then(permission => {
          if (permission === "granted") {
            console.log("تم منح الإذن لعرض الإشعارات.");
          }
        });
      }
    }
    function sendNotification(title, message) {
      if (Notification.permission === "granted") {
        new Notification(title, { body: message });
      } else {
        console.warn("لم يتم منح الإذن لعرض الإشعارات.");
      }
    }
    requestNotificationPermission();

    // أمثلة على فحص بعض الحالات (وصول، انحراف، إلخ)
    function checkForEvents(data) {
      data.forEach(location => {
        if (isAtTargetLocation(location)) {
          sendNotification("وصول جهاز", `الجهاز ${location.device_name} وصل إلى الموقع المحدد.`);
        }
        if (isLongStop(location)) {
          sendNotification("توقف طويل", `الجهاز ${location.device_name} توقف لفترة طويلة.`);
        }
        if (isOffRoute(location)) {
          sendNotification("انحراف عن المسار", `الجهاز ${location.device_name} انحرف عن المسار المحدد.`);
        }
      });
    }

    function isAtTargetLocation(location) {
      const targetLocation = { lat: 30.1234, lng: 31.1234 };
      const distance = getDistance(location.latitude, location.longitude, targetLocation.lat, targetLocation.lng);
      return distance < 100;
    }

    function isLongStop(location) {
      const stopDuration = (new Date() - new Date(location.timestamp)) / 60000;
      return stopDuration > 10;
    }

    function isOffRoute(location) {
      const route = [
        { lat: 30.1234, lng: 31.1234 },
        { lat: 30.2234, lng: 31.2234 }
      ];
      const distance = getDistanceToRoute(location.latitude, location.longitude, route);
      return distance > 500;
    }

    function getDistanceToRoute(lat, lng, route) {
      // يمكنك وضع خوارزمية لحساب أقرب مسافة من خطوط المسار
      return 999999;
    }

    // دالة لتحميل السجلات بصيغة Excel (XLSX)
    function downloadLogsXLSX() {
      var filteredLogs = logs.filter(log => {
        var diff = (new Date(log.exitTime) - new Date(log.entryTime)) / 60000;
        return diff >= 5;
      });
      // تحضير البيانات للورقة
      var ws_data = [
        ["اسم المكان", "وقت الدخول", "وقت الخروج", "الحالة", "البطارية"]
      ];
      filteredLogs.forEach(log => {
        ws_data.push([
          log.placeName,
          formatTo12Hour(log.entryTime),
          formatTo12Hour(log.exitTime),
          log.status,
          log.batteryRange
        ]);
      });

      var wb = XLSX.utils.book_new();
      var ws = XLSX.utils.aoa_to_sheet(ws_data);

      // تنسيق عرض الأعمدة
      ws['!cols'] = [
        { wch: 30 },
        { wch: 25 },
        { wch: 25 },
        { wch: 20 },
        { wch: 15 }
      ];

      // تنسيق الصف الأول (عناوين الجدول)
      var range = XLSX.utils.decode_range(ws['!ref']);
      for (var C = range.s.c; C <= range.e.c; ++C) {
        var cell_address = XLSX.utils.encode_col(C) + "1";
        if (!ws[cell_address]) continue;
        ws[cell_address].s = {
          font: { bold: true },
          alignment: { horizontal: "center", vertical: "center" }
        };
      }

      XLSX.utils.book_append_sheet(wb, ws, "سجلات الدخول والخروج");
      XLSX.writeFile(wb, "logs.xlsx");
    }
  </script>
</body>
</html>
